/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _ADAPTEUR_DONNEES_COMPOSANTS_H_
#define _ADAPTEUR_DONNEES_COMPOSANTS_H_

#include <vector>
#include <map>

#include <src/global/global.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Adapteur { DECL_SHARED_PTR(Composant); DECL_SHARED_PTR(Programme); DECL_SHARED_PTR(Memoire); DECL_SHARED_PTR(NomAdapteur); DECL_SHARED_PTR(CoreDump); DECL_SHARED_PTR(Plugin); } }

namespace SetLgg
{
	namespace Adapteur
	{
		class Composant
		{
			public:
				explicit Composant(const std::string& fichier, const SetLgg::Global::Source::PositionSP& position)
				:_fichier(SetLgg::Global::Chaine::desechappe(fichier)), _position(position) {};
				Composant(const Composant&) = default;
				void operator>>(const std::string& repertoire)
				{
					_fichier = repertoire+_fichier;
				}
				operator std::string() const { return _fichier; }
			protected:
				std::string _fichier;
				SetLgg::Global::Source::PositionSP _position;
		};

		class Programme : private Composant
		{
			public:
				explicit Programme(const std::string& fichier, const SetLgg::Global::Source::PositionSP& position)
				:Composant(fichier,position) {};
				template<typename oStream>
				friend oStream& operator<<(oStream& os, const Programme& programme)
				{
					os << " " << programme._fichier;
					return os;
				};
				using Composant::operator>>;
				using Composant::operator std::string;
		};

		class Memoire : private Composant
		{
			public:
				explicit Memoire(const std::string& fichier, const SetLgg::Global::Source::PositionSP& position)
				:Composant(fichier,position) {};
				template<typename oStream>
				friend oStream& operator<<(oStream& os, const Memoire& memoire)
				{
					os << " " << memoire._fichier;
					return os;
				};
				operator std::string() const
				{
					return _fichier;
				}
				char *pourAppelVM()
				{
					return const_cast<char*>(_fichier.c_str());
				}
				using Composant::operator>>;
				using Composant::operator std::string;
		};

		class NomAdapteur : private Composant
		{
			public:
				explicit NomAdapteur(const std::string& fichier, const SetLgg::Global::Source::PositionSP& position)
				:Composant(fichier,position) {};
				operator const SetLgg::Global::Source::PositionSP () const { return _position; }
				using Composant::operator>>;
				using Composant::operator std::string;
		};

		class CoreDump : private Composant
		{
			public:
				explicit CoreDump(const std::string& fichier, const SetLgg::Global::Source::PositionSP& position)
				:Composant(fichier,position) {};
				template<typename oStream>
				friend oStream& operator<<(oStream& os, const CoreDump& coredump)
				{
					os << " -c " << coredump._fichier;
					return os;
				};
				using Composant::operator>>;
		};

		class Plugin : private Composant
		{
			public:
				explicit Plugin(const std::string& fichier, const SetLgg::Global::Source::PositionSP& position)
				:Composant(fichier,position) {};
				template<typename oStream>
				friend oStream& operator<<(oStream& os, const Plugin& plugin)
				{
					os << " -p " << plugin._fichier;
					return os;
				};
				operator std::string () const
				{
					return _fichier;
				}
				operator char* ()
				{
					return const_cast<char*>(_fichier.c_str());
				}
		};

		typedef std::vector<Plugin> ListePlugins;
	}
}

#endif
