/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <unistd.h>
#include <fstream>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>

#include <src/adapteur/donnees/generationscript.h>
using namespace SetLgg::Adapteur;

GenerationScript::GenerationScript(const NomAdapteur& adapteur, const ListePlugins& plugins, const ListeOptionsGeneration& options, const ListeArgumentsGeneration& arguments)
:GenerationScript(adapteur,Programme(static_cast<std::string>(adapteur)+".setlggasm",adapteur),MemoireSP(new Memoire(static_cast<std::string>(adapteur)+".setlggmem",adapteur)),CoreDumpSP(new CoreDump(std::string("core_")+static_cast<std::string>(adapteur),adapteur)),plugins,options,arguments)
{
}

void GenerationScript::operator() ()
{
	Adapteur::test_coherence(_options,_arguments);
//	_adapteur >> _installation->_adapteur;
	_programme >> _installation->_programme;
	if(_memoire)
	{
		(*_memoire) >> _installation->_memoire;
	}
	if(_coredump)
	{
		(*_coredump) >> _installation->_miseaplat;
	}
	std::ostringstream script;
	script << "#!/bin/sh" << std::endl;

	script << std::endl
		<< "# Initialise option variables" << std::endl;

	for(auto it=_options.cbegin() ; it!=_options.cend() ; ++it)
	{
		script + (**it);
	}

	script << std::endl
		<< "# Parse options" << std::endl;

	script << "OPTERR=0" << std::endl
		<< "while getopts ";
	if(_options.empty())
	{
		script << "?";
	}
	else
	{
		for(auto it=_options.cbegin() ; it!=_options.cend() ; ++it)
		{
			script - (**it);
		}
	}
	script << " ARG" << std::endl
		<< "do" << std::endl
		<< "\tcase $ARG in" << std::endl;

	for(auto it=_options.cbegin() ; it!=_options.cend() ; ++it)
	{
		script << "\t\t" << (*it)->lettre() << ")" << std::endl
			<< "\t\t\t" << (**it) << std::endl
			<< "\t\t;;" << std::endl;
	}
	
	script << "\t\t*)" << std::endl
		<< "\t\t\tcat << EOH >&2" << std::endl
		<< static_cast<std::string>(_adapteur) << " [options]";
	if(not _memoire)
	{
		script << " [memory file]";
	}
	script << " [arguments]" << std::endl
		<< std::endl
		<< "Options:" << std::endl;
	for(auto it=_options.cbegin() ; it!=_options.cend() ; ++it)
	{
		script * (**it);
		script << std::endl;
	}

	script << std::endl
		<< "Mandatory arguments:" << std::endl;
	for(auto it=_arguments.cbegin() ; it!=_arguments.cend() ; ++it)
	{
		script * (*it);
		script << std::endl;
	}
	script << std::endl
		<< "Optional arguments:" << std::endl
		<< "More arguments can be passed to the command after the last mandatory argument." << std::endl
		<< std::endl
		<< "Useful environment variables:" << std::endl
		<< "DEBUG  : can be used to pass a one shot and single option to the virtual machine." << std::endl
		<< "COMMAND: can be used to launch the virtual machine in another command." << std::endl
		<< std::endl
		<< "EOH" << std::endl
		<< "\t\t\texit 2" << std::endl
		<< "\t\t;;" << std::endl
		<< "\tesac" << std::endl
		<< "done" << std::endl
		<< "shift $((${OPTIND}-1))" << std::endl;

	if(not _memoire)
	{
		script << std::endl
			<< "# Set memory file" << std::endl;

		script << "if [ \"x$1\" == \"x\" ]; then" << std::endl
			<< "\techo \"Missing memory file name as first argument\" >&2" << std::endl
			<< "\texit 1" << std::endl
			<< "fi" << std::endl
			<< "MEMORY_FILE=$1" << std::endl
			<< "shift" << std::endl;
	}

	if(not _arguments.empty())
	{
		script << std::endl
			<< "# Parse mandatory arguments" << std::endl;
	}

	for(auto it=_arguments.cbegin() ; it!=_arguments.cend() ; ++it)
	{
		script << "if [ \"x$1\" == \"x\" ] ; then" << std::endl
			<< "\techo \"" << it->_alias << " argument missing\"" << std::endl
			<< "\texit 1" << std::endl
			<< "fi" << std::endl
			<< "ARGUMENT_" << it->_alias << "=\"$1\"" << std::endl
			<< "shift" << std::endl
			<< std::endl;

	}

	if(_arguments.empty())
	{
		script << std::endl;
	}
	script << "# Parse optional arguments" << std::endl;

	script << "ARGUMENTS=\"\"" << std::endl
		<< "for argument in \"$@\" ;" << std::endl
		<< "do" << std::endl
		<< "\tARGUMENTS=\"${ARGUMENTS}:argument \\\"${argument}\\\"\n\"" << std::endl
		<< "done" << std::endl;
	
	script << std::endl
		<< "# Update memory file" << std::endl;
	
	std::string fichier_memoire;
	if(_memoire)
	{
		fichier_memoire = static_cast<std::string>(*_memoire);
	}
	else
	{
		fichier_memoire = "${MEMORY_FILE}";
	}

	script << "if [ ! -d $(dirname \"" << fichier_memoire << "\") ] ; then" << std::endl
		<< "\techo \"Unable to access to the memory file " << fichier_memoire << " for update\" >&2" << std::endl
		<< "\texit 1" << std::endl
		<< "fi" << std::endl;
	script << "(cat << EOM |";
	if(not _installation->_plugin.empty())
	{
		script << " LD_LIBRARY_PATH=\"$LD_LIBRARY_PATH:" << _installation->_plugin << "\"" ;
	}
	script << " " << INSTALL_DIR << "/setlgg_wrapper -q -" << std::endl
		<< ":memory \"" << fichier_memoire << "\"" << std::endl
		<< ":wrapper \"" << static_cast<std::string>(_adapteur) << "\"" << std::endl;
	
	for(auto it=_plugins.cbegin() ; it!=_plugins.cend() ; ++it)
	{
		script << ":plugin \"" << static_cast<std::string>(*it) << "\"" << std::endl;
	}

	for(auto it=_options.cbegin() ; it!=_options.cend() ; ++it)
	{
		script & (**it);
		script << std::endl;
	}

	for(auto it=_arguments.cbegin() ; it!=_arguments.cend() ; ++it)
	{
		script & (*it);
		script << std::endl;
	}

	script << "${ARGUMENTS}" << std::endl
		<< "EOM" << std::endl
		<< ") || exit 1" << std::endl;

	script << std::endl
		<< "# Launch virtual machine" << std::endl
		<< "if [ \"x${DEBUG}\" != \"x\" ]; then" << std::endl
		<< "\tif [ \"${DEBUG:0:1}\" != \"-\" ] ; then" << std::endl
		<< "\t\techo \"DEBUG variable shall contain a single option.\" >&2" << std::endl
		<< "\t\texit 1" << std::endl
		<< "\tfi" << std::endl
		<< "\tDEBUG=\"\\\"${DEBUG}\\\"\"" << std::endl
		<< "fi" << std::endl
		<< std::endl;

	script << "eval";
	if(not _installation->_plugin.empty())
	{
		script << " LD_LIBRARY_PATH=\"$LD_LIBRARY_PATH:" << _installation->_plugin << "\"" ;
	}
	script << " exec ${COMMAND} " << INSTALL_DIR << "/setlgg_machine ${DEBUG}";
	if(_coredump)
	{
		script << (*_coredump);
	}
	for(auto it=_plugins.cbegin() ; it!=_plugins.cend() ; ++it)
	{
		script << (*it);
	}
	script << _programme << " " << fichier_memoire << std::endl;
	std::string nom_fichier = _adapteur;
	{
		if(::access(nom_fichier.c_str(),W_OK))
		{
			if(errno!=ENOENT)
			{
				std::ostringstream details;
				details << SetLgg::Global::Exception::details << std::endl << "access: " << ::strerror(errno);
				throw GenerationImpossible(details.str());
			}
		}
		if(errno!=ENOENT)
		{
			struct stat buf;
			if(::stat(nom_fichier.c_str(),&buf))
			{
				std::ostringstream details;
				details << SetLgg::Global::Exception::details << std::endl << "stat: " << ::strerror(errno);
				throw GenerationImpossible(details.str());
			}
			if(S_ISDIR(buf.st_mode))
			{
				std::ostringstream details;
				details << SetLgg::Global::Exception::details << std::endl << "stat: Is a directory";
				throw GenerationImpossible(details.str());
			}
		}
		std::ofstream fichier(nom_fichier);
		fichier << (script.str());
	}
	if(::chmod(nom_fichier.c_str(),S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH)<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "chmod: " << ::strerror(errno);
		throw GenerationImpossible(details.str());
	}
}
