/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _ADAPTEUR_DONNEES_OPTION_GENERATION_H_
#define _ADAPTEUR_DONNEES_OPTION_GENERATION_H_

#include <vector>
#include <map>

#include <src/global/global.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Adapteur { DECL_SHARED_PTR(OptionGeneration); } }

namespace SetLgg
{
	namespace Adapteur
	{
		class Adapteur;
		class OptionGeneration
		{
			friend class Adapteur;
			public:
				OptionGeneration(const std::string& alias, const std::string& lettre, const std::string& aide, const SetLgg::Global::Source::PositionSP& position)
				:_alias(alias), _lettre(lettre), _aide(aide), _position(position) {};
				OptionGeneration(const OptionGeneration&) = delete;
				virtual ~OptionGeneration() = default;
				virtual void initialise(std::ostream& os) const
				{
					os << "unset OPTION_" << _alias << std::endl;
				}
				virtual void getopts(std::ostream& os) const =0;
				virtual void affectation(std::ostream& os) const =0;
				virtual void aide(std::ostream& os) const =0;
				virtual void manipulationmemoire(std::ostream& os) const =0;
				template<typename oStream>
				friend oStream& operator+(oStream &os, const OptionGeneration& option)
				{
					option.initialise(os);
					return os;
				}
				template<typename oStream>
				friend oStream& operator-(oStream &os, const OptionGeneration& option)
				{
					option.getopts(os);
					return os;
				}
				template<typename oStream>
				friend oStream& operator<<(oStream &os, const OptionGeneration& option)
				{
					option.affectation(os);
					return os;
				}
				template<typename oStream>
				friend oStream& operator*(oStream &os, const OptionGeneration& option)
				{
					option.aide(os);
					return os;
				}
				template<typename oStream>
				friend oStream& operator&(oStream &os, const OptionGeneration& option)
				{
					option.manipulationmemoire(os);
					return os;
				}
				std::string lettre() const
				{
					return _lettre;
				}
			protected:
				std::string _alias;
				std::string _lettre;
				std::string _aide;
				SetLgg::Global::Source::PositionSP _position;
		};

		class OptionGenerationDrapeau : public OptionGeneration
		{
			public:
				OptionGenerationDrapeau(const std::string& alias, const std::string& lettre, const std::string& aide, const SetLgg::Global::Source::PositionSP& position)
				:OptionGeneration(alias,lettre,aide,position) {};
				virtual void initialise(std::ostream& os) const
				{
					os << "OPTION_" << _alias << "=\"disabled\"" << std::endl;
				}
				virtual void getopts(std::ostream& os) const
				{
					os << _lettre;
				}
				virtual void affectation(std::ostream& os) const
				{
					os << "OPTION_" << _alias << "=\"enabled\"";
				}
				virtual void aide(std::ostream& os) const
				{
					os << "-" << _lettre << "\t\t: (Flag " << _alias << ") " << _aide; 
				}
				virtual void manipulationmemoire(std::ostream& os) const
				{
					os << ":option " << _alias << " :${OPTION_" << _alias << "}";
				}
		};

		class OptionGenerationCompteur : public OptionGeneration
		{
			public:
				OptionGenerationCompteur(const std::string& alias, const std::string& lettre, const std::string& aide, const SetLgg::Global::Source::PositionSP& position)
				:OptionGeneration(alias,lettre,aide,position) {};
				virtual void initialise(std::ostream& os) const
				{
					os << "OPTION_" << _alias << "=0" << std::endl;
				}
				virtual void getopts(std::ostream& os) const
				{
					os << _lettre;
				}
				virtual void affectation(std::ostream& os) const
				{
					os << "OPTION_" << _alias << "=$((${OPTION_" << _alias << "}+1))";
				}
				virtual void aide(std::ostream& os) const
				{
					os << "-" << _lettre << "\t\t: (Counter " << _alias << ") " << _aide; 
				}
				virtual void manipulationmemoire(std::ostream& os) const
				{
					os << ":option " << _alias << " :counter ${OPTION_" << _alias << "}";
				}
		};

		class OptionGenerationEntier : public OptionGeneration
		{
			public:
				OptionGenerationEntier(const std::string& alias, const std::string& lettre, const std::string& aide, const SetLgg::Global::Source::PositionSP& position)
				:OptionGeneration(alias,lettre,aide,position) {};
				virtual void getopts(std::ostream& os) const
				{
					os << _lettre << ":";
				}
				virtual void affectation(std::ostream& os) const
				{
					os << "OPTION_" << _alias << "=$((${OPTARG}))";
				}
				virtual void aide(std::ostream& os) const
				{
					os << "-" << _lettre << " <integer>\t: (" << _alias << ") " << _aide; 
				}
				virtual void manipulationmemoire(std::ostream& os) const
				{
					os << ":option " << _alias << " :integer ${OPTION_" << _alias << "}";
				}
		};

		class OptionGenerationChaine : public OptionGeneration
		{
			public:
				OptionGenerationChaine(const std::string& alias, const std::string& lettre, const std::string& aide, const SetLgg::Global::Source::PositionSP& position)
				:OptionGeneration(alias,lettre,aide,position) {};
				virtual void getopts(std::ostream& os) const
				{
					os << _lettre << ":";
				}
				virtual void affectation(std::ostream& os) const
				{
					os << "OPTION_" << _alias << "=\"\\\"${OPTARG}\\\"\"";
				}
				virtual void aide(std::ostream& os) const
				{
					os << "-" << _lettre << " <string>\t: (" << _alias << ") " << _aide; 
				}
				virtual void manipulationmemoire(std::ostream& os) const
				{
					os << ":option " << _alias << " :string ${OPTION_" << _alias << "}";
				}
		};

		typedef std::vector<OptionGenerationSP> ListeOptionsGeneration;
	}
}

#endif
