/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _ADAPTEUR_DONNEES_OPTION_MANIPULATION_H_
#define _ADAPTEUR_DONNEES_OPTION_MANIPULATION_H_

#include <vector>
#include <map>

#include <src/global/global.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Adapteur { DECL_SHARED_PTR(OptionManipulation); } }

namespace SetLgg
{
	namespace Adapteur
	{
		class OptionManipulation
		{
			public:
				OptionManipulation(const std::string& alias)
				:_alias(alias) {};
				OptionManipulation(const OptionManipulation&) = delete;
				virtual ~OptionManipulation() = default;
				operator std::string () const
				{
					return _alias;
				}
				template<typename oStream>
				friend oStream& operator<<(oStream& os, const OptionManipulation& option)
				{
					os << ":goto option_" << option._alias << " :unless " << option._alias << " :defined" << std::endl
						<< ":delete " << option._alias << std::endl
						<< ":label option_" << option._alias << std::endl;
					option.cree(os);
					return os;
				}
				virtual void cree(std::ostream& os) const =0;
			protected:
				std::string _alias;
		};

		class OptionManipulationDrapeau : public OptionManipulation
		{
			public:
				OptionManipulationDrapeau(const std::string& alias, const bool valeur)
				:OptionManipulation(alias), _valeur(valeur) {};
				virtual void cree(std::ostream& os) const
				{
					os << ":new BLN/" << _alias << std::endl;
					if(_valeur)
					{
						os << "TRUE";
					}
					else
					{
						os << "FALSE";
					}
					os << " -> " << _alias << std::endl;
				}
			private:
				bool _valeur;
		};

		class OptionManipulationEntier : public OptionManipulation
		{
			public:
				explicit OptionManipulationEntier(const std::string& alias)
				:OptionManipulation(alias), _valeur() {};
				OptionManipulationEntier(const std::string& alias, const long int valeur)
				:OptionManipulation(alias)
				{
					long int *entier = new long int;
					*entier=valeur;
					_valeur = SHARED_PTR(long int)(entier);
				};
				virtual void cree(std::ostream& os) const
				{
					os << ":new INT/" << _alias << std::endl;
					if(_valeur)
					{
						os << *_valeur << " -> " << _alias << std::endl;
					}
				}
			private:
				SHARED_PTR(long int) _valeur;
		};

		class OptionManipulationChaine : public OptionManipulation
		{
			public:
				explicit OptionManipulationChaine(const std::string& alias)
				:OptionManipulation(alias), _valeur() {};
				OptionManipulationChaine(const std::string& alias, const std::string& valeur)
				:OptionManipulation(alias), _valeur(new std::string(valeur)) {};
				virtual void cree(std::ostream& os) const
				{
					os << ":new STR/" << _alias << std::endl;
					if(_valeur)
					{
						os << "\"" << *_valeur << "\" -> " << _alias << std::endl;
					}
				}
			private:
				SHARED_PTR(std::string) _valeur;
		};

		typedef std::vector<OptionManipulationSP> ListeOptionsManipulation;
	}
}

#endif
