/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _ADAPTEUR_ADAPTEUR_H_
#define _ADAPTEUR_ADAPTEUR_H_
#include <iostream>
#include <unistd.h>

#include <config.h>

#include <src/global/exceptions.h>

#include <src/adapteur/donnees/adapteur.h>
#include <src/adapteur/analyseur/analyseur.h>

using namespace SetLgg::Adapteur;

void affiche_aide(std::ostream& os, const std::string commande)
{
	os	<< commande << " [options] [wrapper_file]"
		<< std::endl << "where the wrapper_file contains the description of the wrapper." << std::endl
		<< "If the file is not present or is equal to -, the standard input is taken instead." << std::endl
		<< std::endl
		<< "Options:" << std::endl
		<< "-p <prefix>    : base directory of installation of the program." << std::endl
		<< "-b <bindir>    : wrapper installation directory. (Default is ${prefix}/bin/)" << std::endl
		<< "-S <setlggdir> : virtual machine location directory. (Default is ${prefix}/share/${wrapper}/)" << std::endl
		<< "-P <prgdir>    : customisation of virtual machine program directory. (Default is ${prefix}/share/${wrapper}/)" << std::endl
		<< "-M <memdir>    : customisation of virtual machine memory directory. (Default is ${prefix}/share/${wrapper}/)" << std::endl
		<< "-C <coredir>   : customisation of virtual machine coredump directory. (Default is /tmp/)" << std::endl
		<< "-L <libdir>    : customisation of virtual machine plugins directory. (Default is ${prefix}/share/${wrapper}/plugins/)" << std::endl
		<< "-i             : display an installation script." << std::endl
		<< "-I <srcdir>    : display an installation script from the specified directory." << std::endl
		<< "-r             : display an uninstallation script." << std::endl
		<< "-R <srcdir>    : display an uninstallation script from the specified directory." << std::endl
		<< "-q             : quiet mode." << std::endl
		<< "-h             : display this help and exit." << std::endl
		<< "-v             : display the software version." << std::endl
		<< std::endl
		<< "Notes:" << std::endl
		<< "- In directories arguments, ${prefix} is replaced by the value of the -p argument, and ${wrapper} is replaced by the wrapper name." << std::endl
		<< "- If the -p option is not present, the command performs an installation in the current directory." << std::endl
		<< "- All outputs can be run in a shell script by surrounding the command with: eval $( ... )" << std::endl
		<< std::endl;
}

void genere_installation(std::ostream& os, const ValeurOptionSP& source)
{
	std::cout	<< "( " << std::endl;
	if(source)
	{
		std::cout	<< "cd " << *source << " ;" << std::endl;
	}
	std::cout	<< std::endl
			<< "if [ \"x$PREFIX\" = x ] ;" << std::endl
			<< "then" << std::endl
			<< "\texit ;" << std::endl
			<< "fi ;" << std::endl
			<< std::endl
			<< "mkdir -p $WRAPPER_DIR ;" << std::endl
			<< "cp $WRAPPER_NAME $WRAPPER_DIR ;" << std::endl
			<< std::endl
			<< "mkdir -p $PROGRAM_DIR ;" << std::endl
			<< "cp $PROGRAM_NAME $PROGRAM_DIR ;" << std::endl
			<< "chmod a+r,go-w $PROGRAM_DIR$PROGRAM_NAME ;" << std::endl
			<< std::endl
			<< "if [ \"x$MEMORY_NAME\" != x ] ;" << std::endl
			<< "then" << std::endl
			<< "\tmkdir -p $MEMORY_DIR ;" << std::endl
			<< "\tif [ -f $MEMORY_NAME ] ;" << std::endl
			<< "\tthen" << std::endl
			<< "\t\tcp $MEMORY_NAME $MEMORY_DIR ;" << std::endl
			<< "\telse" << std::endl
			<< "\t\ttouch $MEMORY_DIR$MEMORY_NAME ;" << std::endl
			<< "\tfi ;" << std::endl
			<< "\tchmod a+rw $MEMORY_DIR$MEMORY_NAME ;" << std::endl
			<< "fi ;" << std::endl
			<< std::endl
			<< "mkdir -p $COREDUMP_DIR ;" << std::endl
			<< "chmod a+rwx,+t $COREDUMP_DIR ;" << std::endl
			<< std::endl
			<< "if [ ! -d $PLUGIN_DIR ] ;" << std::endl
			<< "then" << std::endl
			<< "\techo \"Warning: Plugin directory $PLUGIN_DIR does not exist.\" >&2 ;" << std::endl
			<< "fi ;" << std::endl
			<< std::endl
			<< ")" << std::endl
	;
}

void genere_desintallation(std::ostream& os, const ValeurOptionSP& source)
{
	std::cout	<< "( " << std::endl;
	if(source)
	{
		std::cout	<< "cd " << *source << " ;" << std::endl;
	}
	std::cout	<< std::endl
			<< "if [ \"x$PREFIX\" = x ] ;" << std::endl
			<< "then" << std::endl
			<< "\texit ;" << std::endl
			<< "fi ;" << std::endl
			<< std::endl
			<< "[ -f $WRAPPER_DIR$WRAPPER_NAME ] && rm $WRAPPER_DIR$WRAPPER_NAME ; " << std::endl
			<< "[ -f $PROGRAM_DIR$PROGRAM_NAME ] && rm $PROGRAM_DIR$PROGRAM_NAME ; " << std::endl
			<< "[ -f $MEMORY_DIR$MEMORY_NAME ] && rm $MEMORY_DIR$MEMORY_NAME ; " << std::endl
			<< std::endl
			<< ")" << std::endl
	;
}

int main(int argc, char *argv[])
{
	std::string fichier;
	SetLgg::Adapteur::ValeurOptionSP option_prefixe;
	SetLgg::Adapteur::ValeurOptionSP option_adapteur;
	SetLgg::Adapteur::ValeurOptionSP option_setlgg;
	SetLgg::Adapteur::ValeurOptionSP option_programme;
	SetLgg::Adapteur::ValeurOptionSP option_memoire;
	SetLgg::Adapteur::ValeurOptionSP option_miseaplat;
	SetLgg::Adapteur::ValeurOptionSP option_plugin;
	bool installation=false;
	bool desinstallation=false;
	ValeurOptionSP option_source;
	bool verbeux=true;
	bool creation_repertoires=false;
	opterr = 0;
	for(;;)
	{
		int c;
		c = ::getopt(argc,argv,"hvqp:b:S:P:M:C:L:iI:rR:");
		if(c==EOF)
		{
			break;
		}
		switch(c)
		{ 
			case 'p':
				option_prefixe = ValeurOptionSP(new std::string(optarg)); 
				break;
			case 'b':
				option_adapteur = ValeurOptionSP(new std::string(optarg)); 
				break;
			case 'S':
				option_setlgg = ValeurOptionSP(new std::string(optarg)); 
				break;
			case 'P':
				option_programme = ValeurOptionSP(new std::string(optarg)); 
				break;
			case 'M':
				option_memoire = ValeurOptionSP(new std::string(optarg)); 
				break;
			case 'C':
				option_miseaplat = ValeurOptionSP(new std::string(optarg)); 
				break;
			case 'L':
				option_plugin = ValeurOptionSP(new std::string(optarg)); 
				break;
			case 'i':
				installation = true;
				option_source = ValeurOptionSP();
				break;
			case 'I':
				installation = true;
				option_source = ValeurOptionSP(new std::string(optarg));
				break;
			case 'r':
				desinstallation = true;
				option_source = ValeurOptionSP();
				break;
			case 'R':
				desinstallation = true;
				option_source = ValeurOptionSP(new std::string(optarg));
				break;
			case 'q':
				verbeux = false;
				break;
			case 'h':
				affiche_aide(std::cerr,argv[0]);
				return 0;
				break;
			case 'v':
				std::cerr << argv[0] << " version " << PACKAGE_VERSION << "." << std::endl
					<< std::endl
					<< argv[0] << " Copyright (C) 2012  Julien Bruguier" << std::endl
					<< "This program comes with ABSOLUTELY NO WARRANTY. " << std::endl
					<< "This is free software, and you are welcome to redistribute it" << std::endl
					<< "under certain conditions. Please refer to the licence GPLv3" << std::endl
					<< "for details." << std::endl;

				return 0;
				break;
			default:
				std::cerr << "Invalid option -" << static_cast<char>(optopt) << ". Try -h option." << std::endl;
				return ERREUR_LIGNEDECOMMANDE;
				break;
		}
	}
	if(installation)
	{
		genere_installation(std::cout,option_source);
		return 0;
	}
	if(desinstallation)
	{
		genere_desintallation(std::cout,option_source);
		return 0;
	}
	if(optind<argc)
	{
		fichier = std::string(argv[optind]);
	}
	else
	{
		fichier = "-";
	}
	try
	{
		AdapteurSP adapte=SetLgg::Adapteur::Analyseur::analyse(fichier);
		if(adapte)
		{
			SetLgg::Adapteur::InstallationSP installation(new SetLgg::Adapteur::Installation(option_prefixe,option_adapteur,option_setlgg,option_programme,option_memoire,option_miseaplat,option_plugin,adapte->nom_adapteur(),adapte->nom_programme(),adapte->nom_memoire()));
			if(verbeux)
			{
				std::cout << (*installation) << std::endl;
			}
			adapte->_installation = installation;
			(*adapte)();
		}
	}
	catch(SetLgg::Global::Exception::Generique& e)
	{
		e.format(std::cerr);
		return 1;
	}
	return 0;
}

#endif
