/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _EXCEPTIONS_H_
#define _EXCEPTIONS_H_
#include <iostream>
#include <string>
#include <set>
#include <signal.h>

#include <src/global/global.h>

#include <src/global/sources/position.h>
namespace SetLgg { namespace Global { namespace Exception { DECL_SHARED_PTR(Generique); } } }

namespace SetLgg
{
	namespace Global
	{
		namespace Exception
		{
			std::ostream& details(std::ostream& os);

			class Generique
			{
				public:
					Generique() = default;
					void format(std::ostream& os) const;
					void change_position(SetLgg::Global::Source::PositionSP position)
					{
						if(position)
							_position = position;
					};
					virtual void banzai() const { decltype(*this) exception = *this; throw exception; };
				protected:
					Generique(SetLgg::Global::Source::PositionSP position, std::string message)
						:_position(position),_message(message) {};
					Generique(std::string message)
						:_message(message) {};
					SetLgg::Global::Source::PositionSP _position;
					std::string _message;
					std::string message() const
					{
						if(_position)
							return std::string(*_position)+_message;
						return _message; 
					};
			};


			class Compilation : public Generique
			{
				public:
					virtual void banzai() const { decltype(*this) exception = *this; throw exception; };
					Compilation(SetLgg::Global::Source::PositionSP position, std::string message)
						:Generique(position,std::string("Compilation error: ")+message) {};
			};

			class Execution : public Generique
			{
				public:
					Execution(size_t interruption, std::string message)
						:Generique(std::string("Runtime error (")+Execution::nom_interruption(interruption)+("): ")+message),_interruption(interruption),_origine(message) {};
					operator size_t() const
					{
						return _interruption;
					}
					operator std::string() const
					{
						return _origine;
					}
					static std::string nom_interruption(const size_t& signal)
					{
						switch(signal)
						{
							case SIGHUP:
								return "HUP";
							case SIGINT:	
								return "KBINT";
							case SIGQUIT:
								return "QUIT";
							case SIGILL:
								return "ILL";
							case SIGABRT:
								return "ABRT";
							case SIGFPE:
								return "FPE";
							case SIGSEGV:
								return "SEGV";
							case SIGPIPE:
								return "PIPE";
							case SIGALRM:
								return "ALRM";
							case SIGTERM:
								return "TERM";
							case SIGUSR1:
								return "USR1";
							case SIGUSR2:
								return "USR2";
							case SIGCHLD:
								return "CHLD";
							case SIGCONT:
								return "CONT";
							case SIGTSTP:
								return "TSTP";
							case SIGTTIN:
								return "TTIN";
							case SIGTTOU:
								return "TTOU";
							case SIGBUS:
								return "BUS";
							case SIGPROF:
								return "PROF";
							case SIGURG:
								return "URG";
							case SIGIO:
								return "IO";
							default:
								return "";
								break;
						}
					}
					virtual void banzai() const { decltype(*this) exception = *this; throw exception; };
				private:
					size_t _interruption;
					std::string _origine;
			};

			class ErreurDeSyntaxe : public Compilation
			{
				public:
					ErreurDeSyntaxe(SetLgg::Global::Source::PositionSP position, std::string message)
						:Compilation(position,std::string("Syntax error.")) {};
			};

			class PasDeFichierSource : public Compilation
			{
				public:
					PasDeFichierSource(std::string fichier_source)
						:Compilation(SetLgg::Global::Source::PositionSP(),std::string("Source file ")+fichier_source+std::string(" is invalid.")) {};
			};

			class InstructionIllegale : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionIllegale(const std::string& instruction)
					:Execution(SIGILL,std::string("Illegal instruction ")+instruction) { }
			};

			class Inconnue : public SetLgg::Global::Exception::Generique
			{
				public:
					Inconnue(const std::string& message)
					:Generique(message) { }
			};


			class Integration : public SetLgg::Global::Exception::Generique
			{
				protected:
					Integration(std::string message)
						:Generique(SetLgg::Global::Source::PositionSP(),std::string("Startup error: ")+message) {};
			};

			class Adapteur : public SetLgg::Global::Exception::Generique
			{
				protected:
					Adapteur(std::string message)
						:Generique(SetLgg::Global::Source::PositionSP(),std::string("Wrapper error: ")+message) {};
			};
		}
	}
}
#endif
