/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _POINTEUR_EPHEMERE_H_
#define _POINTEUR_EPHEMERE_H_

#include <src/global/global.h>

#define EPHEMERAL_PTR(Type) SetLgg::Global::PointeurEphemere< Type >

#define DECL_EPHEMERAL_PTR(Type) \
class Type;\
typedef EPHEMERAL_PTR(Type) Type##EP;\
typedef EPHEMERAL_PTR(const Type) Type##CEP

namespace SetLgg
{
	namespace Global
	{
		template<typename Type>
		struct Supprime
		{
			void operator() (const Type* type)
			{
				delete type;
			}
		};

		// Non utilise au final, car il accelere la VM de 0.3% seulement...
		template<typename Type, template<typename Type_> class Suppression=Supprime>
		struct PointeurEphemere
		{
			PointeurEphemere()
			:_valeur(nullptr), _asupprimer(true) {};

			PointeurEphemere(Type *valeur)
			:_valeur(valeur),_asupprimer(true) {};

			PointeurEphemere(const SHARED_PTR(Type)& pointeur)
			:_valeur(pointeur.get()), _asupprimer(false) {};

			PointeurEphemere(const PointeurEphemere& pointeur) = delete;

			PointeurEphemere(PointeurEphemere&& pointeur)
			:_valeur(pointeur._valeur), _asupprimer(pointeur._asupprimer) { };

			~PointeurEphemere()
			{
				if(_asupprimer)
				{
					Suppression<Type> supprime;
					supprime(_valeur);
				}
			};
	
			operator SHARED_PTR(Type) ()
			{
				if(not _asupprimer)
					throw;
				_asupprimer=false;
				SHARED_PTR(Type) pointeur(_valeur);
				return pointeur;
			};

			Type& operator* ()
			{
				return *_valeur;
			};

			const Type& operator* () const
			{
				return *_valeur;
			};

			Type* operator->()
			{
				return _valeur;
			};

			const Type* operator->() const
			{
				return _valeur;
			};

			operator bool () const
			{
				return _valeur!=nullptr;
			};

			template<typename AutreType>
			friend PointeurEphemere<AutreType> dynamic_pointer_cast(const PointeurEphemere<Type>& pointeur)
			{
				PointeurEphemere<AutreType> autre_pointeur(dynamic_cast<AutreType*>(pointeur._valeur), false);
				return autre_pointeur;
			};

			template<typename AutreType>
			friend PointeurEphemere<AutreType> static_pointer_cast(const PointeurEphemere<Type>& pointeur)
			{
				PointeurEphemere<AutreType> autre_pointeur(static_cast<AutreType*>(pointeur._valeur), false);
				return autre_pointeur;
			};

			template<typename AutreType>
			friend PointeurEphemere<AutreType> const_pointer_cast(const PointeurEphemere<Type>& pointeur)
			{
				PointeurEphemere<AutreType> autre_pointeur(const_cast<AutreType*>(pointeur._valeur), false);
				return autre_pointeur;
			};

			template<typename AutreType>
			friend PointeurEphemere<AutreType> reinterpret_pointer_cast(const PointeurEphemere<Type>& pointeur)
			{
				PointeurEphemere<AutreType> autre_pointeur(reinterpret_cast<AutreType*>(pointeur._valeur), false);
				return autre_pointeur;
			};

			private:
				PointeurEphemere(const Type *valeur, const bool asupprimer)
				:_valeur(valeur), _asupprimer(asupprimer) {};
				Type *_valeur;
				bool _asupprimer;

		};
	}
}

#endif
