/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 
#include <sys/types.h>
#include <unistd.h>

#include <src/machine/debogueur/donnees/debogueur.h>
#include <src/machine/debogueur/commandes/analyseur.h>
#include <src/machine/debogueur/directives/analyseur.h>
#include <src/machine/debogueur/donnees/resultat.h>
#include <src/machine/memoire/donnees/chaine.h>
using namespace SetLgg::Machine::Debogueur;


bool Debogueur::execution(SetLgg::Machine::Machine::MachineSP machine)
{
	std::cout << std::endl
		<< "+---------------------------------------------------------------------+" << std::endl
		<< "| Please use the 'help' command to display the debugger command list. |" << std::endl
		<< "+---------------------------------------------------------------------+" << std::endl
		<< std::endl;
	machine->_debogueur->_machine = machine;
	machine->_debogueur->execute_directives(machine);
	machine->_debogueur->affiche_instruction_source();
	machine->_debogueur->affiche_instruction_assembleur(machine);
	while(std::cin.good() or not machine->_debogueur->_prochaine_commande.empty())
	{
		std::string ligne_commande;
		machine->_debogueur->affiche_prompt();
		if(not machine->_debogueur->_prochaine_commande.empty())
		{
			std::cout << machine->_debogueur->_prochaine_commande << std::endl;
			ligne_commande=machine->_debogueur->_prochaine_commande;
			machine->_debogueur->_prochaine_commande.clear();
		}
		else
		{
			std::getline(std::cin,ligne_commande);
		}
		machine->_debogueur->ajoute_commande_historique(ligne_commande);
		std::string commande_complete = "command ";
		commande_complete += ligne_commande;
		if(SetLgg::Machine::Debogueur::AnalyseurCommandes::analyse_chaine(machine->_debogueur,commande_complete)==0)
		{
			std::cout << "Invalid command. Please use the 'help' command for more details on commands." << std::endl;
		}
		if(machine->_debogueur->_sortie!=TypeAction::DEBOGUAGE)
		{
			if(machine->_debogueur->_sortie==TypeAction::SORTIE)
			{
				return false;
			}
			return machine->_debogueur->_execution_en_cours;
		}
	}
	return machine->_debogueur->_execution_en_cours;
}


void Debogueur::directive(const std::string& directive, const SetLgg::Machine::Programme::ProgrammeSP& programme, const SetLgg::Machine::Programme::ConditionSP& condition, const SetLgg::Global::Source::PositionSP& position)
{
	SetLgg::Machine::Programme::AdresseSP adresse(new SetLgg::Machine::Programme::Adresse(programme->_adresse_terminale));
	std::string directive_complete = "code ";
	directive_complete += directive;
	if(SetLgg::Machine::Debogueur::AnalyseurDirectives::analyse_chaine(this->shared_from_this(),directive_complete,condition,adresse,position)==0)
	{
		std::cerr << static_cast<std::string>(*position) << "Invalid debug directive: " <<  directive << std::endl;
	}
}

void Debogueur::directive_debut_trame(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
{
	DirectiveSP directive(new DirectiveDebutTrame(adresse,position,condition));
	ajoute_directive(directive);
	_lignes[position->fichier()][position->ligne()]=adresse;
}

void Debogueur::directive_fin_trame(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
{
	DirectiveSP directive(new DirectiveFinTrame(adresse,position,condition));
	ajoute_directive(directive);
	_lignes[position->fichier()][position->ligne()]=adresse;
}

void Debogueur::directive_instruction(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& code, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
{
	DirectiveSP directive(new DirectiveInstruction(adresse,position,code,condition));
	ajoute_directive(directive);
	_sources[position->fichier()][position->ligne()] += code;
	_lignes[position->fichier()][position->ligne()]=adresse;
}

void Debogueur::directive_symbole(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& symbole, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
{
	DirectiveSP directive(new DirectiveSymbole(adresse,position,symbole,condition));
	ajoute_directive(directive);
	_symboles[symbole] = adresse;
	_lignes[position->fichier()][position->ligne()]=adresse;
}

void Debogueur::directive_variable(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& variable, const SetLgg::Machine::Programme::AdresseMemoireSP& adresse_memoire,  const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
{
	DirectiveSP directive(new DirectiveVariable(adresse,position,variable,adresse_memoire,condition));
	ajoute_directive(directive);
}

void Debogueur::commande_affiche_fichier(const std::string& regex)
{
	if(_lignes.empty())
	{
		std::cout << "No files defined." << std::endl;
		return;
	}
	std::cout << "Defined files matching pattern " << regex << ":" << std::endl;
	SetLgg::Machine::Memoire::Chaine expression(regex);
	Resultat<ColonneCadre<Alignement::DROITE,80>, ColonneCadre<Alignement::GAUCHE,150> > resultat;
	resultat & "File name" & "Lines";
	for(auto itfile=_lignes.cbegin() ; itfile!=_lignes.cend() ; ++itfile)
	{
		SetLgg::Machine::Memoire::Chaine nom(itfile->first);
		try
		{
			if(static_cast<std::string>(nom/expression).size()>0)
			{
				std::ostringstream lignes;
				size_t debut = (itfile->second.begin())->first;
				size_t fin = debut;
				auto itligne = itfile->second.cbegin();
				for(++itligne; itligne!=itfile->second.cend() ; ++itligne)
				{
					if(itligne->first==(fin+1))
					{
						++fin;
					}
					else
					{
						if(debut==fin)
						{
							lignes << debut << ", ";
						}
						else
						{
							lignes << debut << "-" << fin << ", ";
						}
						debut=fin=itligne->first;
					}
				}
				lignes << debut;
				if(debut!=fin)
				{
					lignes << "-" << fin;
				}
				resultat | itfile->first | lignes.str();
			}
		}
		catch(SetLgg::Global::Exception::Execution& e)
		{
		}
	}
	std::cout << resultat;
}

void Debogueur::commande_affiche_historique()
{
	enleve_commande_historique();
	if(_historique.empty())
	{
		std::cout << "History is empty." << std::endl;
		return;
	}
	std::cout << "History:" << std::endl;
	Resultat<NumerosInverses<':'>, Colonne<Alignement::GAUCHE>> resultat;
	for(auto it=_historique.cbegin() ; it!=_historique.cend() ; ++it)
	{
		resultat | "" | (*it);
	}
	std::cout << resultat;
}

void Debogueur::commande_affiche_pileappels()
{
	std::cout << "Backtrace:" << std::endl;
	std::cout << _pile_trames;
}

void Debogueur::commande_affiche_pileappels_liste()
{
	std::cout << "Backtrace (list):" << std::endl;
	std::cout < _pile_trames;
}

void Debogueur::commande_affiche_pointarret()
{
	std::cout << _liste_pointsarret;
}

void Debogueur::commande_affiche_source()
{
	Trame& trame = *_pile_trames;
	affiche_source(trame);
}

void Debogueur::commande_affiche_source(const size_t niveau)
{
	if(niveau>=_pile_trames)
	{
		std::cout << "Frame " << niveau << " does not exist." << std::endl;
		return;
	}
	Trame& trame = _pile_trames[niveau];
	affiche_source(trame);
}

void Debogueur::commande_affiche_source(const std::string& fichier, const size_t ligne)
{
	if(not valide_fichier_ligne(fichier,ligne))
		return;
	affiche_source(fichier, ligne);
}

void Debogueur::commande_affiche_surveillance()
{
	if(not _surveillances)
	{
		std::cout << "No watched addresses defined." << std::endl;
		return;
	}
	Resultat<NumerosCadre, ColonneCadre<Alignement::GAUCHE,15>, ColonneCadre<Alignement::DROITE,50> > resultat;
	_surveillances >> resultat;
	std::cout << resultat;
}

void Debogueur::commande_affiche_symbole(const std::string& regex)
{
	if(_symboles.empty())
	{
		std::cout << "No symbols defined." << std::endl;
		return;
	}
	std::cout << "Defined symbols matching pattern " << regex << ":" << std::endl;
	SetLgg::Machine::Memoire::Chaine expression(regex);
	Resultat<ColonneCadre<Alignement::DROITE,20>,ColonneCadre<Alignement::GAUCHE,100> > resultat;
	resultat & "Address" & "Symbol";
	for(auto it=_symboles.cbegin() ; it!=_symboles.cend() ; ++it)
	{
		try
		{
			SetLgg::Machine::Memoire::Chaine nom(it->first);
			if(static_cast<std::string>(nom/expression).size()>0)
			{
				std::ostringstream oss;
				oss << *(it->second);
				resultat | oss.str() | it->first;
			}
		}
		catch(SetLgg::Global::Exception::Execution& e)
		{
		}
	}
	std::cout << resultat;
}

void Debogueur::commande_affiche_trame()
{
	Trame& trame = *_pile_trames;
	std::cout << trame;
}

void Debogueur::commande_affiche_trame(const size_t niveau)
{
	if(niveau>=_pile_trames)
	{
		std::cout << "Frame " << niveau << " does not exist." << std::endl;
		return;
	}
	Trame& trame = _pile_trames[niveau];
	std::cout << trame;
}

void Debogueur::commande_affiche_variable(const std::string& regex)
{
	ListeVariables liste = _pile_trames;
	if(liste.empty())
	{
		std::cout << "No variables defined." << std::endl;
		return;
	}
	SetLgg::Machine::Memoire::Chaine expression(regex);
	Resultat<ColonneCadre<Alignement::DROITE,20>, ColonneCadre<Alignement::GAUCHE, 50>, ColonneCadre<Alignement::DROITE, 20>, ColonneCadre<Alignement::GAUCHE,100> > resultat;
	resultat & "Name" & "Address" & "Real address" & "File" ;
	for(auto it = liste.cbegin() ; it!=liste.cend() ; ++it)
	{
		try
		{
			SetLgg::Machine::Memoire::Chaine nom(it->first);
			if(static_cast<std::string>(nom/expression).size()>0)
			{
				std::ostringstream ossadr;
				ossadr << *(it->second->_adresse);
				std::ostringstream ossadrreelle;
				if(it->second->_adresse_reelle)
				{
					it->second->_adresse_reelle->format(ossadrreelle);
				}
				std::string strpos;
				if(it->second->_position)
				{
					strpos = it->second->_position->debogueur();
				}
				resultat | it->second->_nom | ossadr.str() | ossadrreelle.str() | strpos;
			}
		}
		catch(SetLgg::Global::Exception::Execution& e)
		{
		}
	}
	std::cout << resultat;
}

void Debogueur::commande_ajoute_interruption(const size_t code)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		return;
	}
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	if(machine->_processeur->existe_interruption(code))
	{
		machine->_processeur->interruption(code);
		std::cout << "Interruption " << SetLgg::Global::Exception::Execution::nom_interruption(code) << " enqueued." << std::endl;
	}
	else
	{
		std::cout << "Interruption " << SetLgg::Global::Exception::Execution::nom_interruption(code) << " not enqueued, as no handler is defined for this interruption." << std::endl;
	}
}

void Debogueur::commande_ajoute_pointdarret(const size_t adresse)
{
	SetLgg::Machine::Programme::AdresseSP vraie_adresse(new SetLgg::Machine::Programme::Adresse(adresse));
	PointArretSP point_arret(new PointArretAdresse(vraie_adresse));
	_liste_pointsarret.ajout(point_arret);
	std::cout << "Breakpoint added to address " << (*vraie_adresse) << "." << std::endl ;
}

void Debogueur::commande_ajoute_pointdarret(const std::string& symbole)
{
	auto it=_symboles.find(symbole);
	PointArretSP point_arret;
	if(it==_symboles.end())
	{
		std::cout << "Symbol " << symbole << " does not exist." << std::endl ;
		SetLgg::Machine::Programme::AdresseSP adresse;
		point_arret = PointArretSP(new PointArretSymbole(adresse,symbole));
	}
	else
	{
		point_arret = PointArretSP(new PointArretSymbole(it->second,symbole));
	}
	_liste_pointsarret.ajout(point_arret);
	std::cout << "Breakpoint added to symbol " << symbole << "." << std::endl ;
}

void Debogueur::commande_ajoute_pointdarret(const std::string& fichier, const size_t ligne)
{
	SetLgg::Machine::Programme::AdresseSP vraie_adresse = valide_fichier_ligne(fichier,ligne);
	PointArretSP point_arret(new PointArretFichier(vraie_adresse,fichier,ligne));
	_liste_pointsarret.ajout(point_arret);
	std::cout << "Breakpoint added to file " << fichier << " " << ligne << "." << std::endl ;
}

void Debogueur::commande_ajoute_surveillance(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse)
{
	SurveillanceSP surveillance(new SurveillanceAdresseMemoire(adresse));
	_surveillances.ajout(surveillance);
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	affiche_surveillances(machine);
	std::cout << "Watched address " << (*adresse) << " added." << std::endl;
}

void Debogueur::commande_ajoute_surveillance(const std::string& variable)
{
	ListeVariables liste = _pile_trames;
	auto var = liste.find(variable);
	if(var==liste.end())
	{
		std::cout << "Variable " << variable << " does not exist." << std::endl;
		return;
	}
	SurveillanceSP surveillance(new SurveillanceVariable(var->second));
	_surveillances.ajout(surveillance);
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	affiche_surveillances(machine);
	std::cout << "Watched variable " << variable << " added." << std::endl;
}

void Debogueur::commande_ecriture_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse, const SetLgg::Machine::Programme::ValeurSP& valeur)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		return;
	}
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	std::ostringstream ossadr;
	ossadr << (*adresse);
	SetLgg::Machine::Memoire::AdresseSP adresse_reelle = Memoire::resolution(adresse,machine);
	if(not adresse_reelle)
	{
		std::cout << "Address " << ossadr.str() << " is invalid." << std::endl; 
		return;
	}
	if(valeur)
	{
		std::ostringstream ossval;
		ossval << (*valeur);
		SetLgg::Machine::Memoire::ValeurCSP valeur_reelle = Memoire::evaluation(valeur,machine);
		if(not valeur_reelle)
		{
			std::cout << "Value " << ossval.str() << " is invalid." << std::endl; 
			return;
		}
		if(not Memoire::ecriture(*adresse_reelle,valeur_reelle,machine))
		{
			std::cout << "Value " << ossval.str() << " not written into address " << ossadr.str() << "." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Value " << ossval.str() << " written into address " << ossadr.str() << "." << std::endl; 
	}
	else
	{
		if(not Memoire::ecriture(*adresse_reelle,SetLgg::Machine::Memoire::ValeurCSP(),machine))
		{
			std::cout << "Address " << ossadr.str() << " not cleared." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Address " << ossadr.str() << " cleared." << std::endl; 
	}
}

void Debogueur::commande_ecriture_memoire(const std::string& variable, const SetLgg::Machine::Programme::ValeurSP& valeur)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		return;
	}
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	ListeVariables liste = _pile_trames;
	auto var = liste.find(variable);
	if(var==liste.end())
	{
		std::cout << "Variable " << variable << " does not exist." << std::endl;
		return;
	}
	if(valeur)
	{
		std::ostringstream ossval;
		ossval << (*valeur);
		SetLgg::Machine::Memoire::ValeurCSP valeur_reelle = Memoire::evaluation(valeur,machine);
		if(not valeur_reelle)
		{
			std::cout << "Value " << ossval.str() << " is invalid." << std::endl; 
			return;
		}
		if(not var->second->ecriture(valeur_reelle,machine))
		{
			std::cout << "Value " << ossval.str() << " not written into variable " << variable << "." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Value " << ossval.str() << " written into variable " << variable << "." << std::endl; 
	}
	else
	{
		if(not var->second->ecriture(SetLgg::Machine::Memoire::ValeurCSP(),machine))
		{
			std::cout << "Variable " << variable << " not cleared." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Variable " << variable << " cleared." << std::endl; 
	}
}

void Debogueur::commande_ecriture_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse, const SetLgg::Machine::Programme::AdresseMemoireSP& valeur)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		return;
	}
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	std::ostringstream ossadr;
	ossadr << (*adresse);
	SetLgg::Machine::Memoire::AdresseSP adresse_reelle = Memoire::resolution(adresse,machine);
	if(not adresse_reelle)
	{
		std::cout << "Address " << ossadr.str() << " is invalid." << std::endl; 
		return;
	}
	if(valeur)
	{
		std::ostringstream ossval;
		ossval << (*valeur);
		SetLgg::Machine::Memoire::ValeurCSP valeur_reelle = Memoire::resolution(valeur,machine);
		if(not valeur_reelle)
		{
			std::cout << "Value " << ossval.str() << " is invalid." << std::endl; 
			return;
		}
		if(not Memoire::ecriture(*adresse_reelle,valeur_reelle,machine))
		{
			std::cout << "Value " << ossval.str() << " not written into address " << ossadr.str() << "." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Value " << ossval.str() << " written into address " << ossadr.str() << "." << std::endl; 
	}
	else
	{
		if(not Memoire::ecriture(*adresse_reelle,SetLgg::Machine::Memoire::ValeurCSP(),machine))
		{
			std::cout << "Address " << ossadr.str() << " not cleared." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Address " << ossadr.str() << " cleared." << std::endl; 
	}
}

void Debogueur::commande_ecriture_memoire(const std::string& variable, const SetLgg::Machine::Programme::AdresseMemoireSP& valeur)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		return;
	}
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	ListeVariables liste = _pile_trames;
	auto var = liste.find(variable);
	if(var==liste.end())
	{
		std::cout << "Variable " << variable << " does not exist." << std::endl;
		return;
	}
	if(valeur)
	{
		std::ostringstream ossval;
		ossval << (*valeur);
		SetLgg::Machine::Memoire::ValeurCSP valeur_reelle = Memoire::resolution(valeur,machine);
		if(not valeur_reelle)
		{
			std::cout << "Value " << ossval.str() << " is invalid." << std::endl; 
			return;
		}
		if(not var->second->ecriture(valeur_reelle,machine))
		{
			std::cout << "Value " << ossval.str() << " not written into variable " << variable << "." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Value " << ossval.str() << " written into variable " << variable << "." << std::endl; 
	}
	else
	{
		if(not var->second->ecriture(SetLgg::Machine::Memoire::ValeurCSP(),machine))
		{
			std::cout << "Variable " << variable << " not cleared." << std::endl; 
			return;
		}
		affiche_surveillances(machine);
		std::cout << "Variable " << variable << " cleared." << std::endl; 
	}
}

void Debogueur::commande_execute_continue()
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	_sonde = SondeSP(new SondeContinuation(*_pile_trames));
	execute_programme_debug(machine);
	affiche_instruction_source();
}

void Debogueur::commande_execute_execution()
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	_sonde = SondeSP(new SondeExecution());
	execute_programme_debug(machine);
	affiche_instruction_source();
}

void Debogueur::commande_execute_instruction()
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	_sonde = SondeSP(new SondeInstruction());
	execute_programme_debug(machine);
	affiche_instruction_assembleur(machine);
}

void Debogueur::commande_execute_pasapas()
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	_sonde = SondeSP(new SondePasAPas());
	execute_programme_debug(machine);
	affiche_instruction_source();
}

void Debogueur::commande_execute_suivant()
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	_sonde = SondeSP(new SondeSuivant(*_pile_trames));
	execute_programme_debug(machine);
	affiche_instruction_source();
}

void Debogueur::commande_explique_adresse(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse)
{
	try
	{
		std::cout << "Explain address " << (*adresse) << ":" << std::endl;
		SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
		adresse->explication(std::cout,machine);
	}
	catch(SetLgg::Global::Exception::Execution& e)
	{
		std::cout << std::endl;
		e.format(std::cout);
	}
}

void Debogueur::commande_lecture_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse)
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	Memoire memoire;
	Resultat<Colonne<Alignement::DROITE,50>, Colonne<Alignement::GAUCHE,15>, Colonne<Alignement::DROITE,15>, Colonne<Alignement::GAUCHE,20>, Colonne<Alignement::DROITE,50> > resultat;
	std::ostringstream oss;
	oss << (*adresse);
	SetLgg::Machine::Memoire::AdresseSP adresse_reelle = Memoire::resolution(adresse,machine);
	if(adresse_reelle)
	{
		memoire = Memoire::lecture(*adresse_reelle,machine);
	}
	resultat & "Address" & "State"       & "Real address"   & "Type"        & "Value";
	resultat | oss.str() | memoire._etat | memoire._adresse | memoire._type | memoire._valeur ;
	std::cout << resultat;
}

void Debogueur::commande_lecture_memoire(const std::string& variable)
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	ListeVariables liste = _pile_trames;
	auto var = liste.find(variable);
	if(var==liste.end())
	{
		std::cout << "Variable " << variable << " does not exist." << std::endl;
		return;
	}
	Resultat<Colonne<Alignement::DROITE,20>, Colonne<Alignement::DROITE,50>, Colonne<Alignement::GAUCHE,15>, Colonne<Alignement::DROITE,15>, Colonne<Alignement::GAUCHE,20>, Colonne<Alignement::DROITE,50> , Colonne<Alignement::GAUCHE,100> > resultat;
	resultat & "Name" & "Address" & "State" & "Real address" & "Type" & "Value" & "File";
	var->second->affiche(resultat,*_pile_trames,machine);
	std::cout << resultat;
}

void Debogueur::commande_liberation_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		return;
	}
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	SetLgg::Machine::Memoire::AdresseSP adresse_reelle = Memoire::resolution(adresse,machine);
	std::ostringstream oss;
	oss << (*adresse);
	if(adresse_reelle)
	{
		if(Memoire::suppression(*adresse_reelle,machine))
		{
			affiche_surveillances(machine);
			std::cout << "Address " << oss.str() << " deleted." << std::endl;
			return;
		}
		std::cout << "Address " << oss.str() << " is not defined." << std::endl;
		return;
	}
	std::cout << "Address " << oss.str() << " is invalid." << std::endl;
}

void Debogueur::commande_liberation_memoire(const std::string& variable)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		return;
	}
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	ListeVariables liste = _pile_trames;
	auto var = liste.find(variable);
	if(var==liste.end())
	{
		std::cout << "Variable " << variable << " does not exist." << std::endl;
		return;
	}
	if(var->second->supprime(machine))
	{
		affiche_surveillances(machine);
		std::cout << "Variable " << variable << " deleted." << std::endl;
		return;
	}
	std::cout << "Variable " << variable << " not deleted." << std::endl;
}

void Debogueur::commande_metaplat_flux()
{
	std::cout << "Dump streams:" << std::endl;
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	std::cout << *(machine->_flux) << std::endl;
}

void Debogueur::commande_metaplat_machine()
{
	std::cout << "Dump virtual machine:" << std::endl;
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	std::cout << *(machine) << std::endl;
}

void Debogueur::commande_metaplat_memoire()
{
	std::cout << "Dump memory:" << std::endl;
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	std::cout << *(machine->_memoire) << std::endl;
}

void Debogueur::commande_metaplat_processeur()
{
	std::cout << "Dump processor:" << std::endl;
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	std::cout << *(machine->_processeur) << std::endl;
}

void Debogueur::commande_metaplat_programme()
{
	std::cout << "Dump program:" << std::endl;
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	std::cout << *(machine->_programme) << std::endl;
}

void Debogueur::commande_rejoue_historique(const size_t commande)
{
	enleve_commande_historique();
	if(commande>=_historique.size())
	{
		std::cout << "Command " << commande << " does not exist." << std::endl;
		return;
	}
	_prochaine_commande = _historique[_historique.size()-(commande+1)] ;
}

void Debogueur::commande_sortie(const TypeAction action)
{
	switch(action)
	{
		case TypeAction::SORTIE:
			std::cout << "Exiting." << std::endl;
			break;
		case TypeAction::CONTINUE:
			if(not _execution_en_cours)
			{
				std::cout << "Execution already terminated." << std::endl;
				return;
			}
			std::cout << "Continuing without debugger." << std::endl;
			break;
		default:
			throw;
	}
	_sortie=action;
}

void Debogueur::commande_supprime_pointdarret(const size_t numero)
{
	if(_liste_pointsarret.enleve(numero))
	{
		std::cout << "Breakpoint number " << numero << " deleted." << std::endl;
	}
}

void Debogueur::commande_supprime_pointdarret_tous()
{
	_liste_pointsarret.enleve_tout();
	std::cout << "All breakpoints deleted." << std::endl;
}

void Debogueur::commande_supprime_surveillance(const size_t numero)
{
	if(_surveillances.enleve(numero))
	{
		std::cout << "Watched address number " << numero << " deleted." << std::endl;
	}
}

void Debogueur::commande_supprime_surveillance_tous()
{
	_surveillances.enleve_tout();
	std::cout << "All watched addresses deleted." << std::endl;
}

void Debogueur::ajoute_commande_historique(const std::string& commande)
{
	_historique.push_back(commande);
}

bool Debogueur::enleve_commande_historique()
{
	if(_historique.size()==0)
	{
		std::cout << "Impossible to execute the last command again." << std::endl;
		return false;
	}
	_historique.pop_back();
	return true;
}

bool Debogueur::execution_pointdarret()
{
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	execute_directives(machine);
	SetLgg::Machine::Programme::Adresse adresse = machine->_processeur->etat_courant();
	if(_sonde)
	{
		_sonde->incremente_nb_instructions();
		if(_sonde->arret_debogage(((adresse<_directives.size())?(_directives[adresse]):(ListeDirectives())),_pile_trames))
		{
			return true;
		}
	}
	return _liste_pointsarret == adresse;
}

void Debogueur::affiche_prompt()
{
	std::cout << std::endl << "--------------------------------------------------------------------------------" << std::endl;
	pid_t pid = ::getpid();
	SetLgg::Machine::Machine::MachineSP machine = _machine.lock();
	SetLgg::Machine::Programme::Adresse adresse = machine->_processeur->etat_courant();
	std::cout << std::endl << "PID " << pid << " address " << adresse  << " debug: ";
}

void Debogueur::ajoute_directive(const DirectiveSP& directive)
{
	size_t adresse=*(directive->_adresse);
	while(_directives.size()<=adresse)
	{
		_directives.push_back(ListeDirectives());
	}
	_directives[adresse].ajout(directive);
}

void Debogueur::ajout_debogueur(const SetLgg::Machine::Debogueur::DebogueurSP& debogueur, const SetLgg::Machine::Programme::Adresse& adresse_terminale)
{
	if(not debogueur)
		return;
	for(auto it=debogueur->_directives.cbegin() ; it!=debogueur->_directives.cend() ; ++it)
	{
		for(auto itt=it->cbegin() ; itt!=it->cend() ; ++itt)
			(*itt)->ajout_debogueur(this,adresse_terminale);
	}
	_liste_pointsarret(_symboles,_lignes);
}

SetLgg::Machine::Programme::AdresseSP Debogueur::valide_fichier_ligne(const std::string& fichier, const size_t ligne)
{
	auto itfichier=_lignes.find(fichier);
	if(itfichier==_lignes.end())
	{
		std::cout << "File " << fichier << " does not exist." << std::endl;
		return SetLgg::Machine::Programme::AdresseSP();
	}
#ifdef SETLGG_DEBUG
	for(auto it = itfichier->second.cbegin() ; it!=itfichier->second.cend() ; ++it)
	{
		std::cout << it->first << " ";
	}
#endif
	auto itligne=itfichier->second.find(ligne);
	if(itligne==itfichier->second.end())
	{
		std::cout << "Line " << ligne << " in file " << fichier << " is not valid." << std::endl;
		return SetLgg::Machine::Programme::AdresseSP();
	}
	return itligne->second;
}

void Debogueur::affiche_source(const std::string& fichier, const size_t ligne)
{
	auto itfichier=_sources.find(fichier);
	if(itfichier==_sources.end())
	{
		std::cout << "Source file " << fichier << " is empty." << std::endl;
		return;
	}
	std::cout << "Source file " << fichier << ":" << std::endl;
	Resultat<Colonne<Alignement::DROITE,20,':'>, Colonne<Alignement::GAUCHE> > resultat;
	for(size_t indice = ((ligne<10)?0:(ligne-10)) ; indice<=(ligne+10) ; ++indice)
	{
		std::ostringstream oss;
		if(indice==ligne)
		{
			oss << "==> ";
		}
		oss << indice;
		auto itligne=itfichier->second.find(indice);
		if(itligne==itfichier->second.end())
		{
			resultat | oss.str() | "";
		}
		else
		{
			resultat | oss.str() | itligne->second;
		}
	}
	std::cout << resultat;
}

void Debogueur::affiche_source(const Trame& trame)
{
	SetLgg::Global::Source::PositionSP position;
	if(trame._instruction_courante)
	{
		position = trame._instruction_courante->_position;
	}
	if(not position and trame._symbole_courant)
	{
		position = trame._symbole_courant->_position;
	}
	if(not position)
	{
		position = trame._position;
	}
	if(not position)
	{
		std::cout << "No file information in frame " << trame._niveau << "." << std::endl;
		return;
	}
	commande_affiche_source(position->fichier(),position->ligne());
}

void Debogueur::execute_directives(SetLgg::Machine::Machine::MachineSP& machine)
{
	SetLgg::Machine::Programme::Adresse adresse = machine->_processeur->etat_courant();
	if(adresse<_directives.size())
	{
		_directives[adresse].raz();
		for(auto it=_directives[adresse]._directives.cbegin() ; it!=_directives[adresse]._directives.cend() ; ++it)
		{
			_pile_trames.traitement(*it,_directives[adresse],machine);
		}
	}
}

void Debogueur::execute_programme_debug(SetLgg::Machine::Machine::MachineSP& machine)
{
	if(not _execution_en_cours)
	{
		std::cout << "Execution already terminated." << std::endl;
		_sonde.reset();
		return;
	}
	_execution_en_cours=false;
	try
	{
		SetLgg::Machine::Processeur::Processeur::execution<SetLgg::Machine::Processeur::Processeur::TypeExecution::DEBUG>(machine);
		_execution_en_cours=true;
		if(machine->_processeur->_interruptions._interruptions_en_attente)
		{
			std::cout << "Interruptions to be processed:";
			std::cout < machine->_processeur->_interruptions._interruptions_en_attente;
			std::cout << std::endl
				<< std::endl;
		}
		affiche_surveillances(machine);
		_liste_pointsarret.pointsarret_atteints(std::cout,machine->_processeur->etat_courant());
	}
	catch(SetLgg::Global::Exception::Execution& e)
	{
		e.format(std::cout);
	}
	catch(SetLgg::Machine::Machine::Machine::CodeRetour& code)
	{
		++_nb_instructions; // car shutdown n'incremente jamais le compteur
		std::cout << std::endl << "Program terminated with return code " << code << "." << std::endl;
	}
	_nb_instructions += _sonde->nb_instructions();
	if(not _execution_en_cours)
	{
		std::cout << "(" << _nb_instructions << " instruction" << ((_nb_instructions==1)?"":"s") << " executed.)" << std::endl;
	}
	_sonde.reset();
}

void Debogueur::affiche_surveillances(SetLgg::Machine::Machine::MachineSP& machine)
{
	if(_surveillances)
	{
		std::cout << "Watched addresses:" << std::endl;
		Resultat<Colonne<Alignement::GAUCHE,15>, Colonne<Alignement::DROITE,50>, Colonne<Alignement::DROITE,50>, Colonne<Alignement::GAUCHE,15>, Colonne<Alignement::DROITE,15>, Colonne<Alignement::GAUCHE,20>, Colonne<Alignement::DROITE,100>, Colonne<Alignement::DROITE,100> > resultat; 
		_surveillances.affiche(resultat,*_pile_trames,machine);
		std::cout << resultat;
	}
}
void  Debogueur::affiche_instruction_source() const
{
	Trame trame = *_pile_trames;
	if(trame._instruction_courante)
	{
		std::cout << "Current instruction: " << *(trame._instruction_courante) << std::endl;
		if(trame._symbole_courant)
		{
			std::cout << "  in current symbol: " << *(trame._symbole_courant) << std::endl;
		}
	}
}

void Debogueur::affiche_instruction_assembleur(SetLgg::Machine::Machine::MachineSP& machine) const
{
	SetLgg::Machine::Programme::Adresse adresse_courante = machine->_processeur->etat_courant();
	if(adresse_courante<=machine->_programme->_adresse_terminale)
	{
		std::cout << "Next assembly instruction: " << *((*(machine->_programme))[adresse_courante]) << std::endl;
	}
}

void Debogueur::aide_related_interface(ResultatAideFeuille& resultat)
{
	resultat | "" | "help help   : display the debugger help.";
	resultat | "" | "help history: display and reexecute debugger commands.";
	resultat | "" | "help quit   : quit the debugger.";
	resultat | "" | "help go     : continue without the debugger.";
}

void Debogueur::aide_related_source(ResultatAideFeuille& resultat)
{
	resultat | "" | "help backtrace     : display the call stack in a complete way.";
	resultat | "" | "help backtrace list: display the call stack in a short way.";
	resultat | "" | "help frame         : describe the content of a frame.";
	resultat | "" | "help source        : display the source code at execution point.";
	resultat | "" | "help source file   : display the source code at given point.";
	resultat | "" | "help list file     : display the source code files.";
	resultat | "" | "help list symbol   : display symbols defined in the source code.";
	resultat | "" | "help list variable : display variables defined in the source code.";
}

void Debogueur::aide_related_machine(ResultatAideFeuille& resultat)
{
	resultat | "" | "help dump machine  : display the internal state of the virtual machine.";
	resultat | "" | "help dump processor: display the internal state of the processor.";
	resultat | "" | "help dump memory   : display the internal state of the memory.";
	resultat | "" | "help dump program  : display the internal state of the program.";
	resultat | "" | "help dump stream   : display the internal state of the streams.";
	resultat | "" | "help explain       : explain how an address is solved in the current context.";
}

void Debogueur::aide_description_execution(ResultatAideFeuille& resultat)
{
	resultat | "" | "When the execution stops, the debugger displays the reached breakpoints, the watched addresses,";
	resultat | "" | "the interruptions that will be taken in account by the processor before the next instruction,";
	resultat | "" | "and a notification when the execution of the program stops.";
	resultat | "" | "";
}

void Debogueur::aide_related_execution(ResultatAideFeuille& resultat)
{
	resultat | "" | "help run        : run the program until its end or a breakpoint.";
	resultat | "" | "help continue   : run the program until the end of the current frame, or a breakpoint.";
	resultat | "" | "help next       : run the program until the next instruction in the current frame, the end of the current frame, or a breakpoint."; 
	resultat | "" | "help step       : run the program until the next instruction or the end of a frame, or a breakpoint.";
	resultat | "" | "help instruction: run the next assembly instruction.";
}

void Debogueur::aide_description_pointsarret(ResultatAideFeuille& resultat)
{
	resultat | "" | "A breakpoint is an address where the processor will temporarly stops the execution of the program,";
	resultat | "" | "and returns to the debugger command line. ";
	resultat | "" | "";
}

void Debogueur::aide_related_pointsarret(ResultatAideFeuille& resultat)
{
	resultat | "" | "help break address   : add a breakpoint on a program address.";
	resultat | "" | "help break symbol    : add a breakpoint on a symbol.";
	resultat | "" | "help break file      : add a breakpoint on a source line.";
	resultat | "" | "help list break      : display the defined breakpoints.";
	resultat | "" | "help delete break    : delete a breakpoint.";
	resultat | "" | "help delete all break: clear all breakpoints.";
}

void Debogueur::aide_description_surveillance(ResultatAideFeuille& resultat)
{
	resultat | "" | "A watched address is a memory address displayed after each memory change or program execution.";
	resultat | "" | "";
}

void Debogueur::aide_related_surveillance(ResultatAideFeuille& resultat)
{
	resultat | "" | "help watch address   : add a watched address on a program address.";
	resultat | "" | "help watch variable  : add a watched address on a symbol.";
	resultat | "" | "help list watch      : display the defined watched addresses.";
	resultat | "" | "help delete watch    : delete a watched address.";
	resultat | "" | "help delete all watch: clear all watched addresses.";
}

void Debogueur::aide_description_memoire(ResultatAideFeuille& resultat)
{
	resultat | "" | "Please refer to the virtual machine documentation for the syntax of addresses and values.";
	resultat | "" | "Variables are linked to the frame where they are defined, and are linked to a memory address.";
	resultat | "" | "";
}

void Debogueur::aide_related_memoire(ResultatAideFeuille& resultat)
{
	resultat | "" | "help read address   : read the memory at the given address.";
	resultat | "" | "help read variable  : read the memory at the address linked to the given variable.";
	resultat | "" | "help write address  : write the memory at the given address.";
	resultat | "" | "help write variable : write the memory at the address linked to the given variable.";
	resultat | "" | "help delete address : delete the memory at the given address.";
	resultat | "" | "help delete variable: delete the memory at the address linked to the given variable.";
}

void Debogueur::commande_aide(const std::string& entree1, const std::string& entree2, const std::string& entree3)
{
	std::cout << std::endl;
	if(Debogueur::aide_rubrique("bt","backtrace",entree1,"",entree2))
	{
		std::cout << "Help: Command backtrace (shortcut: bt)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "backtrace";
		resultat | "" | "";
		resultat | "" | "This command displays the frames in the call stack. The frames are displayed from the more global to the current one.";
		resultat | "" | "The last frame is the current one.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("btl","backtrace",entree1,"list",entree2))
	{
		std::cout << "Help: Command backtrace list (shortcut: btl)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "backtrace list";
		resultat | "" | "";
		resultat | "" | "This command displays the frames in the call stack. The frames are displayed from the more global to the current one.";
		resultat | "" | "The last frame is the current one.";
		resultat | "" | "Each frame is displayed on a single line.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("ba","break",entree1,"address",entree2))
	{
		std::cout << "Help: Command break address (shortcut: ba)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "break address <integer>";
		resultat | "" | "";
		resultat | "" | "This command converts the integer into a program address, and adds a breakpoint on this address.";
		resultat | "" | "";
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("bf","break",entree1,"file",entree2))
	{
		std::cout << "Help: Command break file (shortcut: bf)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "break file <string> <integer>";
		resultat | "" | "";
		resultat | "" | "This command converts the file provided in the string and the line in this file provided in the integer,";
		resultat | "" | "into a program address, and adds a breakpoint on this address.";
		resultat | "" | "";
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("bs","break",entree1,"symbol",entree2))
	{
		std::cout << "Help: Command break symbol (shortcut: bs)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "break symbol <string>";
		resultat | "" | "";
		resultat | "" | "This command converts the symbol provided in the string into a program address, and adds a breakpoint on this address.";
		resultat | "" | "";
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("c","continue",entree1))
	{
		std::cout << "Help: Command continue (shortcut: c)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "continue";
		resultat | "" | "";
		resultat | "" | "This command runs the program until the end of the current frame, or a breakpoint.";
		resultat | "" | "";
		aide_description_execution(resultat);
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_execution(resultat);
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("da","delete",entree1,"address",entree2))
	{
		std::cout << "Help: Command delete address (shortcut: da)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "delete address <address>";
		resultat | "" | "";
		resultat | "" | "This command deletes the memory address given in its argument.";
		resultat | "" | "";
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_memoire(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dab","delete",entree1,"all",entree2,"break",entree3))
	{
		std::cout << "Help: Command delete all break (shortcut: dab)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "delete all break";
		resultat | "" | "";
		resultat | "" | "This command deletes all defined breakpoints.";
		resultat | "" | "";
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("daw","delete",entree1,"all",entree2,"watch",entree3))
	{
		std::cout << "Help: Command delete all watch (shortcut: daw)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "delete all watch";
		resultat | "" | "";
		resultat | "" | "This command deletes all defined watched addresses.";
		resultat | "" | "";
		aide_description_surveillance(resultat);
		resultat | "Related:" | "";
		aide_related_surveillance(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("db","delete",entree1,"break",entree2))
	{
		std::cout << "Help: Command delete break (shortcut: db)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "delete break <number>";
		resultat | "" | "";
		resultat | "" | "This command deletes the breakpoint at the given index in the breakpoint list.";
		resultat | "" | "";
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dv","delete",entree1,"variable",entree2))
	{
		std::cout << "Help: Command delete variable (shortcut: dv)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "delete variable <variable>";
		resultat | "" | "";
		resultat | "" | "This command deletes the memory address linked to the variable.";
		resultat | "" | "";
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_memoire(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dw","delete",entree1,"watch",entree2))
	{
		std::cout << "Help: Command delete watch (shortcut: dw)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "delete watch <number>";
		resultat | "" | "";
		resultat | "" | "This command deletes the watched address at the given index in the watched addresses list.";
		resultat | "" | "";
		aide_description_surveillance(resultat);
		resultat | "Related:" | "";
		aide_related_surveillance(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dvm","dump",entree1,"machine",entree2))
	{
		std::cout << "Help: Command dump machine (shortcut: dvm)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "dump machine";
		resultat | "" | "";
		resultat | "" | "This command displays the internal state of the virtual machine, like the instruction :coredump.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_machine(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dram","dump",entree1,"memory",entree2))
	{
		std::cout << "Help: Command dump memory (shortcut: dram)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "dump memory";
		resultat | "" | "";
		resultat | "" | "This command displays the internal state of the memory, like the instruction :coredump.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_machine(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dcpu","dump",entree1,"processor",entree2))
	{
		std::cout << "Help: Command dump processor (shortcut: dcpu)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "dump processor";
		resultat | "" | "";
		resultat | "" | "This command displays the internal state of the processor, like the instruction :coredump.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_machine(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dprg","dump",entree1,"program",entree2))
	{
		std::cout << "Help: Command dump program (shortcut: dprg)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "dump program";
		resultat | "" | "";
		resultat | "" | "This command displays the internal state of the program, like the instruction :coredump.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_machine(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("dstr","dump",entree1,"stream",entree2))
	{
		std::cout << "Help: Command dump stream (shortcut: dstr)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "dump stream";
		resultat | "" | "";
		resultat | "" | "This command displays the internal state of the streams, like the instruction :coredump.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_machine(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("e","explain",entree1))
	{
		std::cout << "Help: Command explain (shortcut: e)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "explain <address>";
		resultat | "" | "";
		resultat | "" | "This command displays how the given memory address is solved in the current context.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_machine(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("f","frame",entree1))
	{
		std::cout << "Help: Command frame (shortcut: f)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "frame";
		resultat | "" | "";
		resultat | "" | "This command displays the content of the current frame. The result contains:";
		resultat | "" | " * The frame level,";
		resultat | "" | " * The current instruction and its position in the source code if available,";
		resultat | "" | " * The current symbol and its position in the source code if available,";
		resultat | "" | " * The list of the variables defined in the current frame.";
		resultat | "" | "";
		resultat | "Syntax:" | "frame <integer>";
		resultat | "" | "";
		resultat | "" | "This command converts the integer into a program address, and displays what the instruction";
		resultat | "" | "at the given address may do in the current state of the virtual machine.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("g","go",entree1))
	{
		std::cout << "Help: Command go (shortcut: g)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "go";
		resultat | "" | "";
		resultat | "" | "This command quits the debugger. If the program is not terminated, the execution continues without the debugger.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_interface(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("?","help",entree1))
	{
		std::cout << "Help: Command help (shortcut: ?)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "help";
		resultat | "" | "";
		resultat | "" | "This command displays the main menu of the interactive help.";
		resultat | "" | "This menu contains the list of the available commands.";
		resultat | "" | "";
		resultat | "Syntax:" | "help <command without arguments>";
		resultat | "" | "";
		resultat | "" | "This command displays the help page of the given command.";
		resultat | "" | "Each page contains the name, the shortcut that can be used instead of the full command,";
		resultat | "" | "the different syntax of the command with their respective description.";
		resultat | "" | "At the end of the page, some related pages are mentionned.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_interface(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("h","history",entree1))
	{
		std::cout << "Help: Command history (shortcut: h)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "history";
		resultat | "" | "";
		resultat | "" | "This command displays all command lines entered since the startup. The last entry is displayed at the end of the list.";
		resultat | "" | "This command shall be alone on the command line.";
		resultat | "" | "";
		resultat | "Syntax:" | "history <integer>";
		resultat | "" | "";
		resultat | "" | "This command takes the command line in the history at the given index and runs the command line again.";
		resultat | "" | "This command shall be alone on the command line.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_interface(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("i","instruction",entree1))
	{
		std::cout << "Help: Command instruction (shortcut: i)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "instruction";
		resultat | "" | "";
		resultat | "" | "This command runs the next assembly instruction in the program.";
		resultat | "" | "";
		aide_description_execution(resultat);
		resultat | "Related:" | "";
		aide_related_execution(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("int","interruption",entree1))
	{
		std::cout << "Help: Command interruption (shortcut: int)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "interruption <interruption code>";
		resultat | "" | "";
		resultat | "" | "This command enqueues an interruption if a handler for this interruption has been defined in the program.";
		resultat | "" | "The interruption is taken in account on the next execution command.";
		resultat | "" | "Please refer to the virtual machine documentation for the list of available codes.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_machine(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("lb","list",entree1,"break",entree2))
	{
		std::cout << "Help: Command list break (shortcut: lb)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "list break";
		resultat | "" | "";
		resultat | "" | "This command displays the list of the defined breakpoints.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("lf","list",entree1,"file",entree2))
	{
		std::cout << "Help: Command list file (shortcut: lf)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "list file";
		resultat | "" | "";
		resultat | "" | "This command displays the list of source code files, with their valid line numbers.";
		resultat | "" | "";
		resultat | "Syntax:" | "list file <string>";
		resultat | "" | "";
		resultat | "" | "This command displays the list of source code files, with their valid line numbers,";
		resultat | "" | "but only the file names matching the regular expression given in argument are displayed.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("ls","list",entree1,"symbol",entree2))
	{
		std::cout << "Help: Command list symbol (shortcut: ls)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "list symbol";
		resultat | "" | "";
		resultat | "" | "This command displays the list of symbols defined in the source code, with their address in the program.";
		resultat | "" | "";
		resultat | "Syntax:" | "list symbol <string>";
		resultat | "" | "";
		resultat | "" | "This command displays the list of symbols defined in the source code, with their address in the program,";
		resultat | "" | "but only the symbols matching the regular expression given in argument are displayed.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("lv","list",entree1,"variable",entree2))
	{
		std::cout << "Help: Command list variable (shortcut: lv)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "list variable";
		resultat | "" | "";
		resultat | "" | "This command displays the list of variables defined in the source code, with their address in the memory.";
		resultat | "" | "The variable definitions are dependant to the position where the execution is stopped, as the variables have";
		resultat | "" | "a scope delimited by a frame.";
		resultat | "" | "";
		aide_description_memoire(resultat);
		resultat | "Syntax:" | "list variable <string>";
		resultat | "" | "";
		resultat | "" | "This command displays the list of variables defined in the source code, with their address in the memory.";
		resultat | "" | "The variable definitions are dependant to the position where the execution is stopped, as the variables have";
		resultat | "" | "a scope delimited by a frame.";
		resultat | "" | "The variables with names matching the regular expression given in argument are displayed.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("lw","list",entree1,"watch",entree2))
	{
		std::cout << "Help: Command list watch (shortcut: lw)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "list watch";
		resultat | "" | "";
		resultat | "" | "This command displays the list of the defined watched addresses.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_surveillance(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("n","next",entree1))
	{
		std::cout << "Help: Command next (shortcut: n)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "next";
		resultat | "" | "";
		resultat | "" | "This command runs the program until the next instruction in the current frame, or the end of the current frame, or a breakpoint.";
		resultat | "" | "";
		aide_description_execution(resultat);
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_execution(resultat);
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("q","quit",entree1))
	{
		std::cout << "Help: Command quit (shortcut: q)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "quit";
		resultat | "" | "";
		resultat | "" | "This command quits the debugger.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_interface(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("ra","read",entree1,"address",entree2))
	{
		std::cout << "Help: Command read address (shortcut: ra)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "read address <address>";
		resultat | "" | "";
		resultat | "" | "This command reads a value in memory at the given address, and displays the state of the address:";
		resultat | "" | " * Invalid: when the address can not be evaluated,";
		resultat | "" | " * Undefined: when the address has not been allocated,";
		resultat | "" | " * Defined: when the address has been allocated. The value type is mentionned,";
		resultat | "" | " * Initialised: when the address has been allocated and contains a value. The type and the value are mentionned.";
		resultat | "" | "";
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_memoire(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("rv","read",entree1,"variable",entree2))
	{
		std::cout << "Help: Command read variable (shortcut: rv)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "read variable <variable>";
		resultat | "" | "";
		resultat | "" | "This command reads a value in memory at the address linked to the variable, and displays the state of the address:";
		resultat | "" | " * Invalid: when the address can not be evaluated,";
		resultat | "" | " * Undefined: when the address has not been allocated,";
		resultat | "" | " * Defined: when the address has been allocated. The value type is mentionned,";
		resultat | "" | " * Initialised: when the address has been allocated and contains a value. The type and the value are mentionned.";
		resultat | "" | "";
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_memoire(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("r","run",entree1))
	{
		std::cout << "Help: Command run (shortcut: r)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "run";
		resultat | "" | "";
		resultat | "" | "This command runs the program until its end, or a breakpoint.";
		resultat | "" | "";
		aide_description_execution(resultat);
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_execution(resultat);
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("src","source",entree1,"",entree2))
	{
		std::cout << "Help: Command source (shortcut: src)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "source";
		resultat | "" | "";
		resultat | "" | "This command displays the source code around the execution point, in the current frame.";
		resultat | "" | "";
		resultat | "Syntax:" | "source <integer>";
		resultat | "" | "";
		resultat | "" | "This command displays the source code around the execution point, in the frame having the level given in argument.";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("srcf","source",entree1,"file",entree2))
	{
		std::cout << "Help: Command source file (shortcut: srcf)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "source file <filename> <ligne>";
		resultat | "" | "";
		resultat | "" | "This command displays the source code around the given position.";
		resultat | "" | "";
		resultat | "" | "";
		resultat | "Related:" | "";
		aide_related_source(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("s","step",entree1))
	{
		std::cout << "Help: Command step (shortcut: s)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "step";
		resultat | "" | "";
		resultat | "" | "This command runs the program until the next instruction, or a breakpoint.";
		resultat | "" | "";
		aide_description_execution(resultat);
		aide_description_pointsarret(resultat);
		resultat | "Related:" | "";
		aide_related_execution(resultat);
		aide_related_pointsarret(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("wa","watch",entree1,"address",entree2))
	{
		std::cout << "Help: Command watch address (shortcut: wa)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "watch address <address>";
		resultat | "" | "";
		resultat | "" | "This command adds a watched address to the given address.";
		resultat | "" | "";
		aide_description_surveillance(resultat);
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_surveillance(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("wv","watch",entree1,"variable",entree2))
	{
		std::cout << "Help: Command watch variable (shortcut: wv)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "watch address <variable>";
		resultat | "" | "";
		resultat | "" | "This command adds a watched address to the address linked to the given variable.";
		resultat | "" | "The state of a watched variable can be Out of scope when the execution is outside the frame";
		resultat | "" | "where it is defined.";
		resultat | "" | "";
		aide_description_surveillance(resultat);
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_surveillance(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("wra","write",entree1,"address",entree2))
	{
		std::cout << "Help: Command write address (shortcut: wra)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "write address <address> <value or address>";
		resultat | "" | "";
		resultat | "" | "This command writes the value or the address into the memory at the given address.";
		resultat | "" | "";
		resultat | "Syntax:" | "write address <address>";
		resultat | "" | "";
		resultat | "" | "This command clears the memory at the given address. The address is still allocated,";
		resultat | "" | "but the value is removed.";
		resultat | "" | "";
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_memoire(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("wrv","write",entree1,"variable",entree2))
	{
		std::cout << "Help: Command write variable (shortcut: wrv)" << std::endl;
		ResultatAideFeuille resultat;
		resultat | "Syntax:" | "write variable <variable> <value or address>";
		resultat | "" | "";
		resultat | "" | "This command writes the value or the address into the memory at the address linked to the variable.";
		resultat | "" | "";
		resultat | "Syntax:" | "write variable <variable>";
		resultat | "" | "";
		resultat | "" | "This command clears the memory at the address linked to the variable. The address is still allocated,";
		resultat | "" | "but the value is removed.";
		resultat | "" | "";
		aide_description_memoire(resultat);
		resultat | "Related:" | "";
		aide_related_memoire(resultat);
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","delete",entree1,"all",entree2))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Breakpoints:" | "" | "" ;
		resultat | "" | "delete all break" | "clear all breakpoints" ;
		resultat | "Watched addresses:" | "" | "" ;
		resultat | "" | "delete all watch" | "clear all watched expressions" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","backtrace",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Running:" | "" | "" ;
		resultat | "" | "backtrace" | "display the call stack" ;
		resultat | "" | "backtrace list" | "display a summary of the call stack" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","break",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Breakpoints:" | "" | "" ;
		resultat | "" | "break address" | "add a breakpoint on an assembly address" ;
		resultat | "" | "break symbol" | "add a breakpoint on a defined symbol" ;
		resultat | "" | "break file" | "add a breakpoint on a source line" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","delete",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Breakpoints:" | "" | "" ;
		resultat | "" | "delete break" | "delete a breakpoint" ;
		resultat | "" | "delete all break" | "clear all breakpoints" ;
		resultat | "Watched addresses:" | "" | "" ;
		resultat | "" | "delete watch" | "delete a watched expression" ;
		resultat | "" | "delete all watch" | "clear all watched expressions" ;
		resultat | "Memory handling:" | "" | "" ;
		resultat | "" | "delete address" | "free a value in memory at a given address" ;
		resultat | "" | "delete variable" | "free a value in memory at the address linked to a variable" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","dump",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Machine state:" | "" | "" ;
		resultat | "" | "dump machine" | "display the internal state of the virtual machine" ;
		resultat | "" | "dump processor" | "display the internal state of the processor" ;
		resultat | "" | "dump memory" | "display the internal state of the memory" ;
		resultat | "" | "dump program" | "display the internal state of the program" ;
		resultat | "" | "dump stream" | "display the internal state of the stream" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","list",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Source files:" | "" | "" ;
		resultat | "" | "list file" | "display source files and lines" ;
		resultat | "" | "list symbol" | "display defined symbols" ;
		resultat | "" | "list variable" | "display defined variables" ;
		resultat | "Breakpoints:" | "" | "" ;
		resultat | "" | "list break" | "list all breakpoints" ;
		resultat | "Watched addresses:" | "" | "" ;
		resultat | "" | "list watch" | "list all watched expressions" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","source",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Source files:" | "" | "" ;
		resultat | "" | "source" | "display source file at execution position" ;
		resultat | "" | "source file" | "display arbitrary source file position" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","watch",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Watched addresses:" | "" | "" ;
		resultat | "" | "watch address" | "add a watched expression on a memory address" ;
		resultat | "" | "watch variable" | "add a watched expression on a defined variable" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","write",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Memory handling:" | "" | "" ;
		resultat | "" | "write address" | "modify a value in memory at a given address" ;
		resultat | "" | "write variable" | "modify a value in memory at the address linked to a variable" ;
		std::cout << resultat;
	}
	else if(Debogueur::aide_rubrique("","read",entree1))
	{
		std::cout << "Help menu: enter help followed by a command for more details" << std::endl;
		ResultatAideNoeud resultat;
		resultat & "Theme" & "Command" & "Description";
		resultat | "Memory handling:" | "" | "" ;
		resultat | "" | "read address" | "read a value in memory at a given address" ;
		resultat | "" | "read variable" | "read a value in memory at the address linked to a variable" ;
		std::cout << resultat;
	}
	else
	{
		std::cout << "Help menu:" << std::endl;
		Resultat<
			SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::GAUCHE,20>,
			SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::DROITE,30>,
			SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::GAUCHE,5>,
			SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::GAUCHE,85>
				> resultat;
		resultat & "Theme"              & "Command"          & "Short" & "Description";
		resultat | "Interface:"         | ""                 | ""      | "" ;
		resultat | ""                   | "help"             | "?"     | "interactive help" ;
		resultat | ""                   | "history"          | "h"     | "display and run again commands" ;
		resultat | ""                   | "quit"             | "q"     | "exit the debugger" ;
		resultat | ""                   | "go"               | "g"     | "continue without debugger";
		resultat | "Source files:"      | ""                 | ""      | "" ;
		resultat | ""                   | "list file"        | "lf"    | "display source files and lines" ;
		resultat | ""                   | "list symbol"      | "ls"    | "display defined symbols" ;
		resultat | ""                   | "list variable"    | "lv"    | "display defined variables" ;
		resultat | ""                   | "source"           | "src"   | "display source file at execution position" ;
		resultat | ""                   | "source file"      | "srcf"  | "display arbitrary source file position" ;
		resultat | "Running:"           | ""                 | ""      | "" ;
		resultat | ""                   | "backtrace"        | "bt"    | "display the call stack" ;
		resultat | ""                   | "backtrace list"   | "btl"   | "display a summary of the call stack" ;
		resultat | ""                   | "frame"            | "f"     | "display a frame level" ;
		resultat | ""                   | "run"              | "r"     | "execute the program until its end or a breakpoint" ;
		resultat | ""                   | "continue"         | "c"     | "execute the program until the end of the current frame or a breakpoint" ;
		resultat | ""                   | "next"             | "n"     | "execute the program until the next instruction on the current frame or a breakpoint" ;
		resultat | ""                   | "step"             | "s"     | "execute the program until the next instruction or a breakpoint" ;
		resultat | ""                   | "instruction"      | "i"     | "execute one assembly instruction" ;
		resultat | "Breakpoints:"       | ""                 | ""      | "" ;
		resultat | ""                   | "break address"    | "ba"    | "add a breakpoint on an assembly address" ;
		resultat | ""                   | "break symbol"     | "bs"    | "add a breakpoint on a defined symbol" ;
		resultat | ""                   | "break file"       | "bf"    | "add a breakpoint on a source line" ;
		resultat | ""                   | "list break"       | "lb"    | "list all breakpoints" ;
		resultat | ""                   | "delete break"     | "db"    | "delete a breakpoint" ;
		resultat | ""                   | "delete all break" | "dab"   | "clear all breakpoints" ;
		resultat | "Watched addresses:" | ""                 | ""      | "" ;
		resultat | ""                   | "watch address"    | "wa"    | "add a watched expression on a memory address" ;
		resultat | ""                   | "watch variable"   | "wv"    | "add a watched expression on a defined variable" ;
		resultat | ""                   | "list watch"       | "lw"    | "list all watched expressions" ;
		resultat | ""                   | "delete watch"     | "dw"    | "delete a watched expression" ;
		resultat | ""                   | "delete all watch" | "daw"   | "clear all watched expressions" ;
		resultat | "Machine state:"     | ""                 | ""      | "" ;
		resultat | ""                   | "dump machine"     | "dvm"   | "display the internal state of the virtual machine" ;
		resultat | ""                   | "dump processor"   | "dcpu"  | "display the internal state of the processor" ;
		resultat | ""                   | "dump memory"      | "dram"  | "display the internal state of the memory" ;
		resultat | ""                   | "dump program"     | "dprg"  | "display the internal state of the program" ;
		resultat | ""                   | "dump stream"      | "dstr"  | "display the internal state of the stream" ;
		resultat | ""                   | "interruption"     | "int"   | "add an interruption" ;
		resultat | ""                   | "explain"          | "e"     | "explain how an address is solved" ;
		resultat | "Memory handling:"   | ""                 | ""      | "" ;
		resultat | ""                   | "read address"     | "ra"    | "read a value in memory at a given address" ;
		resultat | ""                   | "read variable"    | "rv"    | "read a value in memory at the address linked to a variable" ;
		resultat | ""                   | "write address"    | "wra"   | "modify a value in memory at a given address" ;
		resultat | ""                   | "write variable"   | "wrv"   | "modify a value in memory at the address linked to a variable" ;
		resultat | ""                   | "delete address"   | "da"    | "free a value in memory at a given address" ;
		resultat | ""                   | "delete variable"  | "dv"    | "free a value in memory at the address linked to a variable" ;
		std::cout << resultat;
		std::cout << "Help on each command is available using 'help' followed by the command or its shortcut." << std::endl
			<< std::endl
			<< "Several commands (except history) can be put in a single command line, separated by semicolons." << std::endl
			<< "When no command is entered, the previous command line is repeated." << std::endl
			<< std::endl
			<< "The program can be run only once in the debugger. To run it again, you should restart the virtual machine." << std::endl
			<< "When the go command is used, the program continues outside the debugger." << std::endl;
	} 
}
