/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_DEBOGUEUR_DONNEES_DEBOGUEUR_H_
#define _MACHINE_DEBOGUEUR_DONNEES_DEBOGUEUR_H_

#include <vector>
#include <map>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Debogueur { DECL_SHARED_PTR(Debogueur); } } }

#include <src/machine/machine/machine.h>
#include <src/machine/debogueur/donnees/pointarret.h>
#include <src/machine/debogueur/donnees/surveillance.h>
#include <src/machine/debogueur/donnees/trame.h>
#include <src/machine/debogueur/donnees/directive.h>
#include <src/machine/debogueur/donnees/sonde.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			DECL_SHARED_PTR(Condition);
		}
		namespace Debogueur
		{
			class Debogueur : public std::enable_shared_from_this<Debogueur>
			{
				public:
					enum class TypeAction {DEBOGUAGE, CONTINUE, SORTIE};
					Debogueur()
					:_execution_en_cours(true), _sortie(TypeAction::DEBOGUAGE), _nb_instructions(0) {};
					explicit Debogueur(const std::string& commandes_initiales)
					:_execution_en_cours(true), _sortie(TypeAction::DEBOGUAGE), _prochaine_commande(commandes_initiales), _nb_instructions(0) {};
					Debogueur(const Debogueur&) = delete;
					static bool execution(SetLgg::Machine::Machine::MachineSP machine);
					void directive(const std::string& directive, const SetLgg::Machine::Programme::ProgrammeSP& programme, const SetLgg::Machine::Programme::ConditionSP& condition, const SetLgg::Global::Source::PositionSP& position);
					void directive_debut_trame(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition);
					void directive_fin_trame(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition);
					void directive_instruction(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& code, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition);
					void directive_symbole(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& symbole, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition);
					void directive_variable(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& variable, const SetLgg::Machine::Programme::AdresseMemoireSP& adresse_memoire,  const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition);
					void commande_affiche_fichier(const std::string& regex = std::string("^.*$"));
					void commande_affiche_historique();
					void commande_affiche_pileappels();
					void commande_affiche_pileappels_liste();
					void commande_affiche_pointarret();
					void commande_affiche_source();
					void commande_affiche_source(const size_t niveau);
					void commande_affiche_source(const std::string& fichier, const size_t ligne);
					void commande_affiche_surveillance();
					void commande_affiche_symbole(const std::string& regex = std::string("^.*$"));
					void commande_affiche_trame();
					void commande_affiche_trame(const size_t niveau);
					void commande_affiche_variable(const std::string& regex = std::string("^.*$"));
					void commande_ajoute_interruption(const size_t code);
					void commande_ajoute_pointdarret(const size_t adresse);
					void commande_ajoute_pointdarret(const std::string& adresse);
					void commande_ajoute_pointdarret(const std::string& fichier, const size_t ligne);
					void commande_ajoute_surveillance(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse);
					void commande_ajoute_surveillance(const std::string& variable);
					void commande_ecriture_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse, const SetLgg::Machine::Programme::ValeurSP& valeur=SetLgg::Machine::Programme::ValeurSP());
					void commande_ecriture_memoire(const std::string& variable, const SetLgg::Machine::Programme::ValeurSP& valeur=SetLgg::Machine::Programme::ValeurSP());
					void commande_ecriture_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse, const SetLgg::Machine::Programme::AdresseMemoireSP& valeur);
					void commande_ecriture_memoire(const std::string& variable, const SetLgg::Machine::Programme::AdresseMemoireSP& valeur);
					void commande_execute_continue();
					void commande_execute_execution();
					void commande_execute_instruction();
					void commande_execute_pasapas();
					void commande_execute_suivant();
					void commande_explique_adresse(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse);
					void commande_lecture_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse);
					void commande_lecture_memoire(const std::string& variable);
					void commande_liberation_memoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse);
					void commande_liberation_memoire(const std::string& variable);
					void commande_metaplat_flux();
					void commande_metaplat_machine();
					void commande_metaplat_memoire();
					void commande_metaplat_processeur();
					void commande_metaplat_programme();
					void commande_rejoue_historique(const size_t commande);
					void commande_sortie(const TypeAction action);
					void commande_supprime_pointdarret(const size_t numero);
					void commande_supprime_pointdarret_tous();
					void commande_supprime_surveillance(const size_t numero);
					void commande_supprime_surveillance_tous();
					void commande_aide(const std::string& entree1 = std::string(), const std::string& entree2 = std::string(), const std::string& entree3 = std::string());
					bool execution_pointdarret();
					void affiche_prompt();
					void ajoute_commande_historique(const std::string& commande);
					bool enleve_commande_historique();
					template<typename oStream>
					friend oStream& operator<<(oStream&,const Debogueur&);
					template<typename... Delimiteurs>
					static std::string desechappe(const char& caractere)
					{
						std::string resultat("\\");
						resultat += caractere;
						return resultat;
					}
					template<typename Delimiteur, typename... Delimiteurs>
					static std::string desechappe(const char& caractere, const Delimiteur delimiteur, const Delimiteurs... delimiteurs)
					{
						if(caractere==delimiteur)
						{
							switch(delimiteur)
							{
								case 't':
									return std::string(1,'\t');
									break;
								case 'n':
									return std::string(1,'\n');
									break;
								case 'r':
									return std::string(1,'\r');
									break;
								default:
									return std::string(1,caractere);
									break;
							}
						}
						return Debogueur::desechappe(caractere,delimiteurs...);
					}
					template<typename... Delimiteurs>
					static std::string desechappe(const std::string& texte, Delimiteurs... delimiteurs)
					{
						std::ostringstream resultat;
						bool echappe = false;
						for(auto it = texte.cbegin() ; it!=texte.cend() ; ++it)
						{
							if(echappe)
							{
								resultat << Debogueur::desechappe(*it,delimiteurs...);
								echappe=false;
							}
							else
							{
								if((*it)=='\\')
									echappe=true;
								else
									resultat << (*it);
							}
						}
						return resultat.str();
					};
					void ajout_debogueur(const SetLgg::Machine::Debogueur::DebogueurSP& debogueur, const SetLgg::Machine::Programme::Adresse& adresse_terminale);
				private:
					void ajoute_directive(const DirectiveSP& directive);
					SetLgg::Machine::Programme::AdresseSP valide_fichier_ligne(const std::string& fichier, const size_t ligne);
					void affiche_source(const Trame& trame);
					void affiche_source(const std::string& fichier, const size_t ligne);
					void affiche_surveillances(SetLgg::Machine::Machine::MachineSP& machine);
					void affiche_instruction_source() const;
					void affiche_instruction_assembleur(SetLgg::Machine::Machine::MachineSP& machine) const;
					void execute_directives(SetLgg::Machine::Machine::MachineSP& machine);
					void execute_programme_debug(SetLgg::Machine::Machine::MachineSP& machine);
					std::vector<ListeDirectives> _directives;
					std::map<std::string, std::map<size_t, std::string> > _sources;
					std::map<std::string, std::map<size_t, SetLgg::Machine::Programme::AdresseSP> > _lignes;
					std::map<std::string, SetLgg::Machine::Programme::AdresseSP> _symboles;
					ListePointsArret _liste_pointsarret;
					std::vector<std::string> _historique;
					PileTrames _pile_trames;
					ListeSurveillances _surveillances;
					SondeSP _sonde;
					SetLgg::Machine::Machine::MachineWP _machine;
					bool _execution_en_cours;
					TypeAction _sortie;
					std::string _prochaine_commande;
					unsigned long long int _nb_instructions;
					typedef Resultat<
						SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::GAUCHE,20>,
						SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::DROITE,30>,
						SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::GAUCHE,80>
							> ResultatAideNoeud;
					typedef Resultat<
						SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::GAUCHE,20>,
						SetLgg::Machine::Debogueur::Colonne<SetLgg::Machine::Debogueur::Alignement::GAUCHE>
							> ResultatAideFeuille;
					static void aide_related_interface(ResultatAideFeuille& resultat);
					static void aide_related_source(ResultatAideFeuille& resultat);
					static void aide_related_machine(ResultatAideFeuille& resultat);
					static void aide_description_execution(ResultatAideFeuille& resultat);
					static void aide_related_execution(ResultatAideFeuille& resultat);
					static void aide_description_pointsarret(ResultatAideFeuille& resultat);
					static void aide_related_pointsarret(ResultatAideFeuille& resultat);
					static void aide_description_surveillance(ResultatAideFeuille& resultat);
					static void aide_related_surveillance(ResultatAideFeuille& resultat);
					static void aide_description_memoire(ResultatAideFeuille& resultat);
					static void aide_related_memoire(ResultatAideFeuille& resultat);
					static bool aide_rubrique(const std::string& raccourci) { return true; };
					template<typename Identifiant, typename Entree, typename... Identifiants>
					static bool aide_rubrique(const std::string& raccourci, const Identifiant& identifiant, const Entree& entree, Identifiants... identifiants)
					{
						if((not raccourci.empty()) and (entree==raccourci))
							return true;
						if(entree!=identifiant)
							return false;
						return aide_rubrique("",identifiants...);
					}
			};

			template<typename oStream>
			oStream& operator<<(oStream& os, const Debogueur& debogueur)
			{
				return os;
			}
		}
	}
}

#endif
