/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_DEBOGUEUR_DONNEES_DIRECTIVE_H_
#define _MACHINE_DEBOGUEUR_DONNEES_DIRECTIVE_H_

#include <vector>
#include <map>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Debogueur { DECL_SHARED_PTR(Directive); DECL_SHARED_PTR(DirectiveInstruction); DECL_SHARED_PTR(DirectiveSymbole); DECL_SHARED_PTR(DirectiveVariable); DECL_SHARED_PTR(DirectiveDebutTrame); DECL_SHARED_PTR(DirectiveFinTrame); } } }

#include <src/machine/programme/donnees/adresse.h>
#include <src/machine/memoire/donnees/adresse.h>
#include <src/machine/programme/donnees/adressememoire.h>
#include <src/machine/programme/donnees/condition.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Debogueur
		{
			DECL_SHARED_PTR(Variable);
			class Debogueur;
			class PileTrames;
			class Trame;
			class ListeDirectives;

			class Directive
			{
				friend class Debogueur;
				friend class DirectiveVariable;
				friend class PileTrames;
				friend class Trame;
				friend class ListeDirectives;
				public:
					Directive(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
					:_adresse(adresse), _position(position), _condition(condition) {};
					virtual ~Directive() {};
					virtual bool instruction() { return false; };
					virtual bool fin_trame() { return false; };
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const Directive& directive)
					{
						directive.format(os);
						os << "          (" << directive._position->fichier() << " " << directive._position->ligne() << ")";
						return os;
					}
					virtual void format(std::ostream& os) const {};
					virtual void ajout_debogueur(Debogueur* debogueur, const SetLgg::Machine::Programme::Adresse& adresse) const = 0;
				protected:
					SetLgg::Machine::Programme::AdresseSP _adresse;
					SetLgg::Global::Source::PositionSP _position;
					SetLgg::Machine::Programme::ConditionSP _condition;
			};

			class DirectiveDebutTrame : public Directive
			{
				friend class Trame;
				public:
					DirectiveDebutTrame(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
					:Directive(adresse,position,condition) {};
					virtual ~DirectiveDebutTrame() {};
					virtual void ajout_debogueur(Debogueur* debogueur, const SetLgg::Machine::Programme::Adresse& adresse) const;
				private:
			};

			class DirectiveFinTrame : public Directive
			{
				friend class Trame;
				public:
					DirectiveFinTrame(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const SetLgg::Machine::Programme::ConditionSP& condition)
					:Directive(adresse,position,condition) {};
					virtual ~DirectiveFinTrame() {};
					virtual bool fin_trame() { return true; };
					virtual void ajout_debogueur(Debogueur* debogueur, const SetLgg::Machine::Programme::Adresse& adresse) const;
				private:
			};

			class DirectiveInstruction : public Directive
			{
				friend class Trame;
				public:
					DirectiveInstruction(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const std::string& code, const SetLgg::Machine::Programme::ConditionSP& condition)
					:Directive(adresse,position,condition), _code(code) {};
					virtual ~DirectiveInstruction() {};
					virtual bool instruction() { return true; };
					virtual void format(std::ostream& os) const
					{
						os << _code;
					};
					virtual void ajout_debogueur(Debogueur* debogueur, const SetLgg::Machine::Programme::Adresse& adresse) const;
				private:
					std::string _code;
			};

			class DirectiveSymbole : public Directive
			{
				friend class Trame;
				public:
					DirectiveSymbole(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const std::string& symbole, const SetLgg::Machine::Programme::ConditionSP& condition)
					:Directive(adresse,position,condition), _symbole(symbole) {};
					virtual ~DirectiveSymbole() {};
					virtual void format(std::ostream& os) const
					{
						os << _symbole;
					};
					virtual void ajout_debogueur(Debogueur* debogueur, const SetLgg::Machine::Programme::Adresse& adresse) const;
				private:
					std::string _symbole;
			};

			class DirectiveVariable : public Directive
			{
				friend class Trame;
				public:
					DirectiveVariable(const SetLgg::Machine::Programme::AdresseSP& adresse, const SetLgg::Global::Source::PositionSP& position, const std::string& nom, const SetLgg::Machine::Programme::AdresseMemoireSP& adresse_memoire, const SetLgg::Machine::Programme::ConditionSP& condition)
					:Directive(adresse,position,condition), _nom(nom), _adresse_memoire(adresse_memoire) {};
					virtual ~DirectiveVariable() {};
					VariableSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const size_t niveau);
					virtual void format(std::ostream& os) const
					{
						os << _nom << ": " << *_adresse_memoire;
					};
					virtual void ajout_debogueur(Debogueur* debogueur, const SetLgg::Machine::Programme::Adresse& adresse) const;
				private:
					std::string _nom;
					SetLgg::Machine::Programme::AdresseMemoireSP _adresse_memoire;
			};

			struct SondeContinuation;
			struct SondeSuivant;
			struct SondePasAPas;
			class ListeDirectives
			{
				friend struct SondeContinuation;
				friend struct SondeSuivant;
				friend struct SondePasAPas;
				friend class Debogueur;
				friend class Trame;
				friend class PileTrames;
				public:
					ListeDirectives()
					:_instruction(false), _fin_trame(false) {};
					void ajout(const DirectiveSP& directive)
					{
						_directives.push_back(directive);
					};
					void raz()
					{
						_instruction=false;
						_fin_trame=false;
					}
					const std::vector<DirectiveSP>::const_iterator cbegin() const
					{
						return _directives.cbegin();
					};
					const std::vector<DirectiveSP>::const_iterator cend() const
					{
						return _directives.cend();
					};
				private:
					std::vector<DirectiveSP> _directives;
					bool _instruction;
					bool _fin_trame;
			};

		}
	}
}

#endif
