/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_DEBOGUEUR_DONNEES_POINTDARRET_H_
#define _MACHINE_DEBOGUEUR_DONNEES_POINTDARRET_H_

#include <vector>
#include <map>
#include <set>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Debogueur { DECL_SHARED_PTR(PointArret); } } }

#include <src/machine/debogueur/donnees/resultat.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Debogueur
		{
			class Debogueur;
			class ListePointsArret;

			class PointArret
			{
				friend class Debogueur;
				friend class ListePointsArret;
				public:
					PointArret(const SetLgg::Machine::Programme::AdresseSP& adresse)
					:_adresse(adresse) {};
					virtual ~PointArret() {};
					std::string adresse() const
					{
						std::stringstream oss;
						if(_adresse)
							oss << *_adresse;
						return oss.str();
					};
					virtual std::string type() const = 0;
					virtual std::string details() const = 0;
					virtual void rafraichit_adresses(const std::map<std::string, SetLgg::Machine::Programme::AdresseSP>& symboles, const std::map<std::string, std::map<size_t, SetLgg::Machine::Programme::AdresseSP> >& lignes) {};
				protected:
					SetLgg::Machine::Programme::AdresseSP _adresse;
			};

			class PointArretAdresse : public PointArret
			{
				public:
					PointArretAdresse(const SetLgg::Machine::Programme::AdresseSP& adresse)
					:PointArret(adresse) {};
					virtual ~PointArretAdresse() {};
					virtual std::string type() const
					{
						return "Address";
					};
					virtual std::string details() const
					{
						return "";
					};
				private:
			};

			class PointArretSymbole : public PointArret
			{
				public:
					PointArretSymbole(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& symbole)
					:PointArret(adresse), _symbole(symbole) {};
					virtual ~PointArretSymbole() {};
					virtual std::string type() const
					{
						return "Symbol";
					};
					virtual std::string details() const
					{
						return _symbole;
					};
					virtual void rafraichit_adresses(const std::map<std::string, SetLgg::Machine::Programme::AdresseSP>& symboles, const std::map<std::string, std::map<size_t, SetLgg::Machine::Programme::AdresseSP> >& lignes)
					{
						auto it = symboles.find(_symbole);
						if(it==symboles.end())
							return;
						_adresse = it->second;
					};
				private:
					std::string _symbole;
			};

			class PointArretFichier : public PointArret
			{
				public:
					PointArretFichier(const SetLgg::Machine::Programme::AdresseSP& adresse, const std::string& fichier, const size_t ligne)
					:PointArret(adresse), _fichier(fichier), _ligne(ligne) {};
					virtual ~PointArretFichier() {};
					virtual std::string type() const
					{
						return "File";
					};
					virtual std::string details() const
					{
						std::stringstream oss;
						oss << _fichier << " " << _ligne;
						return oss.str();
					};
					virtual void rafraichit_adresses(const std::map<std::string, SetLgg::Machine::Programme::AdresseSP>& symboles, const std::map<std::string, std::map<size_t, SetLgg::Machine::Programme::AdresseSP> >& lignes)
					{
						auto itf = lignes.find(_fichier);
						if(itf==lignes.end())
							return;
						auto itl = itf->second.find(_ligne);
						if(itl==itf->second.end())
							return;
						_adresse=itl->second;
					};
				private:
					std::string _fichier;
					size_t _ligne;
			};

			class ListePointsArret
			{
				public:
					ListePointsArret() = default;
					~ListePointsArret() {};
					void ajout(PointArretSP pointarret)
					{
						_pointsarret.push_back(pointarret);
						if(pointarret->_adresse)
							_adresses.insert(*(pointarret->_adresse));
					};
					bool enleve(size_t numero)
					{
						if(numero>=_pointsarret.size())
						{
							std::cerr << "Breakpoint number " << numero << " does not exist." << std::endl;
							return false;
						}
						_pointsarret.erase(_pointsarret.begin()+numero);
						rafraichit_adresses();
						return true;
					};
					void enleve_tout()
					{
						_pointsarret.clear();
						_adresses.clear();
					};
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const ListePointsArret& liste)
					{
						if(liste._pointsarret.empty())
						{
							os << "No breakpoints defined." << std::endl;
							return os;
						}
						else
						{
							os << "Defined breakpoints:" << std::endl;
						}
						Resultat<NumerosCadre, ColonneCadre<Alignement::GAUCHE,10>, ColonneCadre<Alignement::DROITE,20>, ColonneCadre<Alignement::GAUCHE,100> > resultat;
						resultat & "" & "Type" & "Address" & "Details" ;
						for(auto it = liste._pointsarret.cbegin() ; it!=liste._pointsarret.cend() ; ++it)
						{
							resultat | "indice" | (*it)->type() | (*it)->adresse() | (*it)->details() ;
						}
						os << resultat;
						return os;
					};
					template<typename oStream>
					void pointsarret_atteints(oStream& os, const SetLgg::Machine::Programme::Adresse& adresse)
					{
						size_t indice=0;
						bool match=false;
						Resultat<Colonne<Alignement::DROITE,10>, Colonne<Alignement::GAUCHE,10>, Colonne<Alignement::DROITE,20>, Colonne<Alignement::GAUCHE,100> > resultat;
						for(auto it=_pointsarret.cbegin() ; it!=_pointsarret.cend() ; ++it,++indice)
						{
							if(((*it)->_adresse) and (adresse==*((*it)->_adresse)))
							{
								match=true;
								std::stringstream oss;
								oss << indice;
								resultat | oss.str() | (*it)->type() | (*it)->adresse() | (*it)->details() ;
							}
						}
						if(match)
						{
							os << "Breakpoint(s) reached:" << std::endl;
							os << resultat;
						}
					}
					bool operator==(const SetLgg::Machine::Programme::Adresse& adresse)
					{
						return (_adresses.find(adresse)!=_adresses.end());
					};
					void operator() (const std::map<std::string, SetLgg::Machine::Programme::AdresseSP>& symboles, const std::map<std::string, std::map<size_t, SetLgg::Machine::Programme::AdresseSP> >& lignes)
					{
						for(auto it=_pointsarret.cbegin() ; it!=_pointsarret.cend() ; ++it)
						{
							(*it)->rafraichit_adresses(symboles,lignes);
						}
						rafraichit_adresses();
					};
				private:
					std::vector<PointArretSP> _pointsarret;
					std::set<SetLgg::Machine::Programme::Adresse> _adresses;
					void rafraichit_adresses()
					{
						_adresses.clear();
						for(auto it=_pointsarret.cbegin() ; it!=_pointsarret.cend() ; ++it)
						{
							if((*it)->_adresse)
								_adresses.insert(*((*it)->_adresse));
						}
					};
			};
		}
	}
}

#endif
