/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_DEBOGUEUR_DONNEES_RESULTAT_H_
#define _MACHINE_DEBOGUEUR_DONNEES_RESULTAT_H_

#include <vector>
#include <map>
#include <set>
#include <sstream>
#include <iomanip>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Debogueur { } } }

#include <src/machine/programme/donnees/adresse.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Debogueur
		{

			enum class Alignement { GAUCHE, DROITE };

			template<Alignement alignement, size_t taille_max=0, char separateur = ' ', char ligne = ' ', char croisement = ' ' >
			struct Colonne
			{
				Colonne()
				:_taille(0),_entete(false) {};
				void ajout(const std::string& texte)
				{
					_valeurs.push_back(texte);
					if(texte.size()>_taille)
					{
						_taille = texte.size();
						if((taille_max>0) and (_taille>taille_max))
						{
							_taille = taille_max;
						}
					}
				};
				size_t _taille;
				std::string format(const size_t indice) const
				{
					std::string format;
					bool depasse=false;
					if(indice<_valeurs.size())
					{
						if(_valeurs[indice].size()>_taille)
						{
							if(alignement==Alignement::GAUCHE)
							{
								format=_valeurs[indice].substr(0,_taille);
							}
							else
							{
								format=_valeurs[indice].substr(_valeurs[indice].size()-_taille,_taille);
							}
							depasse=true;
						}
						else
						{
							std::string blanc(_taille-_valeurs[indice].size(),' ');
							if(alignement==Alignement::GAUCHE)
							{
								format = _valeurs[indice]+blanc;
							}
							else
							{
								format = blanc+_valeurs[indice];
							}
						}
					}
					else
					{
						format = std::string(_taille,' ');
					}
					std::ostringstream formatss;
					formatss << ((depasse and alignement==Alignement::DROITE)?'<':' ')
						<< format
						<< ((depasse and alignement==Alignement::GAUCHE)?'>':' ')
						<< separateur;
					return formatss.str();
				};
				std::string separation() const
				{
					std::string uneligne(_taille+2,ligne);
					uneligne += croisement;
					return uneligne;
				}

				void titre(const std::string& titre)
				{
					_entete = true;
					_titre = titre;
					_taille = titre.size();
					if((taille_max>0) and (_taille>taille_max))
					{
						_taille = taille_max;
					}
				}

				std::string titre() const
				{
					if(not _entete)
						return "";
					std::string titre(" ");
					titre+=_titre.substr(0,_taille);
					std::string blanc(_taille-_titre.size(),' ');
					titre+=blanc;
					titre+=' ';
					titre+=separateur;
					return titre;
				}
				size_t taille() const
				{
					return _valeurs.size();
				};
				private:
					std::string _titre;
					std::vector<std::string> _valeurs;
					bool _entete;
			};

			struct Incremente
			{
				void operator() (std::vector<size_t>& vecteur, const std::string& texte)
				{
					vecteur.push_back(vecteur.size());
				};
			};

			struct Decremente
			{
				void operator() (std::vector<size_t>& vecteur, const std::string& texte)
				{
					vecteur.insert(vecteur.begin(),vecteur.size());
				};
			};
			template<typename Ajout = Incremente, char separateur = ' ', char ligne = ' ', char croisement=' '>
			struct Numeros
			{
				Numeros()
				:_entete(false) {};
				void ajout(const std::string& texte)
				{
					Ajout ajout;
					ajout(_valeurs,texte);
				}
				std::string format(const size_t indice) const
				{
					std::string format;
					if(indice<_valeurs.size())
					{
						std::ostringstream oss;
						oss << std::setw(5) << _valeurs[indice];
						std::string valeur = oss.str();
						std::string blanc(5-valeur.size(),' ');
						format = blanc+valeur;
					}
					else
					{
						format = std::string(5,' ');
					}
					format += separateur;
					return format;
				};
				std::string separation() const
				{
					std::string uneligne(5,ligne);
					uneligne += croisement;
					return uneligne;
				}

				std::string titre() const
				{
					if(not _entete)
						return "";
					return std::string("Index")+separateur;
				}

				void titre(const std::string& titre)
				{
					_entete=true;
				}
				size_t taille() const
				{
					return _valeurs.size();
				};
				protected:
					std::vector<size_t> _valeurs;
					bool _entete;
			};

			template<Alignement alignement, size_t taille_max = 0>
			struct ColonneCadre : public Colonne<alignement,taille_max,'|','-','+'>
			{
			};
			
			struct NumerosCadre : public Numeros<Incremente,'|','-','+'>
			{
			};

			template<char separateur = ' ', char ligne = ' ', char croisement=' '>
			struct NumerosInverses : public Numeros<Decremente,separateur,ligne,croisement>
			{
			};

			struct NumerosInversesCadre : public NumerosInverses<'|','-','+'>
			{
			};

			template<typename... Colonnes>
			class Resultat
			{
				public:
					std::string separation() const
					{
						return "";
					}
					std::string titre() const
					{
						return "";
					}
					std::string format(const size_t indice) const
					{
						return "";
					}
			};

			template<typename Colonne, typename... ColonnesSuivantes>
			class Resultat<Colonne,ColonnesSuivantes...>
			{
				public:
					Resultat<Colonne,ColonnesSuivantes...>() {};
					friend Resultat<ColonnesSuivantes...>& operator&(Resultat<Colonne,ColonnesSuivantes...>& resultat, const std::string& texte) 
					{
						resultat._colonne.titre(texte);
						return resultat._suivantes;
					};
					friend Resultat<ColonnesSuivantes...>& operator|(Resultat<Colonne,ColonnesSuivantes...>& resultat, const std::string& texte) 
					{
						resultat._colonne.ajout(texte);
						return resultat._suivantes;
					};
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const Resultat<Colonne,ColonnesSuivantes...>& resultat)
					{
						if(not resultat._colonne.titre().empty())
						{
							os << resultat._colonne.separation();
							os << resultat._suivantes.separation();
							os << std::endl;
							os << resultat._colonne.titre();
							os << resultat._suivantes.titre();
							os << std::endl;
						}
						os << resultat._colonne.separation();
						os << resultat._suivantes.separation();
						os << std::endl;
						for(size_t indice=0 ; indice<resultat._colonne.taille() ; ++indice)
						{
							os << resultat._colonne.format(indice);
							os << resultat._suivantes.format(indice);
							os << std::endl;
						}
						os << resultat._colonne.separation();
						os << resultat._suivantes.separation();
						os << std::endl;
						return os;
					};
					std::string format(const size_t indice) const
					{
						return _colonne.format(indice)+_suivantes.format(indice);
					}

					std::string separation() const
					{
						return _colonne.separation()+_suivantes.separation();
					}

					std::string titre() const
					{
						return _colonne.titre()+_suivantes.titre();
					}
				private:
					Colonne _colonne;
					Resultat<ColonnesSuivantes...> _suivantes;
			};

		}
	}
}

#endif
