/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_DEBOGUEUR_DONNEES_SONDE_H_
#define _MACHINE_DEBOGUEUR_DONNEES_SONDE_H_

#include <vector>
#include <map>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Debogueur { DECL_SHARED_PTR(Sonde); } } }

#include <src/machine/debogueur/donnees/directive.h>
#include <src/machine/debogueur/donnees/trame.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Debogueur
		{
			struct Sonde
			{
				Sonde()
				:_nb_instructions(0) {};
				virtual ~Sonde() {};
				virtual bool arret_debogage(const ListeDirectives& liste, const PileTrames& pile) const = 0;
				void incremente_nb_instructions()
				{
					++_nb_instructions;
				};
				unsigned long long int nb_instructions() const
				{
					return _nb_instructions;
				};
				private:
					unsigned long long int _nb_instructions;
			};

			struct SondeExecution : public Sonde
			{
				SondeExecution() = default;
				virtual ~SondeExecution() {};
				virtual bool arret_debogage(const ListeDirectives& liste, const PileTrames& pile) const
				{
					return false;
				};
			};

			struct SondeContinuation : public Sonde
			{
				SondeContinuation(const size_t niveau)
				:_niveau(niveau) {};
				virtual ~SondeContinuation() {};
				virtual bool arret_debogage(const ListeDirectives& liste, const PileTrames& pile) const
				{
					return liste._fin_trame and (static_cast<size_t>(*pile)<=_niveau);
				};
				private:
					size_t _niveau;
			};

			struct SondeSuivant : public Sonde
			{
				SondeSuivant(const size_t niveau)
				:_niveau(niveau) {};
				virtual ~SondeSuivant() {};
				virtual bool arret_debogage(const ListeDirectives& liste, const PileTrames& pile) const
				{
					return (liste._fin_trame or liste._instruction) and (static_cast<size_t>(*pile)<=_niveau);
				};
				private:
					size_t _niveau;
			};

			struct SondePasAPas : public Sonde
			{
				SondePasAPas() = default;
				virtual ~SondePasAPas() {};
				virtual bool arret_debogage(const ListeDirectives& liste, const PileTrames& pile) const
				{
					return (liste._fin_trame or liste._instruction);
				};
			};

			struct SondeInstruction : public Sonde
			{
				SondeInstruction() = default;
				virtual ~SondeInstruction() {};
				virtual bool arret_debogage(const ListeDirectives& liste, const PileTrames& pile) const
				{
					return true;
				};
			};

		}
	}
}

#endif
