/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_DEBOGUEUR_DONNEES_SURVEILLANCE_H_
#define _MACHINE_DEBOGUEUR_DONNEES_SURVEILLANCE_H_

#include <vector>
#include <map>
#include <set>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Debogueur { DECL_SHARED_PTR(Surveillance); } } }

#include <src/machine/programme/donnees/adressememoire.h>
#include <src/machine/debogueur/donnees/variable.h>
#include <src/machine/machine/machine.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Debogueur
		{
			class Debogueur;
			class ListeSurveillances;

			class Surveillance
			{
				friend class Debogueur;
				friend class ListeSurveillances;
				public:
					Surveillance() = default;
					virtual ~Surveillance() {};
					template<typename Resultat>
					void affiche(Resultat& resultat, const size_t niveau, SetLgg::Machine::Machine::MachineSP& machine)
					{
						std::string type;
						std::string expression;
						std::string adresse;
						Memoire memoire;
						std::string position;
						renseigne(niveau,machine,type,expression,adresse,memoire,position);
						resultat | type | expression | adresse | memoire._etat | memoire._adresse | memoire._type | memoire._valeur | position;
					}
					virtual void renseigne(const size_t niveau, SetLgg::Machine::Machine::MachineSP& machine, std::string& type, std::string& expression, std::string& adresse, Memoire& memoire, std::string& position) const = 0;
					template<typename Resultat>
					void operator>>(Resultat& resultat)
					{
						resultat | "" | type() | expression();
					};
					virtual std::string type() const = 0;
					virtual std::string expression() const =0;
				private:
			};

			class SurveillanceAdresseMemoire : public Surveillance
			{
				public:
					SurveillanceAdresseMemoire(const SetLgg::Machine::Programme::AdresseMemoireSP& adresse)
					:_adresse(adresse) {};
					virtual ~SurveillanceAdresseMemoire() {};
					virtual void renseigne(const size_t niveau, SetLgg::Machine::Machine::MachineSP& machine, std::string& type, std::string& expression, std::string& adresse, Memoire& memoire, std::string& position) const
					{
						type = this->type();
						expression = this->expression();
						adresse = expression;
						SetLgg::Machine::Memoire::AdresseSP adresse_vraie = Memoire::resolution(_adresse,machine);
						if(adresse_vraie)
						{
							memoire = Memoire::lecture(*adresse_vraie,machine);
						}
					};
					virtual std::string type() const
					{
						return "Address";
					};
					virtual std::string expression() const
					{
						std::ostringstream oss;
						oss << (*_adresse);
						return oss.str();
					}
					private:
					SetLgg::Machine::Programme::AdresseMemoireSP _adresse;
			};

			class SurveillanceVariable : public Surveillance
			{
				public:
					SurveillanceVariable(const VariableSP& variable)
					:_variable(variable) {};
					virtual ~SurveillanceVariable() {};
					virtual void renseigne(const size_t niveau, SetLgg::Machine::Machine::MachineSP& machine, std::string& type, std::string& expression, std::string& adresse, Memoire& memoire, std::string& position) const
					{
						type = this->type();
						expression = this->expression();
						_variable->renseigne(niveau,machine,adresse,memoire,position);
					};
					virtual std::string type() const
					{
						return "Variable";
					};
					virtual std::string expression() const
					{
						return _variable->_nom;
					};
				private:
					VariableSP _variable;
			};

			class ListeSurveillances
			{
				public:
					ListeSurveillances() = default;
					~ListeSurveillances() {};
					void ajout(SurveillanceSP surveillance)
					{
						_surveillances.push_back(surveillance);
					};
					bool enleve(size_t numero)
					{
						if(numero>=_surveillances.size())
						{
							std::cerr << "Watch address number " << numero << " does not exist." << std::endl;
							return false;
						}
						_surveillances.erase(_surveillances.begin()+numero);
						return true;
					};
					void enleve_tout()
					{
						_surveillances.clear();
					}
					template<typename Resultat>
					void operator>>(Resultat& resultat)
					{
						resultat & "" & "Type" & "Expression";
						for(auto it = _surveillances.cbegin() ; it!=_surveillances.cend() ; ++it)
						{
							(*(*it)) >> resultat;
						}
					}
					template<typename Resultat>
					void affiche(Resultat& resultat, size_t niveau, SetLgg::Machine::Machine::MachineSP& machine)
					{
						resultat & "" & "" & "Address" & "State" & "Real address" & "Type" & "Value" & "File";
						for(auto it = _surveillances.cbegin() ; it!=_surveillances.cend() ; ++it)
						{
							(*it)->affiche(resultat,niveau,machine);
						}
					}
					operator bool () const
					{
						return not _surveillances.empty();
					};
				private:
					std::vector<SurveillanceSP> _surveillances;

			};
		}
	}
}

#endif
