/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_FLUX_FLUX_H_
#define _MACHINE_FLUX_FLUX_H_

#include <src/global/global.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Flux { DECL_SHARED_PTR(Flux); } } }

#include <src/machine/flux/fluxgenerique.h>
#include <src/machine/memoire/donnees/nomflux.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Plugin
		{
			struct Interface;
		}
		namespace Flux
		{
			class Flux
			{
				public:
					virtual ~Flux() {};
					virtual void ouverture() = 0 ;
					virtual void fermeture() = 0 ;
					virtual void fermeture_lecture() = 0 ;
					virtual void fermeture_ecriture() = 0 ;
					virtual void ecriture(const std::string&) = 0 ;
					virtual DonneesLues lecture(const TypeLecture&) = 0 ;
					virtual long int cherche(const TypeCherche type, const long int decalage) = 0;
					virtual FluxSP attente(const std::string&) = 0 ;
					virtual bool lisible() = 0;
					virtual bool inscriptible() = 0;
					virtual bool cherchable() = 0;
					virtual void descripteur_debug(const std::string& descripteur) = 0;
					virtual int descripteur_lecture() const = 0;
					virtual int descripteur_ecriture() const = 0;
					virtual std::string tampon_depile(const size_t taille) = 0;
					virtual void tampon_empile(const std::string chaine) = 0;
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select = true) = 0;
					virtual std::set<int> descripteurs_non_systeme() const = 0;
					virtual bool media_actif() const = 0;
					template<typename oStream>
					friend oStream& operator<<(oStream&,const Flux&);
					virtual std::ostream& format(std::ostream&) const =0 ;
				protected:
					Flux() {};
					SetLgg::Global::Source::PositionSP _position;
			};

			template<typename _Profil>
			struct ProfilFlux : public _Profil
			{
				typedef _Profil Profil;
				ProfilFlux(typename Profil::_Type type)
				:_Profil(type) {};
			};

			template<typename oStream>
			oStream& operator<<(oStream& os,const Flux& flux)
			{
				return flux.format(os);
			}

			class FluxFichierLectureSeule : public Flux, private ProfilFlux< FluxGenerique<Fichier<TypeOuvertureFichier::LECTURE>,OuvertureFichier,LectureSimpleFichier,OperationNonSupportee,Fermeture,OperationNonSupportee,Cherche,OperationNonSupportee,OperationNonSupportee> >
			{
				friend class SetLgg::Machine::Plugin::Interface;
				public:
					FluxFichierLectureSeule(const std::string& nom)
					:ProfilFlux(Profil::_Type(nom)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					FluxFichierLectureSeule(const int fd, const std::string& nom)
					:ProfilFlux(Profil::_Type(fd,nom)) {};
			};

			class Stdin : public Flux, private ProfilFlux< FluxGenerique<Fichier<TypeOuvertureFichier::LECTURE>,OperationNonSupportee,LectureSimpleFichier,OperationNonSupportee,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee> >
			{
				friend class SetLgg::Machine::Plugin::Interface;
				public:
					Stdin()
					:ProfilFlux(Profil::_Type(Stdin::nom()))
					{
						_type._descripteur_fichier=0;
					};
					static SetLgg::Machine::Memoire::NomFlux nom()
					{
						static SetLgg::Machine::Memoire::NomFlux nom = SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDIN);
						return nom;
					};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return {};
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					Stdin(const int fd, const std::string& nom)
					:ProfilFlux(Profil::_Type(fd,nom)) {};
			};

			class FluxFichierEcritureSeule : public Flux, private ProfilFlux< FluxGenerique<Fichier<TypeOuvertureFichier::ECRITURE>,OuvertureFichier,OperationNonSupportee,EcritureSimpleFichier,Fermeture,OperationNonSupportee,Cherche,OperationNonSupportee,OperationNonSupportee> >
			{
				friend class SetLgg::Machine::Plugin::Interface;
				public:
					FluxFichierEcritureSeule(const std::string& nom)
					:ProfilFlux(Profil::_Type(nom)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					FluxFichierEcritureSeule(const int fd, const std::string& nom)
					:ProfilFlux(Profil::_Type(fd,nom)) {};
			};

			class FluxFichierEcritureSeuleSysteme : public Flux, protected ProfilFlux< FluxGenerique<Fichier<TypeOuvertureFichier::ECRITURE>,OperationNonSupportee,OperationNonSupportee,EcritureSimpleFichier,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee> >
			{
				friend class SetLgg::Machine::Plugin::Interface;
				public:
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				protected:
					FluxFichierEcritureSeuleSysteme(const std::string& nom)
					:ProfilFlux(Profil::_Type(nom)) {};
				private:
					FluxFichierEcritureSeuleSysteme(const int fd, const std::string& nom)
					:ProfilFlux(Profil::_Type(fd,nom)) {};
			};

			class Stdout : public FluxFichierEcritureSeuleSysteme
			{
				public:
					Stdout()
					:FluxFichierEcritureSeuleSysteme(Stdout::nom())
					{
						_type._descripteur_fichier=1;
					};
					static SetLgg::Machine::Memoire::NomFlux nom()
					{
						static SetLgg::Machine::Memoire::NomFlux nom = SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDOUT);
						return nom;
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return {};
					}
			};

			class Stderr : public FluxFichierEcritureSeuleSysteme
			{
				public:
					Stderr()
					:FluxFichierEcritureSeuleSysteme(Stderr::nom())
					{
						_type._descripteur_fichier=2;
					};
					static SetLgg::Machine::Memoire::NomFlux nom()
					{
						static SetLgg::Machine::Memoire::NomFlux nom = SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDERR);
						return nom;
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return {};
					}
			};

			class FluxFichierEcritureAjout : public Flux, private ProfilFlux< FluxGenerique<Fichier<TypeOuvertureFichier::ECRITUREAJOUT>,OuvertureFichier,OperationNonSupportee,EcritureSimpleFichier,Fermeture,OperationNonSupportee,Cherche,OperationNonSupportee,OperationNonSupportee> >
			{
				friend class SetLgg::Machine::Plugin::Interface;
				public:
					FluxFichierEcritureAjout(const std::string& nom)
					:ProfilFlux(Profil::_Type(nom)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					FluxFichierEcritureAjout(const int fd, const std::string& nom)
					:ProfilFlux(Profil::_Type(fd,nom)) {};
			};

			class FluxFichierLectureEcriture : public Flux, private FluxGenerique<Fichier<TypeOuvertureFichier::LECTUREECRITURE>,OuvertureFichier,LectureSimpleFichier,EcritureSimpleFichier,Fermeture,OperationNonSupportee,Cherche,OperationNonSupportee,OperationNonSupportee>
			{
				friend class SetLgg::Machine::Plugin::Interface;
				typedef FluxGenerique<Fichier<TypeOuvertureFichier::LECTUREECRITURE>,OuvertureFichier,LectureSimpleFichier,EcritureSimpleFichier,Fermeture,OperationNonSupportee,Cherche,OperationNonSupportee,OperationNonSupportee> Profil;
				public:
					FluxFichierLectureEcriture(const std::string& nom)
					:Profil(Profil::_Type(nom)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					FluxFichierLectureEcriture(const int fd, const std::string& nom)
					:Profil(Profil::_Type(fd,nom)) {};
			};

			class FluxSocketTCPServeurAttend : public Flux, private FluxGenerique<SocketTCP<TypeSocketTCP::SERVEUR_ACCEPTE>,OuvertureSocketEcoute,OperationNonSupportee,OperationNonSupportee,Fermeture,Attente,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee>
			{
				friend class SetLgg::Machine::Plugin::Interface;
				typedef FluxGenerique<SocketTCP<TypeSocketTCP::SERVEUR_ACCEPTE>,OuvertureSocketEcoute,OperationNonSupportee,OperationNonSupportee,Fermeture,Attente,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee> Profil;
				public:
					FluxSocketTCPServeurAttend(const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Profil(Profil::_Type(ip_locale,port_local,ip_distante,port_distant)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return confirmation_select; // exception (:wait < * debloque sur un accept())
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					FluxSocketTCPServeurAttend(const int fd, const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Profil(Profil::_Type(fd,ip_locale,port_local,ip_distante,port_distant)) {};
			};

			class FluxSocketTCPServeurConnection : public Flux, private FluxGenerique<SocketTCP<TypeSocketTCP::SERVEUR_CONNECTION>,OperationNonSupportee,LectureSimpleSocket,EcritureSimpleSocket,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee>
			{
				typedef FluxGenerique<SocketTCP<TypeSocketTCP::SERVEUR_CONNECTION>,OperationNonSupportee,LectureSimpleSocket,EcritureSimpleSocket,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee> Profil;
				public:
					FluxSocketTCPServeurConnection(const int descripteur)
					:Profil(Profil::_Type(descripteur)) {};
					FluxSocketTCPServeurConnection(const int descripteur, const std::string& ip_distante, const std::string& port_distant)
					:Profil(Profil::_Type(descripteur,ip_distante,port_distant)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
			};
		
			class FluxSocketTCPClient : public Flux, private FluxGenerique<SocketTCP<TypeSocketTCP::CLIENT>,OuvertureSocketConnection,LectureSimpleSocket,EcritureSimpleSocket,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee>
			{
				friend class SetLgg::Machine::Plugin::Interface;
				typedef FluxGenerique<SocketTCP<TypeSocketTCP::CLIENT>,OuvertureSocketConnection,LectureSimpleSocket,EcritureSimpleSocket,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee> Profil;
				public:
					FluxSocketTCPClient(const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Profil(Profil::_Type(ip_locale,port_local,ip_distante,port_distant)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					FluxSocketTCPClient(const int fd, const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Profil(Profil::_Type(fd,ip_locale,port_local,ip_distante,port_distant)) {};
			};

			class FluxSocketUDP : public Flux, private FluxGenerique<SocketUDP,OuvertureSocketUDP,LectureAvecExpediteur,EcritureAvecDestinataire,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee>
			{
				friend class SetLgg::Machine::Plugin::Interface;
				typedef FluxGenerique<SocketUDP,OuvertureSocketUDP,LectureAvecExpediteur,EcritureAvecDestinataire,Fermeture,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee,OperationNonSupportee> Profil;
				public:
					FluxSocketUDP(const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Profil(Profil::_Type(ip_locale,port_local,ip_distante,port_distant)) {};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						Profil::ecriture(valeur);	
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						return Profil::lecture(type);
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						return Profil::_Lecture::_lisible;
					};
					virtual bool inscriptible()
					{
						return Profil::_Ecriture::_inscriptible;
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
				private:
					FluxSocketUDP(const int fd, const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Profil(Profil::_Type(fd,ip_locale,port_local,ip_distante,port_distant)) {};
			};

			class FluxPaireTubesAnonymes : public Flux, private FluxGenerique<PaireTubesAnonymes,OperationNonSupportee,LectureSimplePaireTubesAnonymes,EcritureSimplePaireTubesAnonymes,FermeturePaireTubesAnonymes,OperationNonSupportee,OperationNonSupportee,FermeturePaireTubesAnonymesLecture,FermeturePaireTubesAnonymesEcriture>
			{
				typedef FluxGenerique<PaireTubesAnonymes,OperationNonSupportee,LectureSimplePaireTubesAnonymes,EcritureSimplePaireTubesAnonymes,FermeturePaireTubesAnonymes,OperationNonSupportee,OperationNonSupportee,FermeturePaireTubesAnonymesLecture,FermeturePaireTubesAnonymesEcriture> Profil;
				public:
					FluxPaireTubesAnonymes(const int lecture, const int ecriture)
					:Profil(Profil::_Type(lecture,ecriture)){};
					virtual void ouverture()
					{
						Profil::ouverture();	
					};
					virtual void fermeture()
					{
						Profil::fermeture();	
					};
					virtual void fermeture_lecture()
					{
						Profil::fermeture_lecture();	
					};
					virtual void fermeture_ecriture()
					{
						Profil::fermeture_ecriture();	
					};
					virtual void ecriture(const std::string& valeur)
					{
						if(Profil::_type._ecriture_possible)
						{
							Profil::ecriture(valeur);
						}
						else
						{
							throw FluxOperationInvalide(Profil::_type._descripteur_debug);
						}
					};
					virtual DonneesLues lecture(const TypeLecture& type)
					{
						if(Profil::_type._lecture_possible)
						{
							return Profil::lecture(type);
						}
						else
						{
							throw FluxOperationInvalide(Profil::_type._descripteur_debug);
						}
					};
					virtual long int cherche(const TypeCherche type, const long int decalage)
					{
						return Profil::cherche(type,decalage);
					};
					virtual FluxSP attente(const std::string& connecteur)
					{
						return Profil::attente(connecteur);
					};
					virtual bool lisible()
					{
						if(Profil::_type._lecture_possible)
						{
							return Profil::_Lecture::_lisible;
						}
						else
						{
							return false;
						}
					};
					virtual bool inscriptible()
					{
						if(Profil::_type._ecriture_possible)
						{
							return Profil::_Ecriture::_inscriptible;
						}
						else
						{
							return false;
						}
					};
					virtual bool cherchable()
					{
						return Profil::_Cherche::_cherchable;
					};
					virtual void descripteur_debug(const std::string& descripteur)
					{
						Profil::descripteur_debug(descripteur);
					};
					virtual int descripteur_lecture() const
					{
						return Profil::descripteur_lecture();
					};
					virtual int descripteur_ecriture() const
					{
						return Profil::descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return Profil::tampon_depile(taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						Profil::tampon_empile(chaine);
					};
					virtual bool pret_a_lire(const TypeLecture mode, bool confirmation_select)
					{
						return Profil::pret_a_lire(mode);
					};
					virtual bool media_actif() const
					{
						return Profil::media_actif();
					};
					virtual std::set<int> descripteurs_non_systeme() const
					{
						return Profil::descripteurs_non_systeme();
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << *static_cast<const Profil*>(this);
					};
			};
		}
	}
}
#endif
