/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <algorithm>
#include <netdb.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <cstring>
#include <cerrno>
#include <src/machine/flux/fluxgenerique.h>
#include <src/machine/flux/flux.h>
using namespace SetLgg::Machine::Flux;
#define TAILLE_BUFFER_LOCAL 1024
#define TAILLE_DONNEES_FLUX 4096
namespace SetLgg
{
	namespace Machine
	{
		namespace Flux
		{
			namespace Outil
			{
				struct Memoire
				{
					Memoire()
					{
						res0 = 0;
						buffer_local=0;
					};

					~Memoire()
					{
						if(res0)
							::freeaddrinfo(res0);
						delete [] buffer_local;
					};
					void operator() (char *buffer)
					{
						delete buffer_local;
						buffer_local=buffer;
					};
					struct addrinfo *res0;
					char *buffer_local;
				};
				struct OuvertureSocket
				{
					template<bool appelle_bind,bool appelle_listen, bool appelle_connect>
						static int ouvrir(const int type, const int protocole, const std::string& ip, const std::string& port, const std::string& d)
						{
							int fd;
							struct addrinfo hints, *res;
							int error;
							std::ostringstream details;
							details << SetLgg::Global::Exception::details;
							SetLgg::Machine::Flux::Outil::Memoire memoire;

							::memset(&hints, 0, sizeof(hints));
							hints.ai_family = PF_UNSPEC;
							hints.ai_socktype = type;
							error = ::getaddrinfo(ip.c_str(), port.c_str(), &hints, &memoire.res0);
							if (error)
							{
								std::ostringstream details;
								details << SetLgg::Global::Exception::details << std::endl << "getaddrinfo: " << ::strerror(errno);
								throw OuvertureFluxImpossible(d,details.str());
							}
							fd = -1;
							for (res = memoire.res0 ; res ; res = res->ai_next)
							{
								fd = ::socket(res->ai_family, res->ai_socktype, res->ai_protocol);
								if (fd < 0)
								{
									Socket::log_erreur(res,details,"socket",errno);
									continue;
								}

								if(appelle_bind)
								{
									if (::bind(fd, res->ai_addr, res->ai_addrlen) < 0)
									{
										Socket::log_erreur(res,details,"bind",errno);
										::close(fd);
										fd = -1;
										continue;
									}
								}

								if(appelle_listen)
								{
									if(::listen(fd,20)<0)
									{
										Socket::log_erreur(res,details,"listen",errno);
										::close(fd);
										fd = -1;
										continue;
									}
								}

								if(appelle_connect)
								{
									if (::connect(fd, res->ai_addr, res->ai_addrlen) < 0)
									{
										Socket::log_erreur(res,details,"connect",errno);
										::close(fd);
										fd = -1;
										continue;
									}
								}
								break;
							}
							if (fd < 0)
							{
								throw OuvertureFluxImpossible(d,details.str());
							}
							return fd;
						}
				};
				void rembobine(const int desc, const bool cherchable, std::stringstream& tampon)
				{
					if(not cherchable)
						return;
					size_t taille = tampon.str().size();
					tampon.str("");
					if(::lseek(desc,-taille,SEEK_CUR)<0)
						throw;
				}
			}
		}
	}
}

bool Socket::resolution_ip_port(const struct sockaddr* sock, const socklen_t longueur, std::string& ip, std::string& port)
{
	char host[TAILLE_DONNEES_FLUX+1];
	char serv[TAILLE_DONNEES_FLUX+1];
	bool resultat=::getnameinfo(sock,longueur,host,TAILLE_DONNEES_FLUX,serv,TAILLE_DONNEES_FLUX,NI_NUMERICHOST)==0;
	if(resultat)
	{
		ip = host;
		port = serv;
	}
	return resultat;
}

void Socket::log_erreur(struct addrinfo *res, std::ostringstream& os, const std::string& fonction, int erreur)
{
	std::string ip;
	std::string port;
	os << std::endl << fonction << " ";
	if(Socket::resolution_ip_port(res->ai_addr,res->ai_addrlen,ip,port))
	{
		os << "(" << ip << " : " << port << ")";
	}
	os << ": " << ::strerror(erreur);
}

int OuvertureFichier::ouvrir(const std::string& f, const int flags, const int flags_statiques, const std::string& d)
{
	struct stat status;
	int retour = ::stat(f.c_str(),&status);
	bool test_fichier_existe = (flags == O_RDONLY);
	if(retour<0)
	{
		if(test_fichier_existe or (errno!=ENOENT))
		{
			std::ostringstream details;
			details << SetLgg::Global::Exception::details << std::endl << "stat ("<< f << "): " << ::strerror(errno); 
			throw OuvertureFluxImpossible(d,details.str());
		}
		test_fichier_existe = false;
	}
	if(test_fichier_existe and (S_ISDIR(status.st_mode)))
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "stat (" << f << "): " << "Is a directory"; 
		throw OuvertureFluxImpossible(d,details.str());
	}
	retour = ::open(f.c_str(),flags_statiques|flags,S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
	if(retour<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "open (" << f << "): " << ::strerror(errno); 
		throw OuvertureFluxImpossible(d,details.str());
	}
	return retour;
}

int OuvertureSocketEcoute::ouvrir(const int type, const int protocole, const std::string& ip_locale, const std::string& port_local, const std::string& d)
{
	return Outil::OuvertureSocket::ouvrir<true,true,false>(type,protocole,ip_locale,port_local,d);
}

int OuvertureSocketConnection::ouvrir(const int type, const int protocole, const std::string& ip_distante, const std::string& port_distant, const std::string& d)
{
	return Outil::OuvertureSocket::ouvrir<false,false,true>(type,protocole,ip_distante,port_distant,d);
}

int OuvertureSocketUDP::ouvrir(const int type, const int protocole, const std::string& ip_locale, const std::string& port_local, const std::string& d)
{
	return Outil::OuvertureSocket::ouvrir<true,false,false>(type,protocole,ip_locale,port_local,d);
}

struct ConversionTypeCherche
{
	static int reference_type(const TypeCherche type)
	{
		switch(type)
		{
			case TypeCherche::ABSOLUE:
				return SEEK_SET;
			case TypeCherche::RELATIF:
				return SEEK_CUR;
			case TypeCherche::FIN:
				return SEEK_END;
		};
		return SEEK_CUR;
	}
};

long int Cherche::cherche(const int desc, TypeCherche type, const long int decalage, const std::string& d)
{
	off_t position = ::lseek(desc,decalage,ConversionTypeCherche::reference_type(type));
	if(position<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "lseek: " << ::strerror(errno);
		throw ChercheFluxImpossible(d,details.str());
	}
	return position;
}

FluxSP Attente::attendre(const int desc, const std::string& connecteur, const std::string& d)
{
	struct sockaddr client;
	socklen_t longueur = sizeof(client);
	int fd;
	errno=0;
	do
	{
		fd = ::accept(desc,&client,&longueur);
	}
	while(errno==EAGAIN);
	if (fd < 0)
	{
		if(errno==EINTR)
		{
			return FluxSP();
		}
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "accept: " << ::strerror(errno); 
		throw AttenteFluxImpossible(d,details.str());
	}
	FluxSP flux;
	std::string ip;
	std::string port;
	if(Socket::resolution_ip_port(&client,longueur,ip,port))
	{
		flux=FluxSP(new FluxSocketTCPServeurConnection(fd,ip,port));
	}
	else
	{
		flux=FluxSP(new FluxSocketTCPServeurConnection(fd));
	}
	flux->descripteur_debug(connecteur);
	return flux;
}

bool LectureSimple::lecture_non_bloquante(const int desc, const std::string& d)
{
	int flags = ::fcntl(desc, F_GETFL, 0);
	::fcntl(desc, F_SETFL, flags bitor O_NONBLOCK);
	char buffer_local[TAILLE_BUFFER_LOCAL];
	int taille;
	while((taille=::read(desc,buffer_local,sizeof(buffer_local)-1))>0)
	{
		_buffer.seekp(0,std::ios_base::end);
		_buffer.write(buffer_local,taille);
	}
	::fcntl(desc, F_SETFL, flags bitand (compl O_NONBLOCK));
	if((taille<0) and (not ((errno == EAGAIN) or (errno == EWOULDBLOCK))))
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "read: " << ::strerror(errno); 
		throw LectureFluxImpossible(d,details.str());
	}
	return taille==0;
}

bool LectureSimple::pret_a_lire(const int desc, const TypeLecture type, const bool cherchable, const std::string& d)
{
	bool fin_fichier = lecture_non_bloquante(desc,d);
	switch(type._type)
	{
		case TypeLecture::Type::DISPONIBLE:
		{
			Outil::rembobine(desc,cherchable,_buffer);
			return true;
		}
		case TypeLecture::Type::LIGNE:
		{
			std::string buffer = _buffer.str();
			Outil::rembobine(desc,cherchable,_buffer);
			return (std::find(buffer.begin(),buffer.end(),'\n')!=buffer.end()) or fin_fichier;
		}
		case TypeLecture::Type::TOUT:
		{
			Outil::rembobine(desc,cherchable,_buffer);
			return fin_fichier;
		}
		case TypeLecture::Type::TAILLE:
		{
			std::string buffer = _buffer.str();
			Outil::rembobine(desc,cherchable,_buffer);
			return (buffer.size()>=type._taille) or fin_fichier;
		}
		default:
		{
			throw;
		}
	}
}

DonneesLues LectureSimple::lire(const int desc, const TypeLecture type, const bool cherchable, const std::string& d)
{
	switch(type._type)
	{
		case TypeLecture::Type::DISPONIBLE:
		{
			if(lecture_non_bloquante(desc,d))
				return DonneesLues();
			std::string resres = _buffer.str();
			_buffer.str("");
			return DonneesLues(resres);
			break;
		}
		case TypeLecture::Type::LIGNE:
		{
			std::string buffer = _buffer.str();
			int taille = 1;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			if(std::find(buffer.begin(),buffer.end(),'\n')==buffer.end())
			{
				int flags = ::fcntl(desc, F_GETFL, 0);
				::fcntl(desc, F_SETFL, flags & (~O_NONBLOCK));
				char buffer_local[TAILLE_BUFFER_LOCAL];
				while((taille=::read(desc,buffer_local,sizeof(buffer_local)-1))>0)
				{
					if(std::find(buffer_local,buffer_local+taille,'\n')!=(buffer_local+taille))
						break;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
					_buffer.seekp(0,std::ios_base::end);
					_buffer.write(buffer_local,taille);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
				}
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
				_buffer.seekp(0,std::ios_base::end);
				_buffer.write(buffer_local,taille);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			}
			std::string resres;
			buffer = _buffer.str();
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			std::getline(_buffer,resres);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			if(_buffer.good() and (buffer.size()==_buffer.str().size()) and (not buffer.empty()))
			{	// parce que parfois, getline ne vire pas les caracteres extraits sans que good() renvoie false...
				_buffer.str(_buffer.str().substr(resres.size()+1));
				_buffer.seekp(0,std::ios_base::end);
			}
			Outil::rembobine(desc,cherchable,_buffer);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			if((taille==0) and resres.empty())
				return DonneesLues();
			return DonneesLues(resres);
			break;
		}
		case TypeLecture::Type::TOUT:
		{
			int flags = ::fcntl(desc, F_GETFL, 0);
			::fcntl(desc, F_SETFL, flags & (~O_NONBLOCK));
			char buffer_local[TAILLE_BUFFER_LOCAL];
			int taille;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str().size() << std::endl;
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			while((taille=::read(desc,buffer_local,sizeof(buffer_local)-1))>0)
			{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
				_buffer.seekp(0,std::ios_base::end);
				_buffer.write(buffer_local,taille);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << taille << " " << _buffer.str().size() << std::endl;
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			}
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << std::endl;
#endif
			if(taille<0)
			{
				std::ostringstream details;
				details << SetLgg::Global::Exception::details << std::endl << "read: " << ::strerror(errno); 
				throw LectureFluxImpossible(d,details.str());
			}
			std::string resres = _buffer.str();
			_buffer.str("");
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << " <" << taille << "> \"" << resres << "\"" <<   std::endl;
SETLGG_TRACE << resres.size() << std::endl;
#endif
			return DonneesLues(resres);
			break;
		}
		case TypeLecture::Type::TAILLE:
		{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << " " << _buffer.str().size() << " vs " << type._taille << std::endl;
#endif
			if(_buffer.str().size()>=type._taille)
			{
				std::string resres = _buffer.str().substr(0,type._taille);
				_buffer.str(_buffer.str().substr(type._taille));
				_buffer.seekp(0,std::ios_base::end);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << " " << _buffer.str().size() << " vs " << type._taille << std::endl;
#endif
				return DonneesLues(resres);
			}
			int flags = ::fcntl(desc, F_GETFL, 0);
			::fcntl(desc, F_SETFL, flags & (~O_NONBLOCK));

			int taille = type._taille-_buffer.str().size();
			Outil::Memoire memoire;
			memoire.buffer_local = new char[taille+1];
			while(taille>0)
			{
				int lu = ::read(desc,memoire.buffer_local,taille);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "::read(..., " << taille << ") => " << lu << std::endl;
#endif
				if(lu<0)
				{
					std::ostringstream details;
					details << SetLgg::Global::Exception::details << std::endl << "read: " << ::strerror(errno); 
					throw LectureFluxImpossible(d,details.str());
				}
				if(lu==0)
				{
					break;
				}
				_buffer.seekp(0,std::ios_base::end);
				_buffer.write(memoire.buffer_local,lu);
				taille -= lu;
			}
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _buffer.str() << " " << _buffer.str().size() << " vs " << type._taille << std::endl;
#endif
			std::string resres = _buffer.str();
			_buffer.str("");
			Outil::rembobine(desc,cherchable,_buffer);
			if((taille==0) and resres.empty())
				return DonneesLues();
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "Donnees lues: " << resres << std::endl;
#endif
			return DonneesLues(resres);
			break;
		}
	}
	return DonneesLues();
}

/*
bool LectureAvecExpediteur::expediteur_accepte(struct sockaddr expediteur, size_t taille_expediteur, struct sockaddr masque, size_t taille_masque)
{
	if(taille_expediteur!=taille_masque)
		return false;
	if(expediteur.sa_family!=masque.sa_family)
		return false;
	switch(expediteur.sa_family)
	{
		case AF_INET:
		{
			struct sockaddr_in *ipv4exp = reinterpret_cast<struct sockaddr_in*>(&expediteur);
			struct sockaddr_in *ipv4msk = reinterpret_cast<struct sockaddr_in*>(&masque);
			if(ipv4exp->sin_port!=ipv4msk->sin_port)
				return false;
			struct in_addr* expip = &ipv4exp->sin_addr;
			struct in_addr* mskip = &ipv4msk->sin_addr;
			for( ; expip<(&(ipv4exp->sin_addr)+sizeof(struct in_addr)) ; ++expip,++mskip)
			{
				char e =*(reinterpret_cast<char*>(expip));
				char m =*(reinterpret_cast<char*>(mskip));
				if((e&m)!=m)
				{
					return false;
				}
			}
			return true;
			break;
		}
		case AF_INET6:
		{
			struct sockaddr_in6 *ipv6exp = reinterpret_cast<struct sockaddr_in6*>(&expediteur);
			struct sockaddr_in6 *ipv6msk = reinterpret_cast<struct sockaddr_in6*>(&masque);
			if(ipv6exp->sin6_port!=ipv6msk->sin6_port)
				return false;
			struct in6_addr* expip = &ipv6exp->sin6_addr;
			struct in6_addr* mskip = &ipv6msk->sin6_addr;
			for( ; expip<(&(ipv6exp->sin6_addr)+sizeof(struct in6_addr)) ; ++expip,++mskip)
			{
				char e =*(reinterpret_cast<char*>(expip));
				char m =*(reinterpret_cast<char*>(mskip));
				if((e&m)!=m)
				{
					return false;
				}
			}
			return true;
			break;
		}
		default:
			break;
	}
	return false;
}
*/

DonneesLues LectureAvecExpediteur::lire(const int desc, const TypeLecture type, const std::string& ip_distante, const std::string& port_distant, const bool cherchable, const std::string& d)
{
	ssize_t taille;
	struct addrinfo hints, *res;
	int error;
	SetLgg::Machine::Flux::Outil::Memoire memoire;
	std::ostringstream details;
	details << SetLgg::Global::Exception::details;

	::memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	error = ::getaddrinfo(ip_distante.c_str(), port_distant.c_str(), &hints, &memoire.res0);
	if (error)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "getaddrinfo: " << ::strerror(errno);
		throw LectureFluxImpossible(d,details.str());
	}
	taille = -1;
	for (res = memoire.res0 ; res ; res = res->ai_next)
	{
		char buffer_longueur[1];
		struct sockaddr expediteur;
		::memset(&expediteur, 0, sizeof(expediteur));
		socklen_t taille_expediteur = sizeof(expediteur);
		taille=::recvfrom(desc,buffer_longueur,0,MSG_PEEK|MSG_TRUNC|((type._type==TypeLecture::Type::DISPONIBLE)?MSG_DONTWAIT:0), &expediteur,&taille_expediteur);
		if (taille>=0)
		{
			memoire(new char[taille+1]);
			taille=::recvfrom(desc,memoire.buffer_local,taille,((type._type==TypeLecture::Type::DISPONIBLE)?MSG_DONTWAIT:0), NULL,NULL);
			if(taille>=0)
			{
				//if(expediteur_accepte(expediteur,taille_expediteur,*(res->ai_addr),res->ai_addrlen))
				//{
					_buffer.seekp(0,std::ios_base::end);
					_buffer.write(memoire.buffer_local,taille);
				//}
				//else
				//	continue;
			}
		}
		if (taille<0)
		{
			_buffer.str("");
			if((errno==EAGAIN) or (errno==EWOULDBLOCK))
			{
				return DonneesLues("");
			}
			Socket::log_erreur(res,details,"recvfrom",errno);
			continue;
		}
		break;
	}
	if (taille < 0)
	{
		throw LectureFluxImpossible(d,details.str());
	}

	std::string resres;
	switch(type._type)
	{
		case TypeLecture::Type::LIGNE:
			{
				std::getline(_buffer,resres);
				break;
			}
		case TypeLecture::Type::TAILLE:
			{
				resres=_buffer.str().substr(0,type._taille);
				break;
			}
		default:
			{
				resres = _buffer.str();
				break;
			}
	}
	_buffer.str("");
	if(taille==0)
		return DonneesLues();
	return DonneesLues(resres);
}

void EcritureSimpleFichier::ecrire(const int desc, const std::string& valeur, const std::string& d)
{
	struct sigaction ignore;
	ignore.sa_flags=0;
	::sigemptyset(&(ignore.sa_mask));
	ignore.sa_handler=SIG_IGN;
	struct sigaction sauvegarde;
	::sigaction(SIGPIPE,&ignore,&sauvegarde);
	ssize_t taille = ::write(desc,valeur.c_str(),valeur.size());
	::sigaction(SIGPIPE,&sauvegarde,NULL);
	if(taille<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "write: " << ::strerror(errno); 
		throw EcritureFluxImpossible(d,details.str());
	}
	if(taille<valeur.size())
	{
		throw EcritureFluxImpossible(d);	
	}
}

void EcritureSimpleSocket::ecrire(const int desc, const std::string& valeur, const std::string& d)
{
	ssize_t taille = ::send(desc,valeur.c_str(),valeur.size(),MSG_NOSIGNAL);
	if(taille<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "write: " << ::strerror(errno); 
		throw EcritureFluxImpossible(d,details.str());
	}
	if(taille<valeur.size())
	{
		throw EcritureFluxImpossible(d);
	}
}

void EcritureAvecDestinataire::ecrire(const int desc, const std::string& ip_distante, const std::string& port_distant, const std::string& valeur, const std::string& d)
{
	int err;
	struct addrinfo hints, *res;
	int error;
	SetLgg::Machine::Flux::Outil::Memoire memoire;
	std::ostringstream details;
	details << SetLgg::Global::Exception::details;

	::memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	error = ::getaddrinfo(ip_distante.c_str(), port_distant.c_str(), &hints, &memoire.res0);
	if (error)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "getaddrinfo: " << ::strerror(errno);
		throw EcritureFluxImpossible(d,details.str());
	}
	err = -1;
	for (res = memoire.res0 ; res ; res = res->ai_next)
	{
		err=::sendto(desc,valeur.c_str(),valeur.size(), MSG_NOSIGNAL, res->ai_addr, res->ai_addrlen); 
		if (err<0)
		{
			Socket::log_erreur(res,details,"sendto",errno);
			continue;
		}

		break;
	}
	if (err < 0)
	{
		throw EcritureFluxImpossible(d,details.str());
	}
}

void Fermeture::fermer(const int desc, const std::string& d)
{
	if(::close(desc)<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "close: " << ::strerror(errno);
		throw FermetureFluxImpossible(d);
	}
}
