/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_FLUX_FLUXGENERIQUE_H_
#define _MACHINE_FLUX_FLUXGENERIQUE_H_

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/socket.h>


#include <src/global/global.h>
#include <src/global/sources/position.h>
#include <src/global/exceptions.h>

#include <src/machine/programme/donnees/valeur.h>
struct sockaddr;

namespace SetLgg
{
	namespace Machine
	{
		namespace Flux
		{
			class Stdin;
			class Stdout;
			class Stderr;
			DECL_SHARED_PTR(Flux);

			struct TypeLecture
			{
				enum class Type {DISPONIBLE, TOUT, LIGNE, TAILLE};
				TypeLecture(const Type type, const size_t taille=0)
				:_type(type), _taille(taille) {}
				Type _type;
				size_t _taille;
			};

			enum class TypeCherche {ABSOLUE,RELATIF,FIN};

			int reference_type(const TypeCherche type);

			template<typename oStream>
				oStream& operator<<(oStream& os, const TypeCherche& type)
				{
					switch(type)
					{
						case TypeCherche::ABSOLUE:
							os << "ABSOLUTE";
							break;
						case TypeCherche::RELATIF:
							os << "RELATIVE";
							break;
						case TypeCherche::FIN:
							os << "END";
							break;
					}
					return os;
				}

			struct Outils
			{
				static std::string depile_tampon(std::stringstream& buffer, const size_t taille)
				{
					std::string str = buffer.str().substr(0,taille);
					if(taille<buffer.str().size())
					{
						std::string reste = buffer.str().substr(taille);
						buffer.str(reste);
					}
					else
					{
						buffer.str("");
					}
					return str;
				}

				static void empile_tampon(std::stringstream& buffer, const std::string& chaine)
				{
					std::string buffer_ = chaine+buffer.str();
					buffer.str(buffer_);
				}
			};

			struct DonneesLues
			{
				DonneesLues()
				:_eof(true), _donnees() {};
				DonneesLues(const std::string& donnees)
				:_eof(false), _donnees(donnees) {};
				operator bool () const { return _eof; };
				operator const std::string& () const { return _donnees; };
				private:
					bool _eof;
					std::string _donnees;
			};

			template<typename Type, typename Ouverture, typename Lecture, typename Ecriture, typename Fermeture, typename Attente, typename Cherche, typename FermetureLecture, typename FermetureEcriture>
			class FluxGenerique
			{
				friend class Stdin;
				friend class Stdout;
				friend class Stderr;

				public:
					FluxGenerique(const Type& type)
					:_type(type) {};
					FluxGenerique(const FluxGenerique<Type,Ouverture,Lecture,Ecriture,Fermeture,Attente,Cherche,FermetureLecture,FermetureEcriture>&) = delete;
					virtual ~FluxGenerique() { };
					void ouverture()
					{
						_Ouverture ouverture;
						ouverture(_type);
					};
					DonneesLues lecture(const TypeLecture& type)
					{
						return _lecture(_type,type,_Cherche::_cherchable);
					};
					void ecriture(const std::string& valeur)
					{
						_Ecriture ecriture;
						ecriture(_type,valeur);
					};
					void fermeture()
					{
						_Fermeture fermeture;
						fermeture(_type);
					};
					long int cherche(const TypeCherche type, const long int decalage)
					{
						_Cherche cherche;
						return cherche(_type,type,decalage);
					};
					FluxSP attente(const std::string& connecteur)
					{
						_Attente attente;
						return attente(_type,connecteur);
					};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const FluxGenerique<Type,Ouverture,Lecture,Ecriture,Fermeture,Attente,Cherche,FermetureLecture,FermetureEcriture>& flux)
					{
						return os << flux._type;
					};
					virtual std::ostream& format(std::ostream& os) const 
					{
						return	os << (*this);
					};
					void descripteur_debug(const std::string& descripteur)
					{
						_type._descripteur_debug=descripteur;
					}
					int descripteur_lecture() const
					{
						return _type.descripteur_lecture();
					};
					int descripteur_ecriture() const
					{
						return _type.descripteur_ecriture();
					};
					virtual std::string tampon_depile(const size_t taille)
					{
						return _lecture.depile(_type,taille);
					};
					virtual void tampon_empile(const std::string chaine)
					{
						_lecture.empile(_type,chaine);
					};
					bool pret_a_lire(const TypeLecture mode)
					{
						return _lecture.pret_a_lire(_type,mode,_Cherche::_cherchable);
					};
					void fermeture_lecture()
					{
						_FermetureLecture fermeture_lecture;
						fermeture_lecture(_type);
					};
					void fermeture_ecriture()
					{
						_FermetureEcriture fermeture_ecriture;
						fermeture_ecriture(_type);
					};
					bool media_actif() const
					{
						return _type.media_actif();
					}
					std::set<int> descripteurs_non_systeme() const
					{
						return _type.descripteurs_non_systeme();
					};
				protected:
					typedef Type _Type;
					typedef Ouverture _Ouverture;
					typedef Lecture _Lecture;
					typedef Ecriture _Ecriture;
					typedef Fermeture _Fermeture;
					typedef Attente _Attente;
					typedef Cherche _Cherche;
					typedef FermetureLecture _FermetureLecture;
					typedef FermetureEcriture _FermetureEcriture;
					Lecture _lecture;
					Type _type;
			};

			class FluxOperationInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					FluxOperationInvalide(const std::string& typeflux)
					:Execution(SIGILL,std::string("Invalid operation on stream ")+typeflux) { };
				
			};

			class OuvertureFluxImpossible : public SetLgg::Global::Exception::Execution
			{
				public:
					OuvertureFluxImpossible(const std::string& descripteur, const std::string& details = std::string())
					:Execution(SIGPIPE,std::string("Impossible to open stream ")+descripteur+details) { };
				
			};

			class LectureFluxImpossible : public SetLgg::Global::Exception::Execution
			{
				public:
					LectureFluxImpossible(const std::string& descripteur, const std::string& details = std::string())
					:Execution(SIGPIPE,std::string("Impossible to read from stream ")+descripteur+details) { };
				
			};

			class EcritureFluxImpossible : public SetLgg::Global::Exception::Execution
			{
				public:
					EcritureFluxImpossible(const std::string& descripteur, const std::string& details = std::string())
					:Execution(SIGPIPE,std::string("Impossible to write into stream ")+descripteur+details) { };
				
			};

			class ChercheFluxImpossible : public SetLgg::Global::Exception::Execution
			{
				public:
					ChercheFluxImpossible(const std::string& descripteur, const std::string& details = std::string())
					:Execution(SIGPIPE,std::string("Impossible to move cursor into stream ")+descripteur+details) { };
				
			};

			class FermetureFluxImpossible : public SetLgg::Global::Exception::Execution
			{
				public:
					FermetureFluxImpossible(const std::string& descripteur, const std::string& details = std::string())
					:Execution(SIGPIPE,std::string("Impossible to close stream ")+descripteur+details) { };
				
			};

			class AttenteFluxImpossible : public SetLgg::Global::Exception::Execution
			{
				public:
					AttenteFluxImpossible(const std::string& descripteur, const std::string& details = std::string())
					:Execution(SIGPIPE,std::string("Impossible to wait for connections on stream ")+descripteur+details) { };
				
			};

			class TypeLectureFluxImpossible : public SetLgg::Global::Exception::Execution
			{
				public:
					TypeLectureFluxImpossible(const std::string& type, const std::string& descripteur)
					:Execution(SIGILL,std::string("Read mode ")+type+" not available on stream "+descripteur) { };
				
			};



			enum class TypeOuvertureFichier { LECTURE, ECRITURE, LECTUREECRITURE, ECRITUREAJOUT };
			template<typename oStream>
				oStream& operator<<(oStream& os, const TypeOuvertureFichier& type)
				{
					switch(type)
					{
						case TypeOuvertureFichier::LECTURE:
							return os << "<";
						case TypeOuvertureFichier::ECRITURE:
							return os << ">";
						case TypeOuvertureFichier::LECTUREECRITURE:
							return os << "< >";
						case TypeOuvertureFichier::ECRITUREAJOUT:
							return os << "> >";
					}
					return os;
				}

			class OperationNonSupportee;
			class OuvertureFichier;
			class Fermeture;
			class EcritureSimpleFichier;
			class LectureSimple;
			class LectureSimpleFichier;
			class LectureSimpleSocket;
			class LectureAvecExpediteur;
			class Cherche;
			template<TypeOuvertureFichier type>
			class Fichier
			{
				friend class OperationNonSupportee;
				friend class Stdin;
				friend class Stdout;
				friend class Stderr;
				friend class OuvertureFichier;
				friend class Fermeture;
				friend class EcritureSimpleFichier;
				friend class LectureSimple;
				friend class LectureSimpleFichier;
				friend class LectureSimpleSocket;
				friend class LectureAvecExpediteur;
				friend class Cherche;
				template<typename Type, typename Ouverture, typename Lecture, typename Ecriture, typename Fermeture, typename Attente, typename Cherche, typename FermetureLecture, typename FermetureEcriture>
					friend class FluxGenerique;
				public:
					explicit Fichier(const std::string& nom)
					:_type(type),_nom(nom),_descripteur_fichier(-1) {};
					Fichier(const int fd, const std::string& nom)
					:_type(type),_nom(nom),_descripteur_fichier(fd) {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const Fichier& fichier)
					{
						os << "File " << fichier._type << " " << (fichier._nom) << " FD: " << fichier._descripteur_fichier;
						return os;
					};
					int descripteur_lecture() const
					{
						return _descripteur_fichier;
					};
					int descripteur_ecriture() const
					{
						return _descripteur_fichier;
					};
					std::set<int> descripteurs_non_systeme() const
					{
						return { _descripteur_fichier };
					}
					bool media_actif() const
					{
						return false;
					}
				private:
					TypeOuvertureFichier _type;
					std::string _nom;
					int mode_ouverture() const
					{
						switch(_type)
						{
							case TypeOuvertureFichier::LECTURE:
								return O_RDONLY;
							case TypeOuvertureFichier::ECRITURE:
								return O_WRONLY|O_CREAT|O_TRUNC;
							case TypeOuvertureFichier::ECRITUREAJOUT:
								return O_WRONLY|O_CREAT|O_APPEND;
							case TypeOuvertureFichier::LECTUREECRITURE:
								return O_RDWR|O_CREAT;
						}
						return 0;
					}
					int _descripteur_fichier;
					std::string _descripteur_debug;
			};

			namespace Outil
			{
				struct OuvertureSocket;
			}
			class OuvertureSocketEcoute;
			class OuvertureSocketConnection;
			class OuvertureSocketUDP;
			class Attente;
			class EcritureSimpleSocket;
			class EcritureAvecDestinataire;
			class Socket
			{
				friend struct Outil::OuvertureSocket;
				friend class OperationNonSupportee;
				friend class Fermeture;
				friend class OuvertureSocketEcoute;
				friend class OuvertureSocketConnection;
				friend class OuvertureSocketUDP;
				friend class Attente;
				friend class EcritureSimpleSocket;
				friend class EcritureAvecDestinataire;
				friend class LectureSimple;
				friend class LectureSimpleFichier;
				friend class LectureSimpleSocket;
				friend class LectureAvecExpediteur;
				friend class Cherche;
				template<typename Type, typename Ouverture, typename Lecture, typename Ecriture, typename Fermeture, typename Attente, typename Cherche, typename FermetureLecture, typename FermetureEcriture>
					friend class FluxGenerique;
				public:
					virtual ~Socket() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const Socket& socket)
					{
						return socket.format(os);
					};

					virtual std::ostream& format(std::ostream& os) const
					{
						return os << "Local: " << (_ip_locale) << " : " << (_port_local) << " Distant: " << (_ip_distante) << " : " << (_port_distant) << " FD: " << _descripteur_fichier;
					}
					virtual int type_socket() const
					{
						return 0;
					};
					virtual int protocole_socket() const
					{
						return 0;
					};
					int descripteur_lecture() const
					{
						return _descripteur_fichier;
					};
					int descripteur_ecriture() const
					{
						return _descripteur_fichier;
					};
					bool media_actif() const
					{
						return true;
					}
					std::set<int> descripteurs_non_systeme() const
					{
						return { _descripteur_fichier };
					}
				protected:
					Socket(const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:_ip_locale(ip_locale), _port_local(port_local), _ip_distante(ip_distante), _port_distant(port_distant),_descripteur_fichier(-1) {};
					Socket(const int descripteur)
					:_descripteur_fichier(descripteur) {};
					Socket(const int descripteur, const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:_ip_locale(ip_locale), _port_local(port_local), _ip_distante(ip_distante), _port_distant(port_distant),_descripteur_fichier(descripteur) {};
					Socket(const int descripteur, const std::string& ip_distante, const std::string& port_distant)
					:_ip_distante(ip_distante), _port_distant(port_distant),_descripteur_fichier(descripteur) {};
				private:
					std::string _ip_locale;
					std::string _port_local;
					std::string _ip_distante;
					std::string _port_distant;
					int _descripteur_fichier;
					std::string _descripteur_debug;
					static bool resolution_ip_port(const struct sockaddr* sock, const socklen_t longueur, std::string& ip, std::string& port);
					static void log_erreur(struct addrinfo *res, std::ostringstream& os, const std::string& fonction, int erreur);
			};
			
			enum class TypeSocketTCP {SERVEUR_ACCEPTE,SERVEUR_CONNECTION,CLIENT};

			template <TypeSocketTCP type>
			class SocketTCP : public Socket
			{
				public:
					SocketTCP(const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Socket(ip_locale,port_local,ip_distante,port_distant), _type(type) {};
					SocketTCP(const int descripteur)
					:Socket(descripteur), _type(type) {};
					SocketTCP(const int descripteur, const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Socket(descripteur,ip_locale,port_local,ip_distante,port_distant), _type(type) {};
					SocketTCP(const int descripteur, const std::string& ip_distante, const std::string& port_distant)
					:Socket(descripteur,ip_distante,port_distant), _type(type) {};
					virtual ~SocketTCP() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const SocketTCP& socketTCP)
					{
						os << "Socket (TCP/" ;
						switch(socketTCP._type)
						{
							case TypeSocketTCP::SERVEUR_ACCEPTE:
								os << "Server-Acceptor";
								break;
							case TypeSocketTCP::SERVEUR_CONNECTION:
								os << "Server-Handler";
								break;
							case TypeSocketTCP::CLIENT:
								os << "Client";
								break;
						}
						os << ") " << static_cast<Socket>(socketTCP); 
						return os;
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
					virtual int type_socket() const
					{
						return SOCK_STREAM;
					};
				private:
					TypeSocketTCP _type;
					
			};

			class SocketUDP : public Socket
			{
				public:
					SocketUDP(const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Socket(ip_locale,port_local,ip_distante,port_distant) {};
					SocketUDP(const int descripteur, const std::string& ip_locale, const std::string& port_local, const std::string& ip_distante, const std::string& port_distant)
					:Socket(descripteur,ip_locale,port_local,ip_distante,port_distant) {};
					virtual ~SocketUDP() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const SocketUDP& socketUDP)
					{
						os << "Socket (UDP) " << static_cast<Socket>(socketUDP); 
						return os;
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
					virtual int type_socket() const
					{
						return SOCK_DGRAM;
					};
			};

			class LectureSimple;
			class LectureSimplePaireTubesAnonymes;
			class EcritureSimplePaireTubesAnonymes;
			class FermeturePaireTubesAnonymes;
			class FermeturePaireTubesAnonymesLecture;
			class FermeturePaireTubesAnonymesEcriture;
			class FluxPaireTubesAnonymes;
			class PaireTubesAnonymes
			{
				friend class OperationNonSupportee;
				friend class LectureSimple;
				friend class LectureSimplePaireTubesAnonymes;
				friend class EcritureSimplePaireTubesAnonymes;
				friend class FermeturePaireTubesAnonymes;
				friend class FermeturePaireTubesAnonymesLecture;
				friend class FermeturePaireTubesAnonymesEcriture;
				friend class FluxPaireTubesAnonymes;
				template<typename Type, typename Ouverture, typename Lecture, typename Ecriture, typename Fermeture, typename Attente, typename Cherche, typename FermetureLecture, typename FermetureEcriture>
					friend class FluxGenerique;
				public:
					PaireTubesAnonymes(const int lecture, const int ecriture)
					:_descripteur_lecture_depuislavm(lecture), _descripteur_ecriture_depuislavm(ecriture), _lecture_possible(true), _ecriture_possible(true)  {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const PaireTubesAnonymes& tubes)
					{
						os << "Pipe";
						if(tubes._lecture_possible)
						{
							os << " < " << tubes._descripteur_lecture_depuislavm;
						}
						if(tubes._lecture_possible and tubes._ecriture_possible)
						{
							os << " ;";
						}
						if(tubes._ecriture_possible)
						{
							os << " > " << tubes._descripteur_ecriture_depuislavm;
						}
						return os;
					};
					int descripteur_lecture() const
					{
						return _descripteur_lecture_depuislavm;
					};
					int descripteur_ecriture() const
					{
						return _descripteur_ecriture_depuislavm;
					};
					bool media_actif() const
					{
						return true;
					}
					std::set<int> descripteurs_non_systeme() const
					{
						return { _descripteur_lecture_depuislavm, _descripteur_ecriture_depuislavm };
					}
				private:
					int _descripteur_lecture_depuislavm;
					int _descripteur_ecriture_depuislavm;
					std::string _descripteur_debug;
					bool _lecture_possible;
					bool _ecriture_possible;
					
			};

			class OuvertureFichier
			{
				public:
					template<TypeOuvertureFichier type>
					void operator()(Fichier<type>& fichier)
					{
						fichier._descripteur_fichier = OuvertureFichier::ouvrir(fichier._nom,fichier.mode_ouverture(),0,fichier._descripteur_debug);
					};
					static int ouvrir(const std::string& f,const int flags,const int flags_statiques,const std::string& d);
			};


			class OuvertureSocketEcoute
			{
				public:
					void operator()(SocketTCP<TypeSocketTCP::SERVEUR_ACCEPTE>& socket)
					{
						socket._descripteur_fichier = OuvertureSocketEcoute::ouvrir(socket.type_socket(),socket.protocole_socket(),socket._ip_locale,socket._port_local,socket._descripteur_debug);
					};
					static int ouvrir(const int type, const int protocole, const std::string& ip_locale, const std::string& port_local, const std::string& d);
			};

			class OuvertureSocketConnection
			{
				public:
					void operator()(SocketTCP<TypeSocketTCP::CLIENT>& socket)
					{
						socket._descripteur_fichier = OuvertureSocketConnection::ouvrir(socket.type_socket(),socket.protocole_socket(),socket._ip_distante,socket._port_distant,socket._descripteur_debug);
					};
					static int ouvrir(const int type, const int protocole, const std::string& ip_distante, const std::string& port_distant, const std::string& d);
			};

			class OuvertureSocketUDP
			{
				public:
					void operator()(SocketUDP& socket)
					{
						socket._descripteur_fichier = OuvertureSocketUDP::ouvrir(socket.type_socket(),socket.protocole_socket(),socket._ip_locale,socket._port_local,socket._descripteur_debug);
					};
					static int ouvrir(const int type, const int protocole, const std::string& ip_locale, const std::string& port_local, const std::string& d);
			};

			class LectureSimple
			{
				public:
					bool buffer_non_vide() const
					{
						return not _buffer.good() or not _buffer.str().empty();
					};
					template<typename Flux>
					bool pret_a_lire(Flux& flux, const TypeLecture mode, const bool cherchable)
					{
						return pret_a_lire(flux.descripteur_lecture(),mode,cherchable,flux._descripteur_debug);
					}
					template<typename Flux>
					std::string depile(Flux& flux, const size_t taille)
					{
						return Outils::depile_tampon(_buffer,taille);
					};
					template<typename Flux>
					void empile(Flux& flux, const std::string& chaine)
					{
						Outils::empile_tampon(_buffer,chaine);
					}
				protected:
					DonneesLues lire(const int desc, const TypeLecture type, const bool cherchable, const std::string& d);
					bool pret_a_lire(const int desc, const TypeLecture type, const bool cherchable, const std::string& d);
				private:
					bool lecture_non_bloquante(const int desc, const std::string& d);
					std::stringstream _buffer;
			};

			class LectureSimpleFichier : public LectureSimple
			{
				public:
					template<typename Flux>
					DonneesLues operator()(Flux& flux, const TypeLecture& type, const bool cherchable)
					{
						return lire(flux._descripteur_fichier,type,cherchable,flux._descripteur_debug);
					};
					static const bool _lisible = true;
			};

			class LectureSimpleSocket : public LectureSimple
			{
				public:
					template<typename Flux>
					DonneesLues operator()(Flux& flux, const TypeLecture& type, const bool cherchable)
					{
						return lire(flux._descripteur_fichier,type,cherchable,flux._descripteur_debug);
					};
					static const bool _lisible = true;
			};

			class LectureSimplePaireTubesAnonymes : public LectureSimple
			{
				public:
					template<typename Flux>
					DonneesLues operator()(Flux& flux, const TypeLecture& type, const bool cherchable)
					{
						return lire(flux._descripteur_lecture_depuislavm,type,cherchable,flux._descripteur_debug);
					};
					static const bool _lisible = true;
			};

			class LectureAvecExpediteur
			{
				public:
					template<typename Flux>
					DonneesLues operator()(Flux& flux, const TypeLecture& type, const bool cherchable)
					{
						return lire(flux._descripteur_fichier,type,flux._ip_distante,flux._port_distant,cherchable,flux._descripteur_debug);
					};
					DonneesLues lire(const int desc, const TypeLecture type, const std::string& ip_distante, const std::string& port_distant, const bool cherchable, const std::string& d);
					/*static bool expediteur_accepte(struct sockaddr expediteur,size_t taille_expediteur,struct sockaddr masque,size_t taille_masque);*/
					bool buffer_non_vide() const
					{
						return not _buffer.good() or not _buffer.str().empty();
					};
					template<typename Flux>
					bool pret_a_lire(Flux& flux, const TypeLecture mode, const bool cherchable)
					{
						return pret_a_lire(flux.descripteur_lecture(),mode,cherchable,flux._descripteur_debug);
					};
					template<typename Flux>
					std::string depile(Flux& flux, const size_t taille)
					{
						return Outils::depile_tampon(_buffer,taille);
					};
					template<typename Flux>
					void empile(Flux& flux, const std::string& chaine)
					{
						Outils::empile_tampon(_buffer,chaine);
					}
					static const bool _lisible = true;
				private:
					bool pret_a_lire(const int desc, const TypeLecture type, const bool cherchable, const std::string& d) { return true; }
					std::stringstream _buffer;
			};

			class EcritureSimpleFichier
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux,const std::string& valeur)
					{
						EcritureSimpleFichier::ecrire(flux._descripteur_fichier,valeur,flux._descripteur_debug);
					};
					static void ecrire(const int desc, const std::string& valeur, const std::string& d);
					static const bool _inscriptible = true;
			};

			class EcritureSimpleSocket
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux,const std::string& valeur)
					{
						EcritureSimpleSocket::ecrire(flux._descripteur_fichier,valeur,flux._descripteur_debug);
					};
					static void ecrire(const int desc, const std::string& valeur, const std::string& d);
					static const bool _inscriptible = true;
			};

			class EcritureSimplePaireTubesAnonymes : public EcritureSimpleFichier
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux,const std::string& valeur)
					{
						EcritureSimpleFichier::ecrire(flux._descripteur_ecriture_depuislavm,valeur,flux._descripteur_debug);
					};
					static const bool _inscriptible = true;
			};

			class EcritureAvecDestinataire
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux,const std::string& valeur)
					{
						EcritureAvecDestinataire::ecrire(flux._descripteur_fichier,flux._ip_distante,flux._port_distant,valeur,flux._descripteur_debug);
					};
					static void ecrire(const int desc, const std::string& ip_distante, const std::string& port_distant, const std::string& valeur, const std::string& d);
					static const bool _inscriptible = true;
			};

			class Cherche
			{
				public:
					template<typename Flux>
					long int operator()(Flux& flux, TypeCherche type, const long int decalage)
					{
						return Cherche::cherche(flux._descripteur_fichier,type,decalage,flux._descripteur_debug);
					};
					static long int cherche(const int desc, TypeCherche type, const long int decalage, const std::string& d);
					static const bool _cherchable = true;
			};

			class Fermeture
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux)
					{
						Fermeture::fermer(flux._descripteur_fichier,flux._descripteur_debug);
					};
					static void fermer(const int desc, const std::string& d);
			};
			
			class FermeturePaireTubesAnonymes
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux)
					{
						if(flux._lecture_possible)
						{
							Fermeture::fermer(flux._descripteur_lecture_depuislavm,flux._descripteur_debug);
							flux._lecture_possible=false;
						}
						if(flux._ecriture_possible)
						{
							Fermeture::fermer(flux._descripteur_ecriture_depuislavm,flux._descripteur_debug);
							flux._ecriture_possible=false;
						}
					};
					static void fermer(const int desc, const std::string& d);
			};
			
			class FermeturePaireTubesAnonymesLecture
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux)
					{
						if(flux._lecture_possible)
						{
							Fermeture::fermer(flux._descripteur_lecture_depuislavm,flux._descripteur_debug);
							flux._lecture_possible=false;
						}
					};
					static void fermer(const int desc, const std::string& d);
			};
			
			class FermeturePaireTubesAnonymesEcriture
			{
				public:
					template<typename Flux>
					void operator()(Flux& flux)
					{
						if(flux._ecriture_possible)
						{
							Fermeture::fermer(flux._descripteur_ecriture_depuislavm,flux._descripteur_debug);
							flux._ecriture_possible=false;
						}
					};
					static void fermer(const int desc, const std::string& d);
			};
			
			class Attente
			{
				public:
					template<typename Flux>
					FluxSP operator()(Flux& flux,const std::string& connecteur)
					{
						return Attente::attendre(flux._descripteur_fichier,connecteur,flux._descripteur_debug);
					};
					static FluxSP attendre(const int desc, const std::string& connecteur, const std::string& d);
			};

	
			class OperationNonSupportee
			{
				public:
				template<typename FluxInterdit>
				FluxSP operator()(const FluxInterdit& flux) const
				{
					throw FluxOperationInvalide(flux._descripteur_debug);
					return FluxSP();
				};
				template<typename FluxInterdit>
				long int operator()(const FluxInterdit& flux, const TypeCherche type, const long int decalage) const
				{
					throw FluxOperationInvalide(flux._descripteur_debug);
					return -1;
				};
				template<typename FluxInterdit>
				FluxSP operator()(const FluxInterdit& flux, const std::string& valeur) const
				{
					throw FluxOperationInvalide(flux._descripteur_debug);
					return FluxSP();
				};
				template<typename FluxInterdit>
				DonneesLues operator()(const FluxInterdit& flux, const TypeLecture& type, const bool cherchable) const
				{
					throw FluxOperationInvalide(flux._descripteur_debug);
					return DonneesLues();
				};
				template<typename FluxInterdit>
				std::string depile(const FluxInterdit& flux, const size_t taille) const
				{
					throw FluxOperationInvalide(flux._descripteur_debug);
					return "";
				};
				template<typename FluxInterdit>
				void empile(const FluxInterdit& flux, const std::string& chaine) const
				{
					throw FluxOperationInvalide(flux._descripteur_debug);
				};
				bool buffer_non_vide() const
				{
					return false;
				};
				template<typename FluxInterdit>
				bool pret_a_lire(const FluxInterdit& flux, const TypeLecture& type, const bool cherchable)
				{
					throw FluxOperationInvalide(flux._descripteur_debug);
					return true;
				};
				static const bool _lisible = false;
				static const bool _inscriptible = false;
				static const bool _cherchable = false;
			};
		}
	}
}
#endif
