/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_FLUX_GESTIONNAIREFLUX_H_
#define _MACHINE_FLUX_GESTIONNAIREFLUX_H_

#include <map>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Flux { DECL_SHARED_PTR(GestionnaireFlux); } } }

#include <src/machine/memoire/donnees/valeur.h>
#include <src/machine/flux/flux.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Flux
		{
			class GestionnaireFlux
			{
				public:
					GestionnaireFlux();
					SetLgg::Machine::Flux::FluxSP operator[](const SetLgg::Machine::Memoire::NomFlux& nom)
					{
						auto it = _flux.find(nom);
						if(it==_flux.end())
							return SetLgg::Machine::Flux::FluxSP();
						return _flux[nom];
					};
					void ouverture(const SetLgg::Machine::Memoire::NomFlux& descripteur, const FluxSP& flux);
					void fermeture(const SetLgg::Machine::Memoire::NomFlux& descripteur);
					const DonneesLues lecture(const SetLgg::Machine::Memoire::NomFlux& descripteur, const TypeLecture& type);
					void ecriture(const SetLgg::Machine::Memoire::NomFlux& descripteur, const std::string& valeur);
					long int cherche(const SetLgg::Machine::Memoire::NomFlux& descripteur, const TypeCherche type, const long int decalage);
					void attente(const SetLgg::Machine::Memoire::NomFlux& descripteur, const SetLgg::Machine::Memoire::NomFlux& connecteur);
					bool lisible(const SetLgg::Machine::Memoire::NomFlux& descripteur);
					bool inscriptible(const SetLgg::Machine::Memoire::NomFlux& descripteur);
					bool cherchable(const SetLgg::Machine::Memoire::NomFlux& descripteur);
					bool ajoute(const SetLgg::Machine::Memoire::NomFlux& descripteur,const FluxSP& flux);
					void fermeture_lecture(const SetLgg::Machine::Memoire::NomFlux& descripteur);
					void fermeture_ecriture(const SetLgg::Machine::Memoire::NomFlux& descripteur);
					std::set<int> descripteurs_non_systeme() const;
					template<typename oStream>
					friend oStream& operator<<(oStream&,const GestionnaireFlux&);
				private:
					std::map<SetLgg::Machine::Memoire::NomFlux,FluxSP> _flux;
			};

			template<typename oStream>
			oStream& operator<<(oStream& os, const GestionnaireFlux& gestionnaireflux)
			{
				os << "# Streams handler:" << std::endl;
				for(auto it = gestionnaireflux._flux.cbegin() ; it!=gestionnaireflux._flux.cend() ; ++it)
				{
					os << "# Stream " << it->first << ": " << *(it->second) << std::endl;
				}
				return os << "# End of streams handler" << std::endl;
			}

                        class FluxDejaOuvert : public SetLgg::Global::Exception::Execution
                        {
                                public:
                                        FluxDejaOuvert(const std::string& descripteur)
                                        :Execution(SIGABRT,descripteur+" is already open"){};
                        };

                        class FluxInexistant : public SetLgg::Global::Exception::Execution
                        {
                                public:
                                        FluxInexistant(const std::string& descripteur)
                                        :Execution(SIGABRT,descripteur+" stream descriptor does not exist"){};
                        };
		}
	}
}

#endif
