/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <fstream>
#include <sys/types.h>
#include <unistd.h>
#include <src/machine/machine/machine.h>
#include <src/machine/programme/analyseur/analyseur.h>
#include <src/machine/memoire/analyseur/analyseur.h>
#include <src/machine/debogueur/donnees/debogueur.h>
using namespace SetLgg::Machine::Machine;

Machine::Machine(const std::string& fichier_programme, const std::string& fichier_memoire, const std::string& fichier_coredump, const std::vector<std::string>& liste_plugins, const TypeExecution debogage, const std::string& commandes_debogueur)
:_processeur(new SetLgg::Machine::Processeur::Processeur()),_flux(new SetLgg::Machine::Flux::GestionnaireFlux()),_plugins(new SetLgg::Machine::Plugin::GestionnairePlugin(liste_plugins)), _fichier_programme(fichier_programme),_fichier_memoire(fichier_memoire), _fichier_coredump(fichier_coredump)
{
	if(debogage==TypeExecution::DEBUG)
	{
		_debogueur = SetLgg::Machine::Debogueur::DebogueurSP(new SetLgg::Machine::Debogueur::Debogueur(commandes_debogueur));
	}
	Processeur::Processeur::enregistrement_signaux(_processeur);
	_programme = SetLgg::Machine::Programme::Analyseur::Analyseur::analyse_fichier(fichier_programme,_plugins,_debogueur);
	_programme->edition_liens();
	_memoire = SetLgg::Machine::Memoire::Analyseur::Analyseur::analyse_fichier(fichier_memoire,_plugins);
}

size_t Machine::execution(MachineSP& machine)
{
	try
	{
		if(machine->_debogueur)
		{
			if(SetLgg::Machine::Debogueur::Debogueur::execution(machine))
			{
				SetLgg::Machine::Processeur::Processeur::execution(machine);
				throw; // on ne devrait jamais arriver ici
			}
			return 0;
		}
		else
		{
			SetLgg::Machine::Processeur::Processeur::execution(machine);
			throw; // on ne devrait jamais arriver ici
		}
	}
	catch(CodeRetour code_retour)
	{
		return code_retour;
	}
}

void Machine::arret(bool coredump_machine_virtuelle, bool ecriture_memoire, size_t code_retour, const std::string message_erreur)
{
	if(coredump_machine_virtuelle and not _fichier_coredump.empty())
	{
		{
			pid_t pid = ::getpid();
			std::ostringstream oss;
			oss << _fichier_coredump << "_" << pid;
			_fichier_coredump = oss.str();
		}
		std::cerr << "Dumping virtual machine into file: " << _fichier_coredump << std::endl;
		std::ofstream fichier_coredump(_fichier_coredump);
		fichier_coredump << (*this);
		if(not message_erreur.empty())
		{
			fichier_coredump << "#" << std::endl << "#" << std::endl << "#" << std::endl << "# Error message:" << std::endl;
			std::istringstream iss(message_erreur);
			while(iss.good())
			{
				std::string str;
				getline(iss,str);
				fichier_coredump << "# " << str << std::endl;
			}
			fichier_coredump << "# End of error message" << std::endl;
		}
		fichier_coredump.close();
		std::cerr << "Core dump generated and saved." << std::endl;
	}
	if(ecriture_memoire and not coredump_machine_virtuelle)
	{
		Machine::ecriture_memoire();
	}
	throw CodeRetour(code_retour);
}

void Machine::ecriture_memoire() const
{
	std::ofstream fichier_memoire(_fichier_memoire);
	fichier_memoire << *(_memoire);
	fichier_memoire.close();
}
