/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <map>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/memoire/donnees/booleen.h>
using namespace SetLgg::Machine::Memoire;

#define COMPILATIONS_CACHE_STATISTIQUES 10000
#define COMPILATIONS_CACHE_TAILLE_NETTOYAGE 3000
namespace SetLgg
{
	namespace Machine
	{
		namespace Memoire
		{
			namespace Outils
			{
				// Optimisation:
				// La compilation des expressions rationnelles met des plombes
				// Je les mets donc en cache, ce qui augmente la memoire de la VM
				// mais evite les recompilations.
				// Quand la taille du cache depasse COMPILATIONS_CACHE_TAILLE_NETTOYAGE elements,
				// une passe de COMPILATIONS_CACHE_STATISTIQUES est effectuee pour determiner les
				// regex les moins utilisees. Toutes les regex qui sont utilisees moins de
				// 2/3*(COMPILATIONS_CACHE_STATISTIQUES/(taille_cache+1)) sont eliminees du cache.
				struct RegexCompilee
				{
					RegexCompilee(const ::regex_t& regex)
					:_regex(regex), _compte(0) { }
					~RegexCompilee()
					{
						::regfree(&_regex);
					}
					::regex_t _regex;
					size_t _compte;
				};

				class RegexCache
				{
					public:
						static RegexCache& instance()
						{
							static RegexCache cache;
							return cache;
						}
						::regex_t *compile_interne(const std::string& motif)
						{
							if(_cache.size()>COMPILATIONS_CACHE_TAILLE_NETTOYAGE)
							{
								++_compte;
								if(_compte==COMPILATIONS_CACHE_STATISTIQUES)
								{
									const size_t niveau=2*(COMPILATIONS_CACHE_STATISTIQUES/(_cache.size()+1))/3;
									std::vector<std::map<std::string,RegexCompilee>::iterator> elements_a_supprimer;
									for(auto it=_cache.begin() ; it!=_cache.end() ; ++it)
									{
										if(it->second._compte>niveau)
										{
											it->second._compte=0;
										}
										else
										{
											elements_a_supprimer.push_back(it);
										}
									}
									for(auto it=elements_a_supprimer.cbegin() ; it!=elements_a_supprimer.cend() ; ++it)
									{
										_cache.erase(*it);
									}
									_compte=0;
								}
							}
							auto itcache = _cache.find(motif);
							if(itcache!=_cache.end())
							{
								if(_cache.size()>COMPILATIONS_CACHE_TAILLE_NETTOYAGE)
								{
									++itcache->second._compte;
								}
								return &(itcache->second._regex);
							}
							::regex_t re;
							int erreur = ::regcomp(&re,motif.c_str(),REG_EXTENDED);
							if(erreur!=0)
							{
								char buffer[1024];
								::regerror(erreur,&re,buffer,1023);
								throw std::string(buffer);
							}
							return &(_cache.insert(std::make_pair(motif,re)).first->second._regex);
						}
						::regex_t *compile(const std::string& motif)
						{
							try
							{
								return compile_interne(motif);
							}
							catch(std::string& erreur)
							{
								std::ostringstream oss;
								oss << SetLgg::Global::Exception::details << std::endl << "regcomp: " << erreur;
								throw MotifInvalide(motif,oss.str());
							}
						}
					private:
						RegexCache()
						:_compte(0) {};
						size_t _compte;
						std::map<std::string,RegexCompilee> _cache;
				};
			}
		}
	}
}

Chaine Chaine::operator/(const Chaine& motif) const
{
	std::string resultat;
	::regex_t *re = Outils::RegexCache::instance().compile(motif._chaine);
	::regmatch_t match;
	if(::regexec(re,_chaine.c_str(),1,&match,0)==0)
	{
		resultat = std::string(_chaine,match.rm_so,match.rm_eo-match.rm_so);
	}
	return Chaine(resultat);
}

Chaine Chaine::operator%(const Chaine& motif) const
{
	std::string resultat = _chaine;
	::regex_t *re = Outils::RegexCache::instance().compile(motif._chaine);
	::regmatch_t match;
	if(::regexec(re,_chaine.c_str(),1,&match,0)==0)
	{
		resultat = std::string(_chaine,match.rm_eo);
	}
	return Chaine(resultat);
}

Booleen Chaine::operator==(const Chaine& chaine) const
{
	return Booleen(_chaine == chaine._chaine);
}

Booleen Chaine::operator!=(const Chaine& chaine) const
{
	return Booleen(_chaine != chaine._chaine);
}

Booleen Chaine::operator<(const Chaine& chaine) const
{
	return Booleen(_chaine < chaine._chaine);
}

Booleen Chaine::operator<=(const Chaine& chaine) const
{
	return Booleen(_chaine <= chaine._chaine);
}

Booleen Chaine::operator>(const Chaine& chaine) const
{
	return Booleen(_chaine > chaine._chaine);
}

Booleen Chaine::operator>=(const Chaine& chaine) const
{
	return Booleen(_chaine >= chaine._chaine);
}

Entier Chaine::operator-(const Entier& indice) const
{
	if((indice<0) or (static_cast<size_t>(indice)>=_chaine.size()))
	{
		std::ostringstream oss;
		Entier::_Entier i = indice;
		oss << i;
		throw IndiceInvalide(oss.str());
	}
	return Entier(_chaine[indice]);
}

::regex_t *Chaine::cache_regex(const std::string& motif)
{
	return Outils::RegexCache::instance().compile_interne(motif);
}
