/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_MEMOIRE_CHAINE_H_
#define _MACHINE_MEMOIRE_CHAINE_H_

#include <string.h>
#include <stdlib.h>
#include <algorithm>
#include <regex.h> // regex non supporte in g++4.4
#include <src/global/global.h>

namespace SetLgg { namespace Machine { namespace Memoire { DECL_SHARED_PTR(Chaine); } } }

#include <src/machine/memoire/donnees/entier.h>
#include <src/machine/programme/donnees/valeur.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Memoire
		{
			class Booleen;
			class Chaine : public SetLgg::Machine::Programme::Valeur
			{
				public:
					typedef std::string _Chaine;
					Chaine(const std::string& chaine)
					:_chaine(chaine){ };
					struct Desechappe {};
					Chaine(const std::string& chaine, Desechappe)
					:_chaine(SetLgg::Global::Chaine::desechappe(chaine)){ };
					virtual ~Chaine() {};
					operator const std::string () const
					{
						return _chaine;
					};
					const Entier taille() const
					{
						return Entier(_chaine.size());
					};
					const Chaine operator+(const Chaine& chaine) const
					{
						return Chaine(_chaine+chaine._chaine);
					};
					Chaine operator/(const Chaine& motif) const;
					Chaine operator%(const Chaine& motif) const;
					Entier operator-(const Entier& indice) const;

					template<typename oStream>
					friend oStream& operator<<(oStream&,const Chaine&);
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
					virtual operator const Type&() const
					{
						return SetLgg::Machine::Memoire::Type::chaine();
					};
					virtual const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						return this->shared_from_this();
					};
					Booleen operator==(const Chaine& chaine) const;
					Booleen operator!=(const Chaine& chaine) const;
					Booleen operator<(const Chaine& chaine) const;
					Booleen operator<=(const Chaine& chaine) const;
					Booleen operator>(const Chaine& chaine) const;
					Booleen operator>=(const Chaine& chaine) const;
					virtual operator SetLgg_Machine_Value() const
					{
						SetLgg_Machine_Value value;
						value._type._type = SetLgg_Machine_ValueTypeCategory::VALUE_STRING;
						value._type._user_name = {0, 0};
						value._initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
						value._value._string = {SetLgg::Global::Chaine::duplique(_chaine.c_str(),_chaine.size()+1), _chaine.size()};
						return value;
					};
					virtual ChaineCSP en_chaine(const std::string& format) const
					{
						ChaineCSP retour(new const Chaine(*this));
						return retour;
					};
					static ::regex_t *cache_regex(const std::string& motif);
					// Optimisation rendue caduque par l'implementation de ::malloc
					//void *operator new(size_t taille)
					//{
					//	return SetLgg::Global::ListeMemoire<Chaine>::instance().alloue(taille);
					//};
					//void operator delete(void *chaine)
					//{
					//	SetLgg::Global::ListeMemoire<Chaine>::instance().libere(chaine);
					//};
				private:
					std::string _chaine;
					virtual ValeurSP copie() const
					{
						ValeurSP valeur(new Chaine(*this));
						return valeur;
					};
				protected:
					explicit Chaine() {};
					
			};
			template<typename oStream>
			oStream& operator<<(oStream& os,const Chaine& chaine)
			{
				os << "\"";
				os << SetLgg::Global::Chaine::echappe(chaine._chaine);
				return os << "\"";
			};

			class ChaineNulle : public Chaine
			{
				public:
					ChaineNulle() {};
					virtual bool est_nulle() const { return true; };
			};

			class MotifInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					MotifInvalide(const std::string& motif, const std::string& details)
					:Execution(SIGABRT,std::string("Invalid pattern ")+motif+details) {};
			};

			class IndiceInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					IndiceInvalide(const std::string& indice)
					:Execution(SIGABRT,std::string("Invalid string index ")+indice) {};
			};
		}
	}
}
#endif
