/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_MEMOIRE_ENTIER_H_
#define _MACHINE_MEMOIRE_ENTIER_H_
#include <sstream>
#include <type_traits>

#include <src/global/global.h>
#include <src/global/exceptions.h>

namespace SetLgg { namespace Machine { namespace Memoire { DECL_SHARED_PTR(Entier); } } }

#include <src/machine/memoire/donnees/valeur.h>
#include <src/machine/programme/donnees/valeur.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Memoire
		{
			class Booleen;
			class Adresse;
			class Entier : public SetLgg::Machine::Programme::Valeur
			{
				public:
					typedef long int _Entier;
					explicit Entier(const long int& entier)
					:_entier(entier) {};
					virtual ~Entier() {};
					virtual operator const Adresse () const;
					operator const long int () const
					{
						return _entier;
					};
					template<typename oStream>
					friend oStream& operator<<(oStream&,const Entier&);
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
					virtual operator const Type&() const
					{
						return SetLgg::Machine::Memoire::Type::entier();
					};
					virtual const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						return this->shared_from_this();
					};
					Entier operator+(const Entier& entier) const
					{
						return Entier(_entier+entier._entier);
					};
					Entier operator-(const Entier& entier) const
					{
						return Entier(_entier-entier._entier);
					};
					Entier operator*(const Entier& entier) const
					{
						return Entier(_entier*entier._entier);
					};
					Entier operator/(const Entier& entier) const;
					Entier operator%(const Entier& entier) const;
					Booleen operator==(const Entier& entier) const;
					Booleen operator!=(const Entier& entier) const;
					Booleen operator<(const Entier& entier) const;
					Booleen operator<=(const Entier& entier) const;
					Booleen operator>(const Entier& entier) const;
					Booleen operator>=(const Entier& entier) const;
					virtual Valeur& operator+=(const long int variation)
					{
						_entier += variation;
						return *this;
					};
					virtual operator SetLgg_Machine_Value() const
					{
						SetLgg_Machine_Value value;
						value._type._type = SetLgg_Machine_ValueTypeCategory::VALUE_INTEGER;
						value._type._user_name = {0, 0};
						value._initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
						value._value._integer = _entier;
						return value;
					};
					virtual ChaineCSP en_chaine(const std::string& format) const;
					// Optimisation rendue caduque par l'implementation de ::malloc
					//void* operator new(size_t taille)
					//{
					//	return SetLgg::Global::ListeMemoire<Entier>::instance().alloue(taille);
					//};
					//void operator delete(void *entier)
					//{
					//	SetLgg::Global::ListeMemoire<Entier>::instance().libere(entier);
					//};
				private:
					long int _entier;
					virtual ValeurSP copie() const
					{
						ValeurSP valeur(new Entier(*this));
						return valeur;
					};
				protected:
					explicit Entier()
					:_entier(0) {};
			};
			template<typename oStream>
			oStream& operator<<(oStream& os,const Entier& entier)
			{
				return os << entier._entier;
			};

			class DivisionOuResteParZero : public SetLgg::Global::Exception::Execution
			{
				public:
					DivisionOuResteParZero(const std::string& operation)
					:Execution(SIGFPE,operation+" by zero") {};
			};


			class EntierNul : public Entier
			{
				public:
					EntierNul() {};
					virtual bool est_nulle() const { return true; };
			};
		}
	}
}
#endif
