/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <algorithm>
#include <src/machine/memoire/donnees/memoire.h>
using namespace SetLgg::Machine::Memoire;

void Memoire::ajout_element(const Adresse& adresse, const ListeAlias& aliases, const ElementSP& element, const SetLgg::Global::Source::PositionSP& position)
{
	Adresse adresse_max = _memoire_utilisee.size();
	if(adresse<adresse_max)
	{
		throw AdresseNonOrdonnee(adresse,position);
	}
	for(auto it = aliases.cbegin() ; it!=aliases.cend() ; ++it)
	{
		auto res = _aliases.insert(std::pair<Alias,Adresse>(*it,adresse));	
		if(not res.second)
		{
			throw AliasDejaExistant(*it,res.first->second,position);
		}
	}
	size_t offset = adresse-adresse_max;
	if(offset>0)
	{
		ajoute_bloc_libre(adresse_max,offset);
	}
	for(Adresse remplir=adresse_max ; remplir<adresse ; ++remplir)
	{
		_memoire_utilisee.push_back(ElementSP());
	}
	_memoire_utilisee.push_back(element);
}

Memoire::BlocMemoire Memoire::allocation(const ListePlacesMemoire& places_memoire)
{
	size_t taille_demandee = 0;
	for(auto place = places_memoire.cbegin() ; place!=places_memoire.cend() ; ++place)
	{
		taille_demandee += place->_repetition ;
	}

	auto bloc_optimal = _memoire_libre_inverse.lower_bound(taille_demandee);
	Adresse adresse;
	if(bloc_optimal == _memoire_libre_inverse.end())
	{
		adresse=Adresse(_memoire_utilisee.size());
		for(size_t indice=0 ; indice<taille_demandee ; ++indice)
		{
			_memoire_utilisee.push_back(ElementSP());
		}
	}
	else
	{
		adresse=bloc_optimal->second;
		size_t taille_dispo = bloc_optimal->first;
		enleve_bloc_libre(adresse);
		if(taille_dispo>taille_demandee)
		{
			ajoute_bloc_libre(adresse+taille_demandee,taille_dispo-taille_demandee);
		}
	}

	Adresse courante = adresse;
	for(auto it=places_memoire.cbegin() ; it!=places_memoire.cend() ; ++it)
	{
		if(it->_alias)
		{
			auto alias=_aliases.find(*(it->_alias));
			if(alias!=_aliases.end())
			{
				throw AliasDejaPresent(*(it->_alias),_aliases[*(it->_alias)]);
			}
			_aliases[*(it->_alias)]=courante;
		}
		for(size_t indice=0 ; indice<(it->_repetition) ; ++indice,++courante)
		{
			ElementSP element(new Element(it->_type));
			_memoire_utilisee[courante]=element;
		}
	}
	Memoire::BlocMemoire bloc_alloue(adresse,taille_demandee);
	return bloc_alloue;
}

Memoire::ListeBlocsMemoire Memoire::ListeBlocsMemoire::agregation(const Memoire::ListeBlocsMemoire& blocs_memoire)
{
	ListeBlocsMemoire blocs_memoire_agreges;

	if(not blocs_memoire)
		return blocs_memoire_agreges;

	BlocMemoire courant = *(blocs_memoire.cbegin());

	auto it=blocs_memoire.cbegin();
	for(++it ; it!=blocs_memoire.cend() ; ++it)
	{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "IT: " << it->_adresse << "," << it->_taille << std::endl;
#endif
		if(not courant.fusion(*it))
		{
			blocs_memoire_agreges += courant;
			courant=*it;
		}
	}
	blocs_memoire_agreges += courant;
	return blocs_memoire_agreges;
}


void Memoire::liberation(const ListeBlocsMemoire& blocs_memoire)
{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
	if(not blocs_memoire)
		return;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
	ListeBlocsMemoire blocs_memoire_agreges = Memoire::ListeBlocsMemoire::agregation(blocs_memoire);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif

	for(auto it=blocs_memoire_agreges.cbegin() ; it!=blocs_memoire_agreges.cend() ; ++it)
	{
		Adresse fin = it->_adresse+it->_taille;
		for(Adresse iit=it->_adresse ; iit!=fin ; ++iit)
		{
			if(not adresse_valide(iit))
			{
				throw LiberationAdresseInvalide(iit);
			}
		}
	}

#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
	for(auto it=blocs_memoire_agreges.cbegin() ; it!=blocs_memoire_agreges.cend() ; ++it)
	{
		Adresse fin = it->_adresse+it->_taille;
		for(Adresse iit=it->_adresse ; iit!=fin ; ++iit)
		{
			_memoire_utilisee[iit].reset();
		}
	}

#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
	std::vector<Alias> aliases;
	for(auto it=blocs_memoire_agreges.cbegin() ; it!=blocs_memoire_agreges.cend() ; ++it)
	{
		for(auto iit=_aliases.cbegin() ; iit!=_aliases.cend() ; ++iit)
		{
			if((it->_adresse<=iit->second) and (iit->second<(it->_adresse+it->_taille)))
			{
				aliases.push_back(iit->first);
			}
		}
	}
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
	for(auto it=aliases.cbegin() ; it!=aliases.cend() ; ++it)
	{
		_aliases.erase(*it);
	}
	for(auto it=blocs_memoire_agreges.cbegin() ; it!=blocs_memoire_agreges.cend() ; ++it)
	{
		Adresse adresse=it->_adresse;
		size_t taille=it->_taille;
		auto bloc_avant = _memoire_libre.lower_bound(adresse);
		auto bloc_apres = bloc_avant;
		AdresseSP adresse_avant;
		AdresseSP adresse_apres;
		if(bloc_avant!=_memoire_libre.begin())
		{
			--bloc_avant;
		}
		
		if((bloc_apres!=_memoire_libre.end()) and (bloc_apres->first==(adresse+taille)))
		{
			taille += bloc_apres->second;
			adresse_apres = AdresseSP(new Adresse(bloc_apres->first));
		}
		
		if((bloc_avant!=_memoire_libre.end()) and ((bloc_avant->first+bloc_avant->second)==adresse))
		{
			adresse_avant = AdresseSP(new Adresse(bloc_avant->first));
			adresse = bloc_avant->first;
			taille += bloc_avant->second;
		}
		if(adresse_avant)
		{
			enleve_bloc_libre(*adresse_avant);
		}
		if(adresse_apres)
		{
			enleve_bloc_libre(*adresse_apres);
		}
		ajoute_bloc_libre(adresse,taille);
	}
	// ameliorer en raccourcissant le vecteur si les derniers elements sont tous vides ?
}

const ValeurSP& Memoire::lecture(const Adresse& adresse, const Type& type_accepte)
{
	if(not adresse_valide(adresse))
	{
		throw LectureAdresseInvalide(adresse);
	}
	Type& type = _memoire_utilisee[adresse]->_type;
	if(type_accepte!=Type::tous())
	{
		if(not (((type_accepte==Type::entier_ou_chaine()) and ((type==Type::entier()) or (type==Type::chaine()))) or ((type_accepte==Type::entier_ou_pointeur()) and ((type==Type::entier()) or (type==Type::pointeur())))))
		{
			if(type!=type_accepte)
			{
				throw LectureTypeInvalide(adresse,type,type_accepte);
			}
		}
	}
	ValeurSP& valeur = _memoire_utilisee[adresse]->_valeur;
	if(not valeur)
	{
		throw LectureAdresseNonInitialisee(adresse,type);
	}
	return valeur;
}

const ValeurSP& Memoire::lecture(const Alias& alias, const Type& type_accepte)
{
	return lecture(resolution_alias(alias),type_accepte);
}

void Memoire::ecriture(const Adresse& adresse, const ValeurCSP& valeur)
{
	SETLGG_TEST(valeur);
	if(not adresse_valide(adresse))
	{
		throw EcritureAdresseInvalide(adresse);
	}
	Type type = _memoire_utilisee[adresse]->_type;
	Type type_accepte = *valeur;
	if(type!=type_accepte)
	{
		throw EcritureTypeInvalide(adresse,type,type_accepte);
	}
	if(valeur->est_nulle())
	{
		_memoire_utilisee[adresse]->deinitialise();
	}
	else
	{
		_memoire_utilisee[adresse]->_valeur=std::const_pointer_cast<Valeur>(valeur);
		// J'aime pas ce const_pointer_cast, mais il me permet de concentrer les risques
		// de modification d'une valeur qui devrait rester constante a un seul endroit,
		// et laisse le compilateur raler aux autres endroits ou une connerie pourrait
		// se glisser...
		// Mais j'aime pas ce const_pointer_cast...
	}
}

void Memoire::ecriture(const Alias& alias, const ValeurCSP& valeur)
{
	ecriture(resolution_alias(alias),valeur);
}


Adresse Memoire::resolution_alias(const Alias& alias) const
{
	auto adresse = _aliases.find(alias);
	if(adresse==_aliases.end())
	{
		throw AliasInconnu(alias);
	}
	return adresse->second;
}

void Memoire::ajoute_bloc_libre(const Adresse& adresse, const size_t& taille)
{
	_memoire_libre[adresse]=taille;
	_memoire_libre_inverse.insert(std::pair<size_t,Adresse>(taille,adresse));
}

struct RechercheParValeur
{
	RechercheParValeur(const Adresse& adresse)
	:_adresse(adresse) {};
	bool operator() (std::pair<size_t,Adresse> valeur)
	{
		return _adresse==valeur.second;
	};
	private:
	Adresse _adresse;
};

void Memoire::enleve_bloc_libre(const Adresse& adresse)
{
	size_t taille = _memoire_libre[adresse];
	_memoire_libre.erase(adresse);
	auto it = _memoire_libre_inverse.equal_range(taille);
	auto it_reel = std::find_if(it.first,it.second,RechercheParValeur(adresse));
	_memoire_libre_inverse.erase(it_reel);
}

bool Memoire::adresse_valide(const Adresse& adresse) const
{
	if(adresse>=_memoire_utilisee.size())
	{
		return false;
	}
	return static_cast<bool>(_memoire_utilisee[adresse]);
}

bool Memoire::adresse_initialisee(const Adresse& adresse) const
{
	if(not adresse_valide(adresse))
	{
		throw AdresseNonValidePourTestInitialisation(adresse);
	}
	return _memoire_utilisee[adresse]->initialise();
}

void Memoire::variation(const Adresse& adresse, const long int variation)
{
	const ValeurSP& valeur = lecture(adresse,Type::entier_ou_pointeur());
	(*valeur) += variation;
}

void Memoire::nettoie(const Adresse& adresse)
{
	if(not adresse_valide(adresse))
	{
		throw AdresseNonValidePourNettoyage(adresse);
	}
	return _memoire_utilisee[adresse]->deinitialise();
}

void Memoire::tronque()
{
	Adresse fin = _memoire_utilisee.size();
	for(auto it=_memoire_libre_inverse.cbegin() ; it!=_memoire_libre_inverse.cend() ; ++it)
	{
		if(it->second+it->first==fin)
		{
			Adresse adresse = it->second;
			enleve_bloc_libre(adresse);
			_memoire_utilisee.resize(adresse);
			return;
		}
	}
}

bool Memoire::ajoute_alias(const Alias& alias, const Adresse& adresse)
{
	auto it = _aliases.find(alias);
	if(it!=_aliases.end())
	{
		return false;
	}
	_aliases[alias]=adresse;
	return true;
}

bool Memoire::supprime_alias(const Alias& alias)
{
	auto it = _aliases.find(alias);
	if(it==_aliases.end())
	{
		return false;
	}
	_aliases.erase(it);
	return true;
}

std::pair<bool,const Type> Memoire::type(const Adresse& adresse)
{
	if(not adresse_valide(adresse))
	{
		return std::pair<bool,const Type>(false,Type::tous());
	}
	const Type& type = *(_memoire_utilisee[adresse]);
	return std::pair<bool,const Type>(true,type);
}

bool Memoire::type(const Adresse& adresse, const Type& type)
{
	if(not adresse_valide(adresse))
	{
		return false;
	}
	ElementSP element(new Element(type));
	_memoire_utilisee[adresse] = element;
	return true;
}

bool Memoire::desinitialise(const Adresse& adresse)
{
	if(not adresse_valide(adresse))
	{
		return false;
	}
	_memoire_utilisee[adresse]->deinitialise();
	return true;
}

Memoire::TypePointe Memoire::type_pointe(const Adresse& adresse)
{
	if(not adresse_valide(adresse))
	{
		return TypePointe::NONDEFINI;
	}
	if(not _memoire_utilisee[adresse]->initialise())
	{
		return TypePointe::NONINITIALISE;
	}
	const Type type = *(_memoire_utilisee[adresse]);
	if(type==Type::pointeur())
	{
		return TypePointe::POINTEUR;
	}
	return TypePointe::VALEUR;
}

Adresse Memoire::suivi(const Adresse& adresse)
{
	if(not adresse_valide(adresse))
		throw;
	ValeurSP valeur = _memoire_utilisee[adresse]->_valeur;
	if(not valeur)
		throw;
	return valeur->suivi();
}

void Memoire::ListeBlocsMemoire::operator-=(const Memoire::ListeBlocsMemoire& blocs)
{
//	Cet algo marche pas, il faut retirer les ensembles d'un bloc
//	Memoire::BlocMemoire::Set restant;
//	for(auto b:_blocs_memoire)
//	{
//		Memoire::BlocMemoire::Set r = b.exclus(blocs._blocs_memoire);
//		restant.insert(r.cbegin(),r.cend());
//#ifdef SETLGG_DEBUG
//SETLGG_TRACE;
//	for(auto rr:restant)
//	{
//		std::cerr << " " << rr;
//	}
//	std::cerr << std::endl;
//#endif
//	}
//	_blocs_memoire = restant;
	
	std::set<Adresse> bornes;
	std::map<Adresse,bool> intervales_this;
	std::map<Adresse,bool> intervales_blocs;
	std::map<Adresse,bool> intervales_resultat;
	for(auto b:_blocs_memoire)
	{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << b._adresse << " ";
#endif
		auto it=intervales_this.find(b._adresse);
		if(it!=intervales_this.end())
		{
#ifdef SETLGG_DEBUG
std::cerr << "change";
#endif
			it->second=true;
		}
		else
		{
#ifdef SETLGG_DEBUG
std::cerr << "add - true";
#endif
			intervales_this[b._adresse]=true;
		}
#ifdef SETLGG_DEBUG
std::cerr << " " << b._adresse+b._taille << " add - false" << std::endl;
#endif
		intervales_this[b._adresse+b._taille]=false;
		bornes.insert(b._adresse);
		bornes.insert(b._adresse+b._taille);
	}
	for(auto b:blocs._blocs_memoire)
	{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << b._adresse << " ";
#endif
		if((not intervales_blocs.empty()) and (intervales_blocs.find(b._adresse)!=intervales_blocs.end()))
		{
#ifdef SETLGG_DEBUG
std::cerr << "change";
#endif
			intervales_blocs[b._adresse]=true;
		}
		else
		{
#ifdef SETLGG_DEBUG
std::cerr << "add - true";
#endif
			intervales_blocs[b._adresse]=true;
		}
#ifdef SETLGG_DEBUG
std::cerr << " " << b._adresse+b._taille << " add - false" << std::endl;
#endif
		intervales_blocs[b._adresse+b._taille]=false;
		bornes.insert(b._adresse);
		bornes.insert(b._adresse+b._taille);
	}
	bool dans_this=false;
	bool dans_blocs=false;
	for(auto b:bornes)
	{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << b << " ";
#endif
		auto it_this = intervales_this.find(b);
		if(it_this!=intervales_this.end())
		{
#ifdef SETLGG_DEBUG
std::cerr << " dans_this " << it_this->second ;
#endif
			dans_this=it_this->second;
		}
		auto it_blocs = intervales_blocs.find(b);
		if(it_blocs!=intervales_blocs.end())
		{
#ifdef SETLGG_DEBUG
std::cerr << " dans_blocs " << it_blocs->second ;
#endif
			dans_blocs=it_blocs->second;
		}
#ifdef SETLGG_DEBUG
std::cerr << " dans_resultat " << (dans_this and not dans_blocs) << std::endl;
#endif
		intervales_resultat[b]=dans_this and not dans_blocs;
	}
	_blocs_memoire.clear();
	bool dans_resultat=false;
	Adresse debut_resultat;
	for(auto b:intervales_resultat)
	{
		if(b.second!=dans_resultat)
		{
			dans_resultat=b.second;
			if(dans_resultat)
			{
				debut_resultat = b.first;
			}
			else
			{
				_blocs_memoire.insert(Memoire::BlocMemoire(debut_resultat,b.first-debut_resultat));
			}
		}
	}
}


bool Memoire::verification_types(const BlocMemoire& bloc, const MemoireSP& memoire) const
{
	if(bloc._taille!=memoire->_memoire_utilisee.size())
		return false;
	for(Adresse adresse=bloc._adresse, nouvelle=Adresse() ; adresse<(bloc._adresse+bloc._taille) ; ++adresse, ++nouvelle)
	{
		if(adresse_valide(adresse) and memoire->adresse_valide(nouvelle))
		{
			if(_memoire_utilisee[adresse]->_type!=memoire->_memoire_utilisee[nouvelle]->_type)
				return false;
		}
		else
		{
			return false;
		}
	}
	return true;
}

bool Memoire::verification_taille(const size_t taille) const
{
	return taille==_memoire_utilisee.size();
}

bool Memoire::importe(const BlocMemoire& bloc, const MemoireSP& memoire)
{
	if(not verification_types(bloc,memoire))
		return false;
	importe_interne(bloc,memoire);
	return true;
}

Memoire::BlocMemoire Memoire::importe(const MemoireSP& memoire)
{
	Memoire::ListePlacesMemoire places = { PlaceMemoire(Type::TypeValeur::ENTIER,memoire->_memoire_utilisee.size(), AliasSP()) };
	BlocMemoire bloc = allocation(places);
	importe_interne(bloc, memoire);
	return bloc;
}


void Memoire::importe_interne(const BlocMemoire& bloc, const MemoireSP& memoire)
{
	for(Adresse adresse=bloc._adresse, nouvelle=Adresse() ; adresse<(bloc._adresse+bloc._taille) ; ++adresse, ++nouvelle)
	{
		_memoire_utilisee[adresse] = memoire->_memoire_utilisee[nouvelle];
	}
}

void Memoire::decalage(const BlocMemoire& bloc, const BlocMemoire& depart)
{
	long int var = bloc._adresse-depart._adresse;
	for(Adresse adresse=bloc._adresse ; adresse<(bloc._adresse+bloc._taille) ; ++adresse)
	{
		if(type_pointe(adresse)==TypePointe::POINTEUR)
		{
			SetLgg::Machine::Memoire::AdresseSP val_adresse = std::dynamic_pointer_cast<SetLgg::Machine::Memoire::Adresse>(_memoire_utilisee[adresse]->_valeur);
			if(depart(*val_adresse))
			{
				variation(adresse,var);
			}
		}
	}
}
