/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_MEMOIRE_MEMOIRE_H_
#define _MACHINE_MEMOIRE_MEMOIRE_H_

#include <vector>
#include <set>
#include <map>
#include <initializer_list>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Memoire { DECL_SHARED_PTR(Memoire); } } }

#include <src/machine/memoire/donnees/placememoire.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Processeur
		{
			class AllocationMemoire;
		}
		namespace Memoire
		{
			typedef std::vector<SetLgg::Machine::Memoire::Alias> ListeAlias;

			class Memoire
			{
				public:
                                        struct BlocMemoire 
                                        {
						typedef std::set<BlocMemoire> Set;
                                                BlocMemoire(const Adresse& adresse, const size_t& taille)
                                                :_adresse(adresse),_taille(taille) {};
						bool operator<(const BlocMemoire& bloc) const
						{
							if(_adresse<bloc._adresse)
								return true;
							if(_adresse>bloc._adresse)
								return false;
							return _taille<bloc._taille;
						};
						bool operator() (const Adresse& adresse) const
						{
							return (adresse>=_adresse) and (adresse<_adresse+_taille);
						}
						bool fusion(const BlocMemoire& bloc)
						{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "Fusion : " << _adresse << "," << _taille << " <=> " << bloc._adresse << "," << bloc._taille << ": " << std::endl; 
#endif
							if(bloc._adresse==_adresse)
							{
								if(bloc._taille>_taille)
									_taille=bloc._taille;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _adresse << "," << _taille << std::endl;
#endif
								return true;
							}
							if(_adresse+_taille>=bloc._adresse)
							{
								_taille = bloc._adresse+bloc._taille-_adresse;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _adresse << "," << _taille << std::endl;
#endif
								return true;
							}
#ifdef SETLGG_DEBUG
SETLGG_TRACE << _adresse << "," << _taille << std::endl;
#endif
							return false;
						};
						BlocMemoire::Set exclus(const BlocMemoire& bloc) const
						{
							BlocMemoire::Set restant;
							Adresse fin_courant = _adresse+_taille;
							Adresse fin_exclus = bloc._adresse+bloc._taille;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "exclus " << bloc << " de " << *this << std::endl;
SETLGG_TRACE << "[ " << _adresse << " ; " << fin_courant << " [ \\ [ " << bloc._adresse << " ; " << fin_exclus << " [" << std::endl;
SETLGG_TRACE << ((bloc._adresse<=_adresse) and (fin_courant<=fin_exclus)) << " " << ((fin_exclus<=_adresse) or (fin_courant<=bloc._adresse)) << " " << ((bloc._adresse<=_adresse) and (fin_exclus<=fin_courant)) << " " << ((_adresse<=bloc._adresse) and (fin_courant<=fin_exclus)) << std::endl;
#endif
							if((bloc._adresse<=_adresse) and (fin_courant<=fin_exclus))
							{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "void" << std::endl;
#endif
								return restant;
							}
							if((fin_exclus<=_adresse) or (fin_courant<=bloc._adresse))
							{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "same " << "[ " << _adresse << " ; " << fin_courant << " [" << std::endl;
#endif
								restant.insert(*this);
								return restant;
							}
							if((bloc._adresse<=_adresse) and (fin_exclus<=fin_courant))
							{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "dessous " << "[ " << fin_exclus << " ; " << fin_courant << " [" << " (" << fin_exclus << "*" << (fin_courant-fin_exclus) << ")" << std::endl;
#endif
								restant.insert(BlocMemoire(fin_exclus,fin_courant-fin_exclus));
								return restant;
							}
							if((_adresse<=bloc._adresse) and (fin_courant<=fin_exclus))
							{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "dessus " << "[ " << _adresse << " ; " << bloc._adresse << " [" << " (" << _adresse << "*" << (fin_courant-bloc._adresse) << ")" << std::endl;
#endif
								restant.insert(BlocMemoire(_adresse,fin_courant-bloc._adresse));
								return restant;
							}
#ifdef SETLGG_DEBUG
SETLGG_TRACE << "autour " << "[ " << _adresse << " ; " << bloc._adresse << " [ union [ " << fin_exclus << " ; " << fin_courant << " [" << " (" << _adresse << "*" << (bloc._adresse-_adresse) << ") union (" << fin_exclus << "*" << fin_courant-fin_exclus << ")" << std::endl;
#endif
							restant.insert(BlocMemoire(_adresse,bloc._adresse-_adresse));
							restant.insert(BlocMemoire(fin_exclus,fin_courant-fin_exclus));
							return restant;
						};
						BlocMemoire::Set exclus(const BlocMemoire::Set& blocs) const
						{
							BlocMemoire::Set resultat;
							for(auto it:blocs)
							{
								BlocMemoire::Set e = exclus(it);
								resultat.insert(e.cbegin(),e.cend());
							}
							return resultat;
						};
						operator std::string () const
						{
							std::ostringstream oss;
							oss << *this;
							return oss.str();
						}
                                                Adresse _adresse;
                                                size_t _taille;
						template<typename oStream>
						friend oStream& operator<<(oStream& os, const BlocMemoire& bloc)
						{
							os << bloc._adresse;
							if(bloc._taille!=1)
							{
								os << "*" << bloc._taille;
							}
							return os;
						}
                                        };
					DECL_SHARED_PTR(BlocMemoire);
					struct ListeBlocsMemoire
					{
						friend class SetLgg::Machine::Processeur::AllocationMemoire;
						ListeBlocsMemoire() = default;
						ListeBlocsMemoire(const std::initializer_list<Memoire::BlocMemoire>& liste)
						:_blocs_memoire(liste.begin(),liste.end()) { };
						void operator+=(const BlocMemoire& bloc) { _blocs_memoire.insert(bloc); }
						void operator+=(const BlocMemoire::Set& blocs) { _blocs_memoire.insert(blocs.cbegin(),blocs.cend()); }
						void operator+=(const ListeBlocsMemoire& blocs) { _blocs_memoire.insert(blocs._blocs_memoire.cbegin(),blocs._blocs_memoire.cend()); }
						void operator-=(const ListeBlocsMemoire& blocs);
						BlocMemoire::Set::const_iterator cbegin() const { return _blocs_memoire.cbegin(); }
						BlocMemoire::Set::const_iterator cend() const { return _blocs_memoire.cend(); }
						static Memoire::ListeBlocsMemoire agregation(const Memoire::ListeBlocsMemoire& blocs_memoire);
						operator bool () const
						{
							return not _blocs_memoire.empty();
						}
						template<typename oStream>
						friend oStream& operator<<(oStream& os, const ListeBlocsMemoire& blocs)
						{
							os << "{";
							for(auto b:blocs._blocs_memoire)
							{
								os << " " << b;
							}
							os << " }";
							return os;
						}
						private:
							BlocMemoire::Set _blocs_memoire;
					};
					typedef std::vector<PlaceMemoire> ListePlacesMemoire;
					Memoire() = default;
					BlocMemoire allocation(const ListePlacesMemoire& places_memoire);
					void liberation(const ListeBlocsMemoire& blocs_memoire);
					const ValeurSP& lecture(const Adresse& adresse, const Type& type_accepte);
					const ValeurSP& lecture(const Alias& alias, const Type& type_accepte);
					// Optimisation: la valeur est directement stockee en memoire, sans aucune copie
					// Attention, la valeur est const_castee !
					void ecriture(const Adresse& adresse, const ValeurCSP& valeur);
					void ecriture(const Alias& alias, const ValeurCSP& valeur);
					Adresse resolution_alias(const Alias& alias) const;
					bool adresse_valide(const Adresse& adresse) const;
					bool adresse_initialisee(const Adresse& adresse) const;
					void ajout_element(const Adresse& adresse, const ListeAlias& aliases, const ElementSP& element, const SetLgg::Global::Source::PositionSP& position);
					void variation(const Adresse& adresse, const long int variation);
					void nettoie(const Adresse& adresse);
					// void compacter(); // Impossible, sinon, 0 -> &1 va foirer si &0 n'est plus alloue apres un :delete
					void tronque();
					bool ajoute_alias(const Alias& alias, const Adresse& adresse);
					bool supprime_alias(const Alias& alias);
					std::pair<bool,const Type> type(const Adresse& adresse);
					bool desinitialise(const Adresse& adresse);
					bool type(const Adresse& adresse, const Type& type);
					enum class TypePointe {NONDEFINI,NONINITIALISE,POINTEUR,VALEUR};
					TypePointe type_pointe(const Adresse& adresse);
					Adresse suivi(const Adresse& adresse);
					template<bool autorise_non_defini, bool autorise_non_initialise>
						MemoireSP exporte(const BlocMemoire& bloc) const
						{
							MemoireSP memoire(new Memoire());
							for(Adresse adresse=bloc._adresse, nouvelle=Adresse() ; adresse<(bloc._adresse+bloc._taille) ; ++adresse, ++nouvelle)
							{
								if(not adresse_valide(adresse))
								{
									if(not autorise_non_defini)
										return MemoireSP();
									else
										continue;
								}
								if(not autorise_non_initialise)
								{
									if(not adresse_initialisee(adresse))
										return MemoireSP();
								}
								memoire->ajout_element(nouvelle,{},_memoire_utilisee[adresse],SetLgg::Global::Source::PositionSP());
							}
							return memoire;
						}

					bool verification_types(const BlocMemoire& bloc, const MemoireSP& memoire) const;
					bool verification_taille(const size_t taille) const;
					bool importe(const BlocMemoire& bloc, const MemoireSP& memoire);
					BlocMemoire importe(const MemoireSP& memoire); // allocation du bloc
					void decalage(const BlocMemoire& bloc, const BlocMemoire& depart);
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const Memoire& memoire)
					{
						std::multimap<Adresse,Alias> aliases_inverses;
						for(auto it = memoire._aliases.cbegin() ; it != memoire._aliases.cend() ; ++it)
						{
							aliases_inverses.insert(std::pair<Adresse,Alias>(it->second,it->first));
						}
						os << "# Memory:" << std::endl;
						Adresse adresse;
						for(auto element = memoire._memoire_utilisee.cbegin() ; element!=memoire._memoire_utilisee.cend() ; ++element,++adresse)
						{
							if(*element)
							{
								os << "#> " << adresse;
								auto alias_limite = aliases_inverses.equal_range(adresse);
								for(auto alias=alias_limite.first ; alias!=alias_limite.second ; ++alias)
								{
									os << " / " << alias->second;
								}
								os << " : " << *(*element) << std::endl;
							}

						}
						os << "# Aliases:" << std::endl;
						for(auto alias = memoire._aliases.cbegin() ; alias!=memoire._aliases.cend() ; ++alias)
						{
							os << "# " << alias->first << " -> " << alias->second << std::endl;
						}
						os << "# Free addresses:" << std::endl;
						for(auto libre = memoire._memoire_libre.cbegin() ; libre!=memoire._memoire_libre.cend() ; ++libre)
						{
							os << "# From " << libre->first << " on " << libre->second << " addresses" << std::endl;
						}
						os << "# Free blocks:" << std::endl;
						for(auto libre = memoire._memoire_libre_inverse.cbegin() ; libre!=memoire._memoire_libre_inverse.cend() ; ++libre)
						{
							os << "# Block of " << libre->first << " free addresses from " << libre->second << std::endl;
						}
						os << "# End of memory" << std::endl;
						return os;
					};
				private:
					std::vector<ElementSP> _memoire_utilisee;
					std::map<Adresse,size_t> _memoire_libre; // Adresse debut, nb adresses consecutives libres
					std::multimap<size_t,Adresse> _memoire_libre_inverse;
					std::map<Alias,Adresse> _aliases;
					// l'inverse est calcule au moment voulu (dump memoire)
					void ajoute_bloc_libre(const Adresse& adresse, const size_t& taille);
					void enleve_bloc_libre(const Adresse& adresse);
					void importe_interne(const BlocMemoire& bloc, const MemoireSP& memoire);
			};

			class AdresseNonOrdonnee : public SetLgg::Global::Exception::Compilation
			{
				public:
					AdresseNonOrdonnee(const Adresse& adresse, const SetLgg::Global::Source::PositionSP& position)
					:Compilation(position,std::string("Address ")+std::string(adresse)+std::string(" misplaced")) {};
			};
			class AliasDejaExistant : public SetLgg::Global::Exception::Compilation
			{
				public:
					AliasDejaExistant(const Alias& alias, const Adresse& adresse, const SetLgg::Global::Source::PositionSP& position)
					:Compilation(position,std::string("Alias ")+std::string(alias)+std::string(" already bound to address ")+std::string(adresse)) {};
			};

			class AliasInconnu : public SetLgg::Global::Exception::Execution
			{
				public:
					AliasInconnu(const Alias& alias)
					:Execution(SIGSEGV,std::string("Alias ")+std::string(alias)+std::string(" is unknown")) {};
			};

			class LectureAdresseInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					LectureAdresseInvalide(const Adresse& adresse)
					:Execution(SIGSEGV,std::string("Read to invalid address ")+std::string(adresse)) {};
			};

			class LectureTypeInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					LectureTypeInvalide(const Adresse& adresse, const Type& type, const Type& type_demande)
					:Execution(SIGSEGV,std::string("Read to address ")+std::string(adresse)+" incorrect as value type is "+std::string(type)+" instead of "+std::string(type_demande)) {};
			};

			class LectureAdresseNonInitialisee : public SetLgg::Global::Exception::Execution
			{
				public:
					LectureAdresseNonInitialisee(const Adresse& adresse, const Type& type)
					:Execution(SIGSEGV,std::string("Read to address ")+std::string(adresse)+" of type "+std::string(type)+" not initialised") {};
			};

			class EcritureAdresseInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					EcritureAdresseInvalide(const Adresse& adresse)
					:Execution(SIGSEGV,std::string("Write to invalid address ")+std::string(adresse)) {};
			};

			class EcritureTypeInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					EcritureTypeInvalide(const Adresse& adresse, const Type& type, const Type& type_demande)
					:Execution(SIGSEGV,std::string("Write to address ")+std::string(adresse)+" incorrect as value type is "+std::string(type)+" instead of "+std::string(type_demande)) {};
			};

			class AliasDejaPresent : public SetLgg::Global::Exception::Execution
			{
				public:
					AliasDejaPresent(const Alias& alias, const Adresse& adresse)
					:Execution(SIGSEGV,std::string("Alias ")+std::string(alias)+std::string(" already bound to address ")+std::string(adresse)) {};
			};

			class LiberationAdresseInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					LiberationAdresseInvalide(const Adresse& adresse)
					:Execution(SIGSEGV,std::string("Address ")+std::string(adresse)+std::string(" invalid for deletion")) {};
			};

                        class AdresseNonValidePourTestInitialisation : public SetLgg::Global::Exception::Execution
                        {
                                public:
                                        AdresseNonValidePourTestInitialisation(const Adresse& adresse)
                                        :Execution(SIGSEGV,std::string("Initialisation test failed for invalid address ")+std::string(adresse)) {};
                        };

                        class AdresseNonValidePourNettoyage : public SetLgg::Global::Exception::Execution
                        {
                                public:
                                        AdresseNonValidePourNettoyage(const Adresse& adresse)
                                        :Execution(SIGSEGV,std::string("Clear on invalid address ")+std::string(adresse)) {};
                        };

		}
	}
}

#endif
