/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef __MACHINE_PLUGIN_DONNEES_CONTENU_H__
#define __MACHINE_PLUGIN_DONNEES_CONTENU_H__

#include <vector>
#include <src/global/global.h>

namespace SetLgg { namespace Machine { namespace Plugin { DECL_SHARED_PTR(Contenu); DECL_SHARED_PTR(ContenuType); DECL_SHARED_PTR(ContenuInstruction); } } }

#include <src/machine/memoire/donnees/type.h>
#include <src/machine/plugin/donnees/typeutilisateur.h>
#include <src/machine/plugin/donnees/instructionutilisateur.h>
#include <src/machine/programme/donnees/instructionutilisateur.h>
#include <src/machine/plugin/interface/setlgg_machine.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Plugin
		{
			class Plugin;
			typedef std::vector<ContenuSP> ListeContenus;
			class Contenu
			{
				public:
					virtual ~Contenu() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const Contenu& contenu)
					{
						return contenu.format(os);
					};
					virtual std::ostream& format(std::ostream& os) const = 0;
				protected:
					Contenu() {};
			};

			class ContenuType : public Contenu
			{
				friend class Plugin;
				public:
					virtual ~ContenuType() {};
					ContenuType(const std::string& nom_type)
					:_nom_type(nom_type) {};
					virtual std::ostream& format(std::ostream& os) const
					{
						os << "TYPE " << _nom_type;
						return os;
					};
				private:
					std::string _nom_type;
			};

			class ContenuInstructionParametreAdresse;
			class ContenuInstructionParametreType;

			class Parametre
			{
				friend class ContenuInstructionParametreAdresse;
				friend class ContenuInstructionParametreType;
				public:
					Parametre(const SetLgg::Machine::Memoire::ValeurCSP& valeur)
					:_valeur(valeur) {};
					Parametre(const SetLgg::Machine::Memoire::AdresseSP& adresse)
					:_adresse(adresse) {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const Parametre& parametre)
					{
						if(parametre._valeur)
						{
							os << *(parametre._valeur);
						}
						if(parametre._adresse)
						{
							os << *(parametre._adresse);
						}
						return os;
					};
					virtual operator const SetLgg_Machine_Value () const
					{
						SetLgg_Machine_Value value;
						if(_adresse)
						{
							value = *_adresse;
						}
						if(_valeur)
						{
							value = *_valeur;
						}
						return value;
					};
				private:
					SetLgg::Machine::Memoire::ValeurCSP _valeur;
					SetLgg::Machine::Memoire::AdresseSP _adresse;
			};
			typedef std::vector<Parametre> ListeParametres;


			DECL_SHARED_PTR(ContenuInstructionParametre);
			typedef std::vector<ContenuInstructionParametreSP> ListeContenuInstructionParametre;

			class ContenuInstructionParametre
			{
				public:
					ContenuInstructionParametre() = default;
					virtual ~ContenuInstructionParametre() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const ContenuInstructionParametre& contenu)
					{
						return contenu.format(os);
					};
					virtual std::ostream& format(std::ostream& os) const = 0;
					virtual void conformite(const SetLgg::Machine::Plugin::Parametre& parametre, const std::string& nom_instruction, const size_t indice) const =0;
			};

			class InstructionParametreIncorrectAdresse : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionParametreIncorrectAdresse(const std::string& nom_instruction, const std::string& indice)
					:Execution(SIGILL,std::string("Parameter of instruction ")+nom_instruction+" at position "+indice+" is not an address") {};
			};

			class ContenuInstructionParametreAdresse : public ContenuInstructionParametre
			{
				public:
					ContenuInstructionParametreAdresse() = default;
					virtual ~ContenuInstructionParametreAdresse() {};
					virtual std::ostream& format(std::ostream& os) const
					{
						os << "ADDRESS";
						return os;
					};
					virtual void conformite(const SetLgg::Machine::Plugin::Parametre& parametre, const std::string& nom_instruction, const size_t indice) const
					{
						if(not parametre._adresse)
						{
							std::ostringstream oss;
							oss << indice;
							throw InstructionParametreIncorrectAdresse(nom_instruction,oss.str());
						}
					};
			};

			class InstructionParametreIncorrectType : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionParametreIncorrectType(const std::string& nom_instruction, const std::string& indice)
					:Execution(SIGILL,std::string("Parameter of instruction ")+nom_instruction+" at position "+indice+" is not a value") {};
			};

			class InstructionParametreIncorrectTypeMauvaisType : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionParametreIncorrectTypeMauvaisType(const std::string& nom_instruction, const std::string& indice, const std::string& type_fourni, const std::string& type_demande)
					:Execution(SIGILL,std::string("Parameter of instruction ")+nom_instruction+" at position "+indice+" has type "+type_fourni+" instead of "+type_demande) {};
			};

			class ContenuInstructionParametreType : public ContenuInstructionParametre
			{
				public:
					ContenuInstructionParametreType(const SetLgg::Machine::Memoire::TypeSP type)
					:_type(type) {};
					virtual ~ContenuInstructionParametreType() {};
					virtual std::ostream& format(std::ostream& os) const
					{
						os << *_type;
						return os;
					};
					virtual void conformite(const SetLgg::Machine::Plugin::Parametre& parametre, const std::string& nom_instruction, const size_t indice) const
					{
						if(not parametre._valeur)
						{
							std::ostringstream oss;
							oss << indice;
							throw InstructionParametreIncorrectType(nom_instruction,oss.str());
						}
						SetLgg::Machine::Memoire::ValeurCSP valeur = parametre._valeur;
						SetLgg::Machine::Memoire::Type type_fourni = *valeur;
						if((type_fourni)!=(*_type))
						{
							std::ostringstream oss;
							oss << indice;
							throw InstructionParametreIncorrectTypeMauvaisType(nom_instruction,oss.str(),type_fourni,*_type);
						}
					};
				private:
					SetLgg::Machine::Memoire::TypeSP _type;
			};

			class ContenuInstruction : public Contenu
			{
				friend class Plugin;
				public:
					ContenuInstruction(const std::string& nom_instruction, const ListeContenuInstructionParametre parametres, bool retour)
					:_nom_instruction(nom_instruction), _parametres(parametres), _retour(retour) {};
					virtual ~ContenuInstruction() {};
					virtual std::ostream& format(std::ostream& os) const
					{
						os << "INSTRUCTION " << _nom_instruction;
						for(auto it=_parametres.cbegin() ; it!=_parametres.cend() ; ++it)
						{
							os << " " << *(*it);
						}
						if(not _retour)
						{
							os << " NO";
						}
						os << " RETURN";
						return os;
					};
				private:
					std::string _nom_instruction;
					ListeContenuInstructionParametre _parametres;
					bool _retour;
			};
		}
	}
}

#endif
