/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef __MACHINE_PLUGIN_DONNEES_DEPENDANCES_H__
#define __MACHINE_PLUGIN_DONNEES_DEPENDANCES_H__

#include <vector>
#include <map>
#include <src/global/global.h>

namespace SetLgg { namespace Machine { namespace Plugin { DECL_SHARED_PTR(Dependance); typedef std::vector<DependanceSP> ListeDependances; } } }

#include <src/machine/plugin/donnees/plugin.h>
#include <src/machine/plugin/donnees/typeutilisateur.h>
#include <src/machine/plugin/donnees/instructionutilisateur.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Plugin
		{
			class Dependance
			{
				public:
					virtual ~Dependance() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const Dependance& dependance)
					{
						return dependance.format(os);
					};
					virtual std::ostream& format(std::ostream& os) const = 0;
					virtual void verifiedependance(const std::string& nom_plugin, const std::map<std::string,PluginSP>& plugins, const std::map<std::string,TypeUtilisateurSP>& types, const std::map<std::string,InstructionUtilisateurSP>& instructions) const = 0;
				protected:
					Dependance(const std::string& nom_plugin)
					:_nom_plugin(nom_plugin) {};
					std::string _nom_plugin;
			};

			class PluginDependancePluginManquant : public SetLgg::Global::Exception::Integration
			{
				public:
					PluginDependancePluginManquant(const std::string& nom_plugin_manquant, const std::string& nom_plugin)
					:Integration(std::string("Missing dependency plugin ")+nom_plugin_manquant+" for plugin "+nom_plugin) {};
			};

			class DependancePlugin : public Dependance
			{
				public:
					DependancePlugin(const std::string& nom_plugin)
					:Dependance(nom_plugin) {};
					virtual ~DependancePlugin() {};
					virtual void verifiedependance(const std::string& nom_plugin, const std::map<std::string,PluginSP>& plugins, const std::map<std::string,TypeUtilisateurSP>& types, const std::map<std::string,InstructionUtilisateurSP>& instructions) const
					{
						if(plugins.find(_nom_plugin)==plugins.end())
						{
							throw PluginDependancePluginManquant(_nom_plugin,nom_plugin);
						}
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						os << "PLUGIN " << _nom_plugin;
						return os;
					};
			};

			class PluginDependanceTypeManquant : public SetLgg::Global::Exception::Integration
			{
				public:
					PluginDependanceTypeManquant(const std::string& nom_type_manquant, const std::string& nom_plugin)
					:Integration(std::string("Missing dependency type ")+nom_type_manquant+" for plugin "+nom_plugin) {};
			};

			class PluginDependanceTypeDansPluginManquant : public SetLgg::Global::Exception::Integration
			{
				public:
					PluginDependanceTypeDansPluginManquant(const std::string& nom_type, const std::string& nom_plugin_manquant, const std::string& nom_plugin)
					:Integration(std::string("Type dependency ")+nom_type+" not defined in plugin "+nom_plugin_manquant+" for plugin "+nom_plugin) {};
			};

			class DependanceType : public Dependance
			{
				public:
					DependanceType(const std::string& nom_type, const std::string& nom_plugin)
					:Dependance(nom_plugin), _nom_type(nom_type) {};
					virtual ~DependanceType() {};
					virtual void verifiedependance(const std::string& nom_plugin, const std::map<std::string,PluginSP>& plugins, const std::map<std::string,TypeUtilisateurSP>& types, const std::map<std::string,InstructionUtilisateurSP>& instructions) const
					{
						auto it = types.find(_nom_type);
						if(it==types.end())
						{
							throw PluginDependanceTypeManquant(_nom_type,nom_plugin);
						}
						if(not _nom_plugin.empty())
						{
							if(it->second->_nom_plugin!=_nom_plugin)
							{
								throw PluginDependanceTypeDansPluginManquant(_nom_type,_nom_plugin,nom_plugin);
							}
						}
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						os << "TYPE " << _nom_type;
						if(not _nom_plugin.empty())
						{
							os << " IN " << _nom_plugin;
						}
						return os;
					};
				private:
					std::string _nom_type;
			};

			class PluginDependanceInstructionManquant : public SetLgg::Global::Exception::Integration
			{
				public:
					PluginDependanceInstructionManquant(const std::string& nom_instruction_manquant, const std::string& nom_plugin)
					:Integration(std::string("Missing dependency instruction ")+nom_instruction_manquant+" for plugin "+nom_plugin) {};
			};

			class PluginDependanceInstructionDansPluginManquant : public SetLgg::Global::Exception::Integration
			{
				public:
					PluginDependanceInstructionDansPluginManquant(const std::string& nom_instruction, const std::string& nom_plugin_manquant, const std::string& nom_plugin)
					:Integration(std::string("Instruction dependency ")+nom_instruction+" not defined in plugin "+nom_plugin_manquant+" for plugin "+nom_plugin) {};
			};

			class DependanceInstruction : public Dependance
			{
				public:
					DependanceInstruction(const std::string& nom_instruction, const std::string& nom_plugin)
					:Dependance(nom_plugin), _nom_instruction(nom_instruction) {};
					virtual ~DependanceInstruction() {};
					virtual void verifiedependance(const std::string& nom_plugin, const std::map<std::string,PluginSP>& plugins, const std::map<std::string,TypeUtilisateurSP>& types, const std::map<std::string,InstructionUtilisateurSP>& instructions) const
					{
						auto it = instructions.find(_nom_instruction);
						if(it==instructions.end())
						{
							throw PluginDependanceInstructionManquant(_nom_instruction,nom_plugin);
						}
						if(not _nom_plugin.empty())
						{
							if(it->second->_nom_plugin!=_nom_plugin)
							{
								throw PluginDependanceInstructionDansPluginManquant(_nom_instruction,_nom_plugin,nom_plugin);
							}
						}
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						os << "INSTRUCTION " << _nom_instruction;
						if(not _nom_plugin.empty())
						{
							os << " IN " << _nom_plugin;
						}
						return os;
					};
				private:
					std::string _nom_instruction;
			};

		}
	}
}

#endif
