/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <src/global/global.h>
#include <src/global/exceptions.h>

#include <src/machine/plugin/interface/setlgg_machine.h>
#include <src/machine/plugin/donnees/plugin.h>
#include <src/machine/programme/analyseur/analyseur.h>
#include <src/machine/programme/donnees/programme.h>
#include <src/machine/programme/donnees/instructionretour.h>
#include <src/machine/programme/donnees/instructionenvoi.h>
#include <src/machine/programme/donnees/instructionreception.h>
#include <src/machine/programme/donnees/format.h>
#include <src/machine/machine/machine.h>
#include <src/machine/memoire/analyseur/analyseur.h>
#include <src/machine/memoire/donnees/entier.h>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/memoire/donnees/booleen.h>
#include <src/machine/memoire/donnees/adresse.h>
#include <src/machine/memoire/donnees/utilisateur.h>
#include <src/machine/flux/gestionnaireflux.h>

class PointeurConfigurationInvalide : public SetLgg::Global::Exception::Integration
{
	public:
		PointeurConfigurationInvalide()
		:Integration("Invalid pointer to configuration string in plugin registration") {};
};

class PointeurPluginInvalide : public SetLgg::Global::Exception::Integration
{
	public:
		PointeurPluginInvalide()
		:Integration("Invalid pointer to plugin handler in plugin registration") {};
};

class PointeurMachineInvalide : public SetLgg::Global::Exception::Integration
{
	public:
		PointeurMachineInvalide(const std::string& callback)
		:Integration("Invalid pointer to virtual machine handler in plugin callback "+callback) {};
};

class PointeurLabelInvalideProcesseur : public SetLgg::Global::Exception::Integration
{
	public:
		PointeurLabelInvalideProcesseur(const std::string& nom)
		:Integration("Invalid pointer to label name in plugin callback "+nom+"") {};
};

class PointeurMessageErreurInvalide : public SetLgg::Global::Exception::Integration
{
	public:
		PointeurMessageErreurInvalide(const std::string& callback)
		:Integration("Invalid pointer to error message in plugin callback "+callback) {};
};

class PointeurMarqueurInvalide : public SetLgg::Global::Exception::Integration
{
	public:
		PointeurMarqueurInvalide(const std::string& callback)
		:Integration("Invalid pointer to checkpoint in plugin callback "+callback) {};
};

class ErreurUtilisateur : public SetLgg::Global::Exception::Execution
{
	public:
		ErreurUtilisateur(const unsigned int interruption, const std::string& message)
		:Execution(interruption,message) {};
};

class PointeurAliasInvalideMemoire : public SetLgg::Global::Exception::Integration
{
	public:
		PointeurAliasInvalideMemoire(const std::string& nom)
		:Integration("Invalid pointer to memory alias in plugin callback "+nom+"") {};
};

class EntreeInvalide : public SetLgg::Global::Exception::Integration
{
	public:
		EntreeInvalide(const std::string& entree, const std::string& callback)
		:Integration(std::string("Invalid ")+entree+" provided in input of plugin callback "+callback) {};
};

namespace SetLgg
{
	namespace Machine
	{
		namespace Plugin
		{
			struct Interface
			{
				Interface(void *machine,const std::string& callback)
				{
					if(not machine)
					{
						throw PointeurMachineInvalide(callback);
					}
					_machine = *(reinterpret_cast<SetLgg::Machine::Machine::MachineSP*>(machine)) ;
				};
				static size_t adresse_programme(const SetLgg::Machine::Programme::Adresse& adresse)
				{
					return adresse;
				};
				static size_t adresse_memoire(const SetLgg::Machine::Memoire::Adresse& adresse)
				{
					return adresse;
				};
				SetLgg::Machine::Processeur::ProcesseurSP& processeur()
				{
					return _machine->_processeur;
				};
				SetLgg::Machine::Processeur::PileEtats& pile_retour()
				{
					return _machine->_processeur->_pile_etats;
				};
				SetLgg::Machine::Programme::ProgrammeSP& programme()
				{
					return _machine->_programme;
				}
				SetLgg::Machine::Debogueur::DebogueurSP& debogueur()
				{
					return _machine->_debogueur;
				}
				SetLgg::Machine::Memoire::MemoireSP& memoire()
				{
					return _machine->_memoire;
				}
				SetLgg::Machine::Plugin::GestionnairePluginSP& plugins()
				{
					return _machine->_plugins;
				}
				SetLgg::Machine::Flux::GestionnaireFluxSP& flux()
				{
					return _machine->_flux;
				}
				const SetLgg::Machine::Programme::Adresse& adresse_terminale()
				{
					return _machine->_programme->_adresse_terminale;
				}
				static SetLgg::Machine::Flux::FluxSP ouvre_flux_fichier(const int fd, const SetLgg_Machine_String name, const SetLgg_Machine_FileMode mode)
				{
					std::string nom_fichier(name._buffer,name._size);
					switch(mode)
					{
						case READONLY:
							return SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxFichierLectureSeule(fd,nom_fichier));
						case WRITEONLY:
							return SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxFichierEcritureSeule(fd,nom_fichier));
						case READWRITE:
							return SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxFichierLectureEcriture(fd,nom_fichier));
						default:
							throw;
					}
					throw;
				}
				static SetLgg::Machine::Flux::FluxSP ouvre_flux_socketTCP(const int fd, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port)
				{
					std::string str_local_ip(local_ip._buffer,local_ip._size);
					std::string str_local_port(local_port._buffer,local_port._size);
					std::string str_distant_ip(distant_ip._buffer,distant_ip._size);
					std::string str_distant_port(distant_port._buffer,distant_port._size);
					switch(mode)
					{
						case READONLY:
							return SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxSocketTCPServeurAttend(fd,str_local_ip,str_local_port,str_distant_ip,str_distant_port));
						case WRITEONLY:
							return SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxSocketTCPClient(fd,str_local_ip,str_local_port,str_distant_ip,str_distant_port));
						default:
							throw;
					}
					throw;
				}
				static SetLgg::Machine::Flux::FluxSP ouvre_flux_socketUDP(const int fd, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port)
				{
					std::string str_local_ip(local_ip._buffer,local_ip._size);
					std::string str_local_port(local_port._buffer,local_port._size);
					std::string str_distant_ip(distant_ip._buffer,distant_ip._size);
					std::string str_distant_port(distant_port._buffer,distant_port._size);
					return SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxSocketUDP(fd,str_local_ip,str_local_port,str_distant_ip,str_distant_port));
				}
				static int descripteur_lecture(const SetLgg::Machine::Flux::FluxSP& flux)
				{
					return flux->descripteur_lecture();
				}
				static int descripteur_ecriture(const SetLgg::Machine::Flux::FluxSP& flux)
				{
					return flux->descripteur_ecriture();
				}
				static std::string tampon_depile_flux(const SetLgg::Machine::Flux::FluxSP& flux, const size_t taille)
				{
					return flux->tampon_depile(taille);
				}
				static void tampon_empile_flux(const SetLgg::Machine::Flux::FluxSP& flux, const std::string& chaine)
				{
					flux->tampon_empile(chaine);
				}
				SetLgg::Machine::Machine::MachineSP _machine;

				struct Programme
				{
					SetLgg::Machine::Programme::ProgrammeSP _programme;
					SetLgg::Machine::Debogueur::DebogueurSP _debogueur;
					size_t taille() const
					{
						return _programme->taille();
					}
					std::pair<bool,SetLgg::Machine::Programme::Adresse> resolution_label(const std::string& label) const
					{
						return _programme->lien(label);
					}
				};
			};
			struct Outils
			{
				static SetLgg::Machine::Memoire::Type convertit_type(const std::string& nom_instruction, SetLgg::Machine::Plugin::GestionnairePluginSP plugins, const SetLgg_Machine_ValueType& type)
				{
					switch(type._type)
					{
						case VALUE_INTEGER:
							return SetLgg::Machine::Memoire::Type::entier();
						case VALUE_STRING:
							return SetLgg::Machine::Memoire::Type::chaine();
						case VALUE_BOOLEAN:
							return SetLgg::Machine::Memoire::Type::booleen();
						case VALUE_POINTER:
							return SetLgg::Machine::Memoire::Type::pointeur();
						case VALUE_INOUTREF:
							return SetLgg::Machine::Memoire::Type::nom_flux();
						case VALUE_USERTYPE:
							{
								std::string nom(type._user_name._buffer,type._user_name._size);
								SetLgg::Machine::Plugin::TypeUtilisateurSP type_utilisateur = plugins->type(nom);
								return SetLgg::Machine::Memoire::Utilisateur(type_utilisateur,nom);
							}
							break;
						case VALUE_ALL:
							return SetLgg::Machine::Memoire::Type::tous();
						default:
							break;
					}
					throw EntreeInvalide("type",nom_instruction);
				};

				static SetLgg::Machine::Memoire::NomFlux convertit_nom_flux(const std::string& nom_instruction, const SetLgg_Machine_InOutReference& ior)
				{
					switch(ior._type)
					{
						case SetLgg_Machine_InOutReferenceType::IOR_STDIN:
							{
								return SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDIN);
								break;
							}
						case SetLgg_Machine_InOutReferenceType::IOR_STDOUT:
							{
								return SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDOUT);
								break;
							}
						case SetLgg_Machine_InOutReferenceType::IOR_STDERR:
							{
								return SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDERR);
								break;
							}
						case SetLgg_Machine_InOutReferenceType::IOR_CUSTOM:
							{
								if(not ior._name._buffer)
								{
									throw EntreeInvalide("IOR name invalid",nom_instruction);
								}
								return SetLgg::Machine::Memoire::NomFlux(std::string(ior._name._buffer,ior._name._size));
								break;
							}
						default:
							throw EntreeInvalide("IOR type invalid",nom_instruction);
					}
				}

				static SetLgg::Machine::Memoire::ValeurSP convertit_valeur(const std::string& nom_instruction, SetLgg::Machine::Plugin::GestionnairePluginSP plugins, const SetLgg_Machine_Value& valeur)
				{
					SetLgg::Machine::Memoire::ValeurSP valeur_retour;
					switch(valeur._type._type)
					{
						case SetLgg_Machine_ValueTypeCategory::VALUE_INTEGER:
							if(valeur._initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::EntierNul());
							else
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Entier(valeur._value._integer));
							break;
						case SetLgg_Machine_ValueTypeCategory::VALUE_STRING:
							if(valeur._initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::ChaineNulle());
							else
							{
								if(not valeur._value._string._buffer)
								{
									throw EntreeInvalide("value",nom_instruction);
								}
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Chaine(std::string(valeur._value._string._buffer,valeur._value._string._size)));
							}
							break;
						case SetLgg_Machine_ValueTypeCategory::VALUE_BOOLEAN:
							if(valeur._initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::BooleenNul());
							else
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Booleen(valeur._value._boolean));
							break;
						case SetLgg_Machine_ValueTypeCategory::VALUE_POINTER:
							if(valeur._initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::AdresseNulle());
							else
								valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Adresse(valeur._value._pointer));
							break;
						case SetLgg_Machine_ValueTypeCategory::VALUE_INOUTREF:
							{
								if(valeur._initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
									valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFluxNul());
								else
								{
									switch(valeur._value._inoutref._type)
									{
										case SetLgg_Machine_InOutReferenceType::IOR_STDIN:
											{
												if(valeur._value._inoutref._name._buffer)
												{
													throw EntreeInvalide("value",nom_instruction);
												}
												valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDIN));
												break;
											}
										case SetLgg_Machine_InOutReferenceType::IOR_STDOUT:
											{
												if(valeur._value._inoutref._name._buffer)
												{
													throw EntreeInvalide("value",nom_instruction);
												}
												valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDOUT));
												break;
											}
										case SetLgg_Machine_InOutReferenceType::IOR_STDERR:
											{
												if(valeur._value._inoutref._name._buffer)
												{
													throw EntreeInvalide("value",nom_instruction);
												}
												valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDERR));
												break;
											}
										case SetLgg_Machine_InOutReferenceType::IOR_CUSTOM:
											{
												if(not valeur._value._inoutref._name._buffer)
												{
													throw EntreeInvalide("value",nom_instruction);
												}
												valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(std::string(valeur._value._inoutref._name._buffer,valeur._value._inoutref._name._size)));
												break;
											}
										default:
											throw EntreeInvalide("value",nom_instruction);
									}
								}

							}
							break;
						case SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE:
							{
								if(not valeur._type._user_name._buffer)
								{
									throw EntreeInvalide("value",nom_instruction);
								}
								std::string str_type(valeur._type._user_name._buffer,valeur._type._user_name._size);
								if(valeur._initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
									valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::UtilisateurNul(plugins->type(str_type)));
								else
								{
									if(not valeur._value._user_type)
									{
										throw EntreeInvalide("value",nom_instruction);
									}
									{
										valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Utilisateur(plugins->type(str_type),valeur._value._user_type));
									}
								}
							}
							break;
						default:
							{
								throw EntreeInvalide("value",nom_instruction);
							}
							break;
					}
					return valeur_retour;
				};
			};
		}
	}
}

extern "C"
{
void setlgg_machine_plugin_register(void *plugin_handler, const char *plugin_configuration)
{
	if(not plugin_handler)
	{
		throw PointeurPluginInvalide();
	}
	if(not plugin_configuration)
	{
		throw PointeurConfigurationInvalide();
	}
	std::string config(plugin_configuration);
	SetLgg::Machine::Plugin::Plugin *plugin=reinterpret_cast<SetLgg::Machine::Plugin::Plugin *>(plugin_handler);
	plugin->configuration(config);
}

SetLgg_Machine_Return setlgg_machine_error(void *machine, const unsigned int interruption, const char *error_message)
{
	if(not machine)
	{
		throw PointeurMachineInvalide(__func__);
	}
	if(not error_message)
	{
		throw PointeurMessageErreurInvalide(__func__);
	}
	std::string str(error_message);
	throw ErreurUtilisateur(interruption,str);
}

SetLgg_Machine_Return setlgg_machine_error_string(void *machine, const unsigned int interruption, const SetLgg_Machine_String error_message)
{
	if(not machine)
	{
		throw PointeurMachineInvalide(__func__);
	}
	if(not error_message._buffer)
	{
		throw PointeurMessageErreurInvalide(__func__);
	}
	std::string str(error_message._buffer,error_message._size);
	throw ErreurUtilisateur(interruption,str);
}

void setlgg_machine_machine_shutdown(void *machine, const unsigned char core_dump, const unsigned char memory_dump, const unsigned int return_code)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	machine_virtuelle._machine->arret(core_dump>0,memory_dump>0,return_code);
}

SetLgg_Machine_String setlgg_machine_value_convert_string(char *string)
{
	SetLgg_Machine_String str = {string, ::strlen(string)};
	return str;
}

SetLgg_Machine_Value* setlgg_machine_value_new_null_integer()
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_INTEGER;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::NULL_VALUE;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_null_string()
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_STRING;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::NULL_VALUE;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_null_boolean()
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_BOOLEAN;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::NULL_VALUE;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_null_pointer()
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_POINTER;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::NULL_VALUE;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_null_inoutref()
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_INOUTREF;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::NULL_VALUE;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_null_usertype(const char *name)
{
	auto taille=::strlen(name);
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE;
	value->_type._user_name = {SetLgg::Global::Chaine::duplique(name,taille+1), taille};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::NULL_VALUE;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_null_usertype_string(const SetLgg_Machine_String name)
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE;
	value->_type._user_name = {SetLgg::Global::Chaine::duplique(name._buffer,name._size), name._size};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::NULL_VALUE;
	return value;
}


SetLgg_Machine_Value* setlgg_machine_value_new_integer(const long int integer)
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_INTEGER;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._integer = integer;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_string(const char *string)
{
	auto taille=::strlen(string);
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_STRING;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._string._buffer = SetLgg::Global::Chaine::duplique(string,taille+1);
	value->_value._string._size=taille;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_sized_string(const char *string, const size_t size)
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_STRING;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._string._buffer = SetLgg::Global::Chaine::duplique(string,size);
	value->_value._string._size=size;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_boolean(const unsigned char boolean)
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_BOOLEAN;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._boolean = boolean;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_pointer(const size_t address)
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_POINTER;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._pointer = address;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_inoutref(const SetLgg_Machine_InOutReferenceType type, const char *name)
{
	auto taille=::strlen(name);
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_INOUTREF;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._inoutref._type=type;
	if(type==SetLgg_Machine_InOutReferenceType::IOR_CUSTOM)
	{
		value->_value._inoutref._name = {SetLgg::Global::Chaine::duplique(name,taille+1),taille};
	}
	else
	{
		value->_value._inoutref._name = {0,0};
	}
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_inoutref_string(const SetLgg_Machine_InOutReferenceType type, const SetLgg_Machine_String name)
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_INOUTREF;
	value->_type._user_name = {0, 0};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._inoutref._type=type;
	if(type==SetLgg_Machine_InOutReferenceType::IOR_CUSTOM)
	{
		value->_value._inoutref._name = {SetLgg::Global::Chaine::duplique(name._buffer,name._size),name._size};
	}
	else
	{
		value->_value._inoutref._name = {0,0};
	}
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_usertype(const char *name, void *handler)
{
	auto taille=::strlen(name);
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE;
	value->_type._user_name = {SetLgg::Global::Chaine::duplique(name,taille+1), taille};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._user_type = handler;
	return value;
}

SetLgg_Machine_Value* setlgg_machine_value_new_usertype_string(const SetLgg_Machine_String name, void *handler)
{
	SetLgg_Machine_Value *value = new SetLgg_Machine_Value;
	value->_type._type = SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE;
	value->_type._user_name = {SetLgg::Global::Chaine::duplique(name._buffer,name._size), name._size};
	value->_initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
	value->_value._user_type = handler;
	return value;
}

void setlgg_machine_value_clear(SetLgg_Machine_Value *value)
{
	if(value->_initialisation==SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE)
	{
		value->_initialisation=SetLgg_Machine_ValueInitialisation::NULL_VALUE;
		switch(value->_type._type)
		{
			case SetLgg_Machine_ValueTypeCategory::VALUE_STRING:
				::free(const_cast<char*>(value->_value._string._buffer));
				break;
			case SetLgg_Machine_ValueTypeCategory::VALUE_INOUTREF:
				if(value->_value._inoutref._type==SetLgg_Machine_InOutReferenceType::IOR_CUSTOM)
				{
					::free(const_cast<char*>(value->_value._inoutref._name._buffer));
				}
				break;
			case SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE:
				::free(const_cast<char*>(value->_type._user_name._buffer));
				break;
			default:
				return;
		}
	}
}

void setlgg_machine_value_delete(SetLgg_Machine_Value *value)
{
	if(not value)
		return;
	::setlgg_machine_value_clear(value);
	delete value;
}

SetLgg_Machine_Return setlgg_machine_value_initialised(SetLgg_Machine_Value* value)
{
	if(not value)
		return SetLgg_Machine_Return::UNDEFINED_MEMORY;
	if(value->_initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
		return SetLgg_Machine_Return::UNINITIALISED_MEMORY;
	return SetLgg_Machine_Return::OK;
}

void setlgg_machine_return_value_delete(SetLgg_Machine_ReturnValue *return_value)
{
	::setlgg_machine_value_clear(&(return_value->_value));
	return_value->_return=SetLgg_Machine_Return::VOID;
}

SetLgg_Machine_Return setlgg_machine_processor_exist_label(void *machine, const SetLgg_Machine_String label)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not label._buffer)
	{
		throw PointeurLabelInvalideProcesseur(__func__);
	}
	std::string str_label(label._buffer,label._size);
	if(machine_virtuelle.programme()->lien(str_label).first)
		return OK;
	return INEXISTANT_LABEL;
}

void setlgg_machine_internal_processor_jump_address(void *machine, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Programme::Adresse adresse(address);
	machine_virtuelle.processeur()->saut(adresse);
}

SetLgg_Machine_Return setlgg_machine_processor_jump_label(void *machine, const SetLgg_Machine_String label)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not label._buffer)
	{
		throw PointeurLabelInvalideProcesseur(__func__);
	}
	std::string str_label(label._buffer,label._size);
	auto adresse = machine_virtuelle.programme()->lien(str_label);
	if(not adresse.first)
		return INEXISTANT_LABEL;
	machine_virtuelle.processeur()->saut(adresse.second);
	return OK;
}

void setlgg_machine_internal_processor_call_address(void *machine, const size_t function_address, const size_t parameter_address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Programme::Adresse adresse_fonction(function_address);
	SetLgg::Machine::Memoire::Adresse adresse_parametres(parameter_address);
	machine_virtuelle.processeur()->appel(adresse_fonction,adresse_parametres);
}

SetLgg_Machine_Return setlgg_machine_processor_call_label(void *machine, const SetLgg_Machine_String label, const size_t parameter_address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not label._buffer)
	{
		throw PointeurLabelInvalideProcesseur(__func__);
	}
	std::string str_label(label._buffer,label._size);
	auto adresse = machine_virtuelle.programme()->lien(str_label);
	if(not adresse.first)
		return INEXISTANT_LABEL;
	SetLgg::Machine::Memoire::Adresse adresse_parametres(parameter_address);
	machine_virtuelle.processeur()->appel(adresse.second,adresse_parametres);
	return OK;
}

void setlgg_machine_processor_return(void *machine, const unsigned int delete_memory)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Programme::InstructionRetour::retour(machine_virtuelle._machine,delete_memory);
}

size_t setlgg_machine_internal_processor_next_instruction(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto etat = machine_virtuelle.processeur()->etat_courant();
	SetLgg::Machine::Programme::Adresse adresse = etat;
	return machine_virtuelle.adresse_programme(adresse);
}

SetLgg_Machine_Return setlgg_machine_processor_interrupt(void *machine, const unsigned int interruption)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not machine_virtuelle.processeur()->interruption(interruption))
		return INVALID_INTERRUPTION;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_internal_processor_interruption_register_address(void *machine, const unsigned int interruption, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Programme::AdresseSP adresse(new SetLgg::Machine::Programme::Adresse(address));
	if(not machine_virtuelle.processeur()->modifier_interruption(interruption,adresse))
		return INVALID_INTERRUPTION;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_processor_interruption_register_label(void *machine, const unsigned int interruption, const SetLgg_Machine_String label)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not label._buffer)
	{
		throw PointeurLabelInvalideProcesseur(__func__);
	}
	std::string str_label(label._buffer,label._size);
	auto adresse = machine_virtuelle.programme()->lien(str_label);
	if(not adresse.first)
		return INEXISTANT_LABEL;
	SetLgg::Machine::Programme::AdresseSP adressesp(new SetLgg::Machine::Programme::Adresse(adresse.second));
	if(not machine_virtuelle.processeur()->modifier_interruption(interruption,adressesp))
		return INVALID_INTERRUPTION;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_processor_interruption_unregister(void *machine, const unsigned int interruption)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Programme::AdresseSP adresse;
	if(not machine_virtuelle.processeur()->modifier_interruption(interruption,adresse))
		return INVALID_INTERRUPTION;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_processor_checkpoint_add(void *machine, const SetLgg_Machine_String checkpoint)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not checkpoint._buffer)
	{
		throw PointeurMarqueurInvalide(__func__);
	}
	std::string str_marqueur(checkpoint._buffer,checkpoint._size);
	machine_virtuelle.processeur()->ajoute_marqueur(str_marqueur);
	return OK;
}

SetLgg_Machine_Return setlgg_machine_processor_checkpoint_remove(void *machine, const SetLgg_Machine_String checkpoint)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not checkpoint._buffer)
	{
		throw PointeurMarqueurInvalide(__func__);
	}
	std::string str_marqueur(checkpoint._buffer,checkpoint._size);
	machine_virtuelle.processeur()->enleve_marqueur(str_marqueur);
	return OK;
}

SetLgg_Machine_Return setlgg_machine_processor_checkpoint_reached(void *machine, const SetLgg_Machine_String checkpoint)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not checkpoint._buffer)
	{
		throw PointeurMarqueurInvalide(__func__);
	}
	std::string str_marqueur(checkpoint._buffer,checkpoint._size);
	if(machine_virtuelle.processeur()->presence_marqueur(str_marqueur))
		return OK;
	else
		return CHECKPOINT_NOT_REACHED;
}

SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_dup_frame(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(machine_virtuelle.pile_retour().duplique())
		return OK;
	else
		return INEXISTANT_CALLSTACK_FRAME;
}

SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_drop_frame(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(machine_virtuelle.pile_retour().elimine())
		return OK;
	else
		return INEXISTANT_CALLSTACK_FRAME;
}

SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_swap_frame(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(machine_virtuelle.pile_retour().echange())
		return OK;
	else
		return INEXISTANT_CALLSTACK_FRAME;
}

SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_roll_frame(void *machine, const unsigned int frame_level)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(machine_virtuelle.pile_retour().rotation(frame_level))
		return OK;
	else
		return INEXISTANT_CALLSTACK_FRAME;
}

SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_unroll_frame(void *machine, const unsigned int frame_level)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(machine_virtuelle.pile_retour().rotation_inverse(frame_level))
		return OK;
	else
		return INEXISTANT_CALLSTACK_FRAME;
}

void setlgg_machine_internal_processor_callstack_push_current_frame(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto etat_courant = machine_virtuelle.processeur()->etat_courant();
	machine_virtuelle.pile_retour().empile(etat_courant);
}

SetLgg_Machine_ReturnFrame setlgg_machine_internal_processor_callstack_find_frame_address_interne(SetLgg::Machine::Plugin::Interface& machine_virtuelle, const SetLgg::Machine::Programme::Adresse& adresse)
{
	SetLgg_Machine_ReturnFrame resultat;
	auto res = machine_virtuelle.pile_retour().recherche(adresse);
	if(res>=machine_virtuelle.pile_retour().taille())
	{
		resultat._return = FRAME_NOT_FOUND;
		resultat._frame = 0;
		return resultat;
	}
	resultat._return = OK;
	resultat._frame = res;
	return resultat;
}

SetLgg_Machine_ReturnFrame setlgg_machine_internal_processor_callstack_find_frame_label(void *machine, const SetLgg_Machine_String label)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not label._buffer)
	{
		throw PointeurLabelInvalideProcesseur(__func__);
	}
	std::string str_label(label._buffer,label._size);
	auto adresse = machine_virtuelle.programme()->lien(str_label);
	if(not adresse.first)
	{
		SetLgg_Machine_ReturnFrame resultat;
		resultat._return = INEXISTANT_LABEL;
		resultat._frame = 0;
		return resultat;
	}
	return setlgg_machine_internal_processor_callstack_find_frame_address_interne(machine_virtuelle,adresse.second);
}

SetLgg_Machine_ReturnFrame setlgg_machine_internal_processor_callstack_find_frame_address(void *machine, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Programme::Adresse adresse(address);
	return setlgg_machine_internal_processor_callstack_find_frame_address_interne(machine_virtuelle,adresse);
}

unsigned int setlgg_machine_internal_processor_callstack_size(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	return machine_virtuelle.pile_retour().taille();
}


void setlgg_machine_internal_processor_register_allocation(void *machine, const size_t address, const size_t size)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Memoire::BlocMemoire bloc(address,size);
	machine_virtuelle.processeur()->ajoute_allocation_memoire(bloc);
}

void setlgg_machine_internal_processor_unregister_allocation(void *machine, const size_t address, const size_t size)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Memoire::BlocMemoire bloc(address,size);
	SetLgg::Machine::Memoire::Memoire::ListeBlocsMemoire liste = { bloc };
	machine_virtuelle.processeur()->ajoute_liberation_memoire(liste);
}

void* setlgg_machine_internal_processor_save(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Processeur::ProcesseurSP *vrai_processeur = new SetLgg::Machine::Processeur::ProcesseurSP(machine_virtuelle.processeur());
	return vrai_processeur;
}

void setlgg_machine_internal_processor_load(void *machine, void *processor)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Processeur::ProcesseurSP *vrai_processeur = reinterpret_cast<SetLgg::Machine::Processeur::ProcesseurSP*>(processor);
	machine_virtuelle.processeur() = SetLgg::Machine::Processeur::ProcesseurSP(*vrai_processeur);
	SetLgg::Machine::Processeur::GestionnaireInterruptions::enregistrement_signaux(machine_virtuelle.processeur());
}

void setlgg_machine_internal_processor_reset(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	machine_virtuelle.processeur() = SetLgg::Machine::Processeur::ProcesseurSP(new SetLgg::Machine::Processeur::Processeur());
	SetLgg::Machine::Processeur::GestionnaireInterruptions::enregistrement_signaux(machine_virtuelle.processeur());
}

void setlgg_machine_internal_processor_destruct(void *processor)
{
	if(not processor)
		return;
	SetLgg::Machine::Processeur::ProcesseurSP *vrai_processeur = reinterpret_cast<SetLgg::Machine::Processeur::ProcesseurSP*>(processor);
	delete vrai_processeur;
}

inline SetLgg_Machine_ReturnAddress setlgg_machine_memory_exist_alias_interne(SetLgg::Machine::Plugin::Interface& machine_virtuelle, const std::string& nom_instruction, const SetLgg_Machine_String alias)
{
	if(not alias._buffer)
	{
		throw PointeurAliasInvalideMemoire(nom_instruction);
	}
	std::string str(alias._buffer,alias._size);
	SetLgg::Machine::Memoire::Alias lalias(str);
	SetLgg_Machine_ReturnAddress r;
	try
	{
		auto adresse = machine_virtuelle.memoire()->resolution_alias(lalias);
		r._return = OK;
		r._address = adresse;
	}
	catch(SetLgg::Machine::Memoire::AliasInconnu& e)
	{
		r._return = INEXISTANT_ALIAS;
	}
	return r;
}

size_t setlgg_machine_memory_new(void *machine, const size_t nb_of_blocs, const SetLgg_Machine_MemoryTypedBloc blocs[])
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Memoire::ListePlacesMemoire places;
	for(auto itbloc=blocs ; itbloc!=(blocs+nb_of_blocs) ; ++itbloc)
	{
		SetLgg::Machine::Memoire::PlaceMemoire place(SetLgg::Machine::Plugin::Outils::convertit_type(__func__,machine_virtuelle.plugins(),itbloc->_type),itbloc->_size,SetLgg::Machine::Memoire::AliasSP());
		places.push_back(place);
	}
	auto bloc_alloue = machine_virtuelle.memoire()->allocation(places);
	machine_virtuelle.processeur()->ajoute_allocation_memoire(bloc_alloue);
	return bloc_alloue._adresse;
}

SetLgg_Machine_Return setlgg_machine_memory_delete(void *machine,const size_t nb_of_blocs, const SetLgg_Machine_MemoryBloc blocs[])
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Memoire::ListeBlocsMemoire blocs_memoire;
	for(auto itbloc=blocs ; itbloc!=(blocs+nb_of_blocs) ; ++itbloc)
	{
		blocs_memoire += SetLgg::Machine::Memoire::Memoire::BlocMemoire(itbloc->_address,itbloc->_size);
	}
	try
	{
		machine_virtuelle.memoire()->liberation(blocs_memoire);
		machine_virtuelle.processeur()->ajoute_liberation_memoire(blocs_memoire);
	}
	catch(SetLgg::Machine::Memoire::LiberationAdresseInvalide& e)
	{
		return INVALID_ADDRESS;
	}
	return OK;
}

inline SetLgg_Machine_ReturnValue setlgg_machine_memory_read_address_interne(SetLgg::Machine::Plugin::Interface& machine_virtuelle, const SetLgg::Machine::Memoire::Adresse& adresse, const SetLgg_Machine_ValueType& asked_type, const std::string& nom_instruction)
{
	SetLgg::Machine::Memoire::Type type_demande = SetLgg::Machine::Plugin::Outils::convertit_type(nom_instruction,machine_virtuelle.plugins(),asked_type);
	SetLgg_Machine_ReturnValue r;
	try
	{
		r._value = *(machine_virtuelle.memoire()->lecture(adresse,type_demande));
		r._return = OK;
	}
	catch(SetLgg::Machine::Memoire::LectureAdresseInvalide& e)
	{
		r._return = UNDEFINED_MEMORY;
	}
	catch(SetLgg::Machine::Memoire::LectureAdresseNonInitialisee& e)
	{
		r._return = UNINITIALISED_MEMORY;
	}
	catch(SetLgg::Machine::Memoire::LectureTypeInvalide& e)
	{
		r._return = INVALID_TYPE;
	}
	return r;
}

SetLgg_Machine_ReturnValue setlgg_machine_memory_read_address(void *machine, const size_t address, const SetLgg_Machine_ValueType asked_type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Adresse adresse(address);
	return setlgg_machine_memory_read_address_interne(machine_virtuelle,adresse,asked_type,__func__);
}

SetLgg_Machine_ReturnValue setlgg_machine_memory_read_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType asked_type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not alias._buffer)
	{
		throw PointeurAliasInvalideMemoire(__func__);
	}
	std::string str(alias._buffer,alias._size);
	SetLgg::Machine::Memoire::Alias lalias(str);
	try
	{
		auto adresse = machine_virtuelle.memoire()->resolution_alias(lalias);
		return setlgg_machine_memory_read_address_interne(machine_virtuelle,adresse,asked_type,__func__);
	}
	catch(SetLgg::Machine::Memoire::AliasInconnu& e)
	{
		SetLgg_Machine_ReturnValue r;
		r._return = INEXISTANT_ALIAS;
		return r;
	}
}

inline SetLgg_Machine_Return setlgg_machine_memory_write_address_interne(SetLgg::Machine::Plugin::Interface& machine_virtuelle, const SetLgg::Machine::Memoire::Adresse& adresse, const SetLgg_Machine_Value& value, const std::string& nom_instruction)
{
	try
	{
		SetLgg::Machine::Memoire::ValeurCSP valeur = SetLgg::Machine::Plugin::Outils::convertit_valeur(nom_instruction,machine_virtuelle.plugins(),value);
		machine_virtuelle.memoire()->ecriture(adresse,valeur);
		return OK;
	}
	catch(SetLgg::Machine::Memoire::EcritureAdresseInvalide& e)
	{
		return UNDEFINED_MEMORY;
	}
	catch(SetLgg::Machine::Memoire::EcritureTypeInvalide& e)
	{
		return INVALID_TYPE;
	}
}

SetLgg_Machine_Return setlgg_machine_memory_write_address(void *machine, const size_t address, const SetLgg_Machine_Value value)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Adresse adresse(address);
	return setlgg_machine_memory_write_address_interne(machine_virtuelle,adresse,value,__func__);
}

SetLgg_Machine_Return setlgg_machine_memory_write_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_Value value)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg_Machine_ReturnAddress ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	if(ra._return==OK)
	{
		SetLgg::Machine::Memoire::Adresse adresse(ra._address);
		return setlgg_machine_memory_write_address_interne(machine_virtuelle,adresse,value,__func__);
	}
	return ra._return;
}

SetLgg_Machine_Return setlgg_machine_memory_state_address(void *machine, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Adresse adresse(address);
	if(not machine_virtuelle.memoire()->adresse_valide(adresse))
		return UNDEFINED_MEMORY;
	if(not machine_virtuelle.memoire()->adresse_initialisee(adresse))
		return UNINITIALISED_MEMORY;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_memory_state_alias(void *machine, const SetLgg_Machine_String alias)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg_Machine_ReturnAddress ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	if(ra._return==OK)
	{
		SetLgg::Machine::Memoire::Adresse adresse(ra._address);
		if(not machine_virtuelle.memoire()->adresse_valide(adresse))
			return UNDEFINED_MEMORY;
		if(not machine_virtuelle.memoire()->adresse_initialisee(adresse))
			return UNINITIALISED_MEMORY;
		return OK;
	}
	return ra._return;
}

SetLgg_Machine_Return setlgg_machine_memory_link_alias(void *machine, const SetLgg_Machine_String alias, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not alias._buffer)
	{
		throw PointeurAliasInvalideMemoire(__func__);
	}
	std::string str(alias._buffer,alias._size);
	SetLgg::Machine::Memoire::Alias lalias(str);
	SetLgg::Machine::Memoire::Adresse adresse(address);
	if(machine_virtuelle.memoire()->ajoute_alias(lalias,adresse))
		return OK;
	else
		return ALIAS_ALREADY_EXISTS;
}

SetLgg_Machine_Return setlgg_machine_memory_unlink_alias(void *machine, const SetLgg_Machine_String alias)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not alias._buffer)
	{
		throw PointeurAliasInvalideMemoire(__func__);
	}
	std::string str(alias._buffer,alias._size);
	SetLgg::Machine::Memoire::Alias lalias(str);
	if(machine_virtuelle.memoire()->supprime_alias(lalias))
		return OK;
	else
		return INEXISTANT_ALIAS;

}

SetLgg_Machine_ReturnAddress setlgg_machine_memory_exist_alias(void *machine, const SetLgg_Machine_String alias)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	return setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
}

size_t setlgg_machine_memory_current(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto etat = machine_virtuelle.processeur()->etat_courant();
	SetLgg::Machine::Memoire::Adresse adresse = etat;
	return machine_virtuelle.adresse_memoire(adresse);
}

void setlgg_machine_internal_memory_current(void *machine, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	machine_virtuelle.processeur()->change_adresse_courante(address);
}

void setlgg_machine_internal_memory_truncate(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	machine_virtuelle.memoire()->tronque();
}

SetLgg_Machine_ReturnType setlgg_machine_internal_memory_read_type_address(void *machine, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg_Machine_ReturnType r;
	SetLgg::Machine::Memoire::Adresse adresse(address);
	auto type = machine_virtuelle.memoire()->type(adresse);
	if(type.first)
	{
		r._type = type.second;
		r._return = OK;
	}
	else
	{
		r._return = UNDEFINED_MEMORY;
	}
	return r;
}

SetLgg_Machine_ReturnType setlgg_machine_internal_memory_read_type_alias(void *machine, const SetLgg_Machine_String alias)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	SetLgg_Machine_ReturnType r;
	if(ra._return==OK)
	{
		SetLgg::Machine::Memoire::Adresse adresse(ra._address);
		auto type = machine_virtuelle.memoire()->type(adresse);
		if(type.first)
		{
			r._type = type.second;
			r._return = OK;
		}
		else
		{
			r._return = UNDEFINED_MEMORY;
		}
	}
	else
	{
		r._return = ra._return;
	}
	return r;
}

SetLgg_Machine_Return setlgg_machine_internal_memory_write_type_address(void *machine, const size_t address, const SetLgg_Machine_ValueType type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Adresse adresse(address);
	SetLgg::Machine::Memoire::Type type_interne = SetLgg::Machine::Plugin::Outils::convertit_type(__func__,machine_virtuelle.plugins(),type);
	if(machine_virtuelle.memoire()->type(adresse,type_interne))
		return OK;
	else
		return UNDEFINED_MEMORY;
}

SetLgg_Machine_Return setlgg_machine_internal_memory_write_type_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	if(ra._return==OK)
	{
		SetLgg::Machine::Memoire::Type type_interne = SetLgg::Machine::Plugin::Outils::convertit_type(__func__,machine_virtuelle.plugins(),type);
		SetLgg::Machine::Memoire::Adresse adresse(ra._address);
		if(machine_virtuelle.memoire()->type(adresse,type_interne))
			return OK;
		else
			return UNDEFINED_MEMORY;
	}
	return ra._return;
}

SetLgg_Machine_Return setlgg_machine_memory_uninitialise_address(void *machine, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::Adresse adresse(address);
	if(machine_virtuelle.memoire()->desinitialise(adresse))
		return OK;
	else
		return UNDEFINED_MEMORY;
}

SetLgg_Machine_Return setlgg_machine_memory_uninitialise_alias(void *machine, const SetLgg_Machine_String alias)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	if(ra._return==OK)
	{
		SetLgg::Machine::Memoire::Adresse adresse(ra._address);
		if(machine_virtuelle.memoire()->desinitialise(adresse))
			return OK;
		else
			return UNDEFINED_MEMORY;
	}
	return ra._return;
}

inline SetLgg_Machine_Return setlgg_machine_internal_memory_pointer_address_interne(SetLgg::Machine::Plugin::Interface& machine_virtuelle, const size_t address, const std::string& nom_instruction)
{
	SetLgg::Machine::Memoire::Adresse adresse(address);
	switch(machine_virtuelle.memoire()->type_pointe(adresse))
	{
		case SetLgg::Machine::Memoire::Memoire::TypePointe::NONDEFINI:
			return UNDEFINED_MEMORY;
		case SetLgg::Machine::Memoire::Memoire::TypePointe::NONINITIALISE:
			return UNINITIALISED_MEMORY;
		case SetLgg::Machine::Memoire::Memoire::TypePointe::POINTEUR:
			return IS_A_POINTER;
		case SetLgg::Machine::Memoire::Memoire::TypePointe::VALEUR:
			return IS_A_VALUE;
	}
	throw;
}

inline SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_solve_address_interne(SetLgg::Machine::Plugin::Interface& machine_virtuelle, const size_t address, const SetLgg_Machine_ValueType asked_type, const std::string& nom_instruction)
{
	SetLgg::Machine::Memoire::Adresse adresse(address);
	std::set<SetLgg::Machine::Memoire::Adresse> adresses_parcourues;
	while(machine_virtuelle.memoire()->type_pointe(adresse)==SetLgg::Machine::Memoire::Memoire::TypePointe::POINTEUR)
	{
		if(adresses_parcourues.find(adresse)!=adresses_parcourues.end())
		{
			SetLgg_Machine_ReturnValue rv;
			rv._return = POINTER_LOOP;
			return rv;
		}
		adresse = machine_virtuelle.memoire()->suivi(adresse);
	}
	SetLgg_Machine_ReturnValue rv;
	rv._return = setlgg_machine_internal_memory_pointer_address_interne(machine_virtuelle,adresse,nom_instruction);
	switch(rv._return)
	{
		case IS_A_VALUE:
		case IS_A_POINTER:
			{
				SetLgg::Machine::Memoire::Type type = SetLgg::Machine::Plugin::Outils::convertit_type(nom_instruction,machine_virtuelle.plugins(),asked_type);
				rv._value = *(machine_virtuelle.memoire()->lecture(adresse,type));
			}
			break;
		default:
			break;
	}
	return rv;
}

SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_solve_address(void *machine, const size_t address, const SetLgg_Machine_ValueType asked_type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	return setlgg_machine_internal_memory_solve_address_interne(machine_virtuelle,address,asked_type,__func__);
}

SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_solve_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType asked_type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	SetLgg_Machine_ReturnValue rv;
	if(ra._return==OK)
	{
		return setlgg_machine_internal_memory_solve_address_interne(machine_virtuelle,ra._address,asked_type,__func__);
	}
	rv._return = rv._return;
	return rv;
}

inline SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_limited_solve_address_interne(SetLgg::Machine::Plugin::Interface& machine_virtuelle, const unsigned int max, const size_t address, const SetLgg_Machine_ValueType asked_type, const std::string& nom_instruction)
{
	SetLgg::Machine::Memoire::Adresse adresse(address);
	unsigned int indice = max;
	while(indice--)
	{
		if(machine_virtuelle.memoire()->type_pointe(adresse)!=SetLgg::Machine::Memoire::Memoire::TypePointe::POINTEUR)
			break;
		adresse = machine_virtuelle.memoire()->suivi(adresse);
	}
	SetLgg_Machine_ReturnValue rv;
	rv._return = setlgg_machine_internal_memory_pointer_address_interne(machine_virtuelle,adresse,nom_instruction);
	switch(rv._return)
	{
		case IS_A_VALUE:
		case IS_A_POINTER:
			{
				SetLgg::Machine::Memoire::Type type = SetLgg::Machine::Plugin::Outils::convertit_type(nom_instruction,machine_virtuelle.plugins(),asked_type);
				rv._value = *(machine_virtuelle.memoire()->lecture(adresse,type));
			}
			break;
		default:
			break;
	}
	return rv;
}

SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_limited_solve_address(void *machine, const unsigned int max, const size_t address, const SetLgg_Machine_ValueType asked_type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	return setlgg_machine_internal_memory_limited_solve_address_interne(machine_virtuelle,max,address,asked_type,__func__);
}

SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_limited_solve_alias(void *machine, const unsigned int max, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType asked_type)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	SetLgg_Machine_ReturnValue rv;
	if(ra._return==OK)
	{
		return setlgg_machine_internal_memory_limited_solve_address_interne(machine_virtuelle,max,ra._address,asked_type,__func__);
	}
	rv._return = rv._return;
	return rv;
}


SetLgg_Machine_Return setlgg_machine_internal_memory_pointer_address(void *machine, const size_t address)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	return setlgg_machine_internal_memory_pointer_address_interne(machine_virtuelle,address,__func__);
}

SetLgg_Machine_Return setlgg_machine_internal_memory_pointer_alias(void *machine, const SetLgg_Machine_String alias)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	auto ra = setlgg_machine_memory_exist_alias_interne(machine_virtuelle,__func__,alias);
	if(ra._return==OK)
	{
		SetLgg::Machine::Memoire::Adresse adresse(ra._address);
		return setlgg_machine_internal_memory_pointer_address_interne(machine_virtuelle,ra._address,__func__);
	}
	return ra._return;
}

void* setlgg_machine_internal_memory_save(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::MemoireSP *vraie_memoire = new SetLgg::Machine::Memoire::MemoireSP(machine_virtuelle.memoire());
	return vraie_memoire;
}

void setlgg_machine_internal_memory_load(void *machine, void *memory)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::MemoireSP *vraie_memoire = reinterpret_cast<SetLgg::Machine::Memoire::MemoireSP*>(memory);
	machine_virtuelle.memoire() = SetLgg::Machine::Memoire::MemoireSP(*vraie_memoire);
}

void setlgg_machine_internal_memory_reset(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	machine_virtuelle.memoire() = SetLgg::Machine::Memoire::MemoireSP(new SetLgg::Machine::Memoire::Memoire());
}

void setlgg_machine_internal_memory_destruct(void *memory)
{
	if(not memory)
		return;
	SetLgg::Machine::Memoire::MemoireSP *vraie_memoire = reinterpret_cast<SetLgg::Machine::Memoire::MemoireSP*>(memory);
	delete vraie_memoire;
}

SetLgg_Machine_ReturnMemory setlgg_machine_internal_memory_compile(void *machine, const SetLgg_Machine_String code)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg_Machine_ReturnMemory rm;
	if(not code._buffer)
	{
		throw EntreeInvalide("memory code",__func__);
	}
	std::string str_code(code._buffer,code._size);
	SetLgg::Machine::Memoire::MemoireSP memoire;
	try
	{
		memoire = SetLgg::Machine::Memoire::Analyseur::Analyseur::analyse_chaine(str_code,machine_virtuelle.plugins());
		rm._memory = new SetLgg::Machine::Memoire::MemoireSP(memoire);
		rm._return = OK;
	}
	catch(SetLgg::Global::Exception::Compilation& e)
	{
		rm._memory = 0;
		rm._return = INVALID_CODE;
	}
	return rm;

}

SetLgg_Machine_ReturnProgram setlgg_machine_internal_program_compile(void *machine, const SetLgg_Machine_String code)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg_Machine_ReturnProgram rp;
	if(not code._buffer)
	{
		throw EntreeInvalide("program code",__func__);
	}
	std::string str_code(code._buffer,code._size);
	SetLgg::Machine::Plugin::Interface::Programme *programme = new SetLgg::Machine::Plugin::Interface::Programme;
	try
	{
		if(machine_virtuelle.debogueur())
		{
			programme->_debogueur = SetLgg::Machine::Debogueur::DebogueurSP(new SetLgg::Machine::Debogueur::Debogueur());
		}
		programme->_programme = SetLgg::Machine::Programme::ProgrammeSP(SetLgg::Machine::Programme::Analyseur::Analyseur::analyse_chaine(str_code,machine_virtuelle.plugins(),programme->_debogueur));
		rp._program = programme;
		rp._return = OK;
	}
	catch(SetLgg::Global::Exception::Compilation& e)
	{
		delete programme;
		rp._program = 0;
		rp._return = INVALID_CODE;
	}
	return rp;
}

void setlgg_machine_internal_program_destruct(void *program)
{
	if(not program)
	{
		throw EntreeInvalide("program",__func__);
	}
	SetLgg::Machine::Plugin::Interface::Programme *prog = reinterpret_cast<SetLgg::Machine::Plugin::Interface::Programme*>(program);
	delete prog;	
}

void* setlgg_machine_internal_program_save(void *machine)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Plugin::Interface::Programme* prog = new SetLgg::Machine::Plugin::Interface::Programme;
	prog->_programme = machine_virtuelle.programme();
	prog->_debogueur = machine_virtuelle.debogueur();
	return prog;
}

void setlgg_machine_internal_program_append(void *machine, const void *program)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not program)
	{
		throw EntreeInvalide("program",__func__);
	}
	const SetLgg::Machine::Plugin::Interface::Programme *prog = reinterpret_cast<const SetLgg::Machine::Plugin::Interface::Programme*>(program);
	SetLgg::Machine::Programme::Adresse adresse_terminale = machine_virtuelle.adresse_terminale();
	machine_virtuelle.programme()->ajout_programme(prog->_programme);
	machine_virtuelle.programme()->edition_liens();
	if(machine_virtuelle.debogueur())
	{
		machine_virtuelle.debogueur()->ajout_debogueur(prog->_debogueur,adresse_terminale);
	}
}

void setlgg_machine_internal_program_load(void *machine, void *program)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not program)
	{
		throw EntreeInvalide("program",__func__);
	}
	SetLgg::Machine::Plugin::Interface::Programme *prog = reinterpret_cast<SetLgg::Machine::Plugin::Interface::Programme*>(program);
	(prog->_programme)->edition_liens();
	machine_virtuelle.programme() = prog->_programme;
	machine_virtuelle.debogueur() = prog->_debogueur;
}

size_t setlgg_machine_internal_program_size(void *program)
{
	if(not program)
	{
		throw EntreeInvalide("program",__func__);
	}
	SetLgg::Machine::Plugin::Interface::Programme *prog = reinterpret_cast<SetLgg::Machine::Plugin::Interface::Programme*>(program);
	return prog->taille();
}

SetLgg_Machine_ReturnAddress setlgg_machine_internal_program_find_label(void *program, const SetLgg_Machine_String label)
{
	if(not program)
	{
		throw EntreeInvalide("program",__func__);
	}
	if(not label._buffer)
	{
		throw EntreeInvalide("label",__func__);
	}
	std::string str(label._buffer,label._size);
	SetLgg::Machine::Plugin::Interface::Programme *prog = reinterpret_cast<SetLgg::Machine::Plugin::Interface::Programme*>(program);
	auto resultat = prog->resolution_label(str);
	SetLgg_Machine_ReturnAddress retour;
	if(resultat.first)
	{
		retour._return = SetLgg_Machine_Return::OK;
		retour._address = SetLgg::Machine::Plugin::Interface::adresse_programme(resultat.second);
	}
	else
	{
		retour._return = SetLgg_Machine_Return::INEXISTANT_LABEL;
	}
	return retour;
}

SetLgg_Machine_Return setlgg_machine_stream_open_file(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String name)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not name._buffer)
	{
		throw EntreeInvalide("file name",__func__);
	}
	SetLgg::Machine::Flux::FluxSP flux;
	std::string nom(name._buffer,name._size);
	switch(mode)
	{
		case READONLY:
			flux = SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxFichierLectureSeule(nom));
			break;
		case WRITEONLY:
			flux = SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxFichierEcritureSeule(nom));
			break;
		case READWRITE:
			flux = SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxFichierLectureEcriture(nom));
			break;
		default:
			return INVALID_FILEMODE;
			break;
	}
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	try
	{
		machine_virtuelle.flux()->ouverture(nom_flux,flux);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxDejaOuvert& e)
	{
		return STREAM_ALREADY_EXISTS;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_open_tcp(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not local_ip._buffer)
	{
		throw EntreeInvalide("local ip address",__func__);
	}
	if(not local_port._buffer)
	{
		throw EntreeInvalide("local port",__func__);
	}
	if(not distant_ip._buffer)
	{
		throw EntreeInvalide("distant ip address",__func__);
	}
	if(not distant_port._buffer)
	{
		throw EntreeInvalide("distant port",__func__);
	}
	SetLgg::Machine::Flux::FluxSP flux;
	std::string str_local_ip(local_ip._buffer,local_ip._size);
	std::string str_local_port(local_port._buffer,local_port._size);
	std::string str_distant_ip(distant_ip._buffer,distant_ip._size);
	std::string str_distant_port(distant_port._buffer,distant_port._size);
	switch(mode)
	{
		case READONLY:
			flux = SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxSocketTCPServeurAttend(str_local_ip,str_local_port,str_distant_ip,str_distant_port));
			break;
		case WRITEONLY:
			flux = SetLgg::Machine::Flux::FluxSP(new SetLgg::Machine::Flux::FluxSocketTCPClient(str_local_ip,str_local_port,str_distant_ip,str_distant_port));
			break;
		default:
			return INVALID_FILEMODE;
			break;
	}
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	try
	{
		machine_virtuelle.flux()->ouverture(nom_flux,flux);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxDejaOuvert& e)
	{
		return STREAM_ALREADY_EXISTS;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_open_udp(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not local_port._buffer)
	{
		throw EntreeInvalide("local ip address",__func__);
	}
	if(not local_port._buffer)
	{
		throw EntreeInvalide("local port",__func__);
	}
	if(not distant_ip._buffer)
	{
		throw EntreeInvalide("distant ip address",__func__);
	}
	if(not distant_port._buffer)
	{
		throw EntreeInvalide("distant port",__func__);
	}
	std::string str_local_ip(local_ip._buffer,local_ip._size);
	std::string str_local_port(local_port._buffer,local_port._size);
	std::string str_distant_ip(distant_ip._buffer,distant_ip._size);
	std::string str_distant_port(distant_port._buffer,distant_port._size);
	SetLgg::Machine::Flux::FluxSP flux(new SetLgg::Machine::Flux::FluxSocketUDP(str_local_ip,str_local_port,str_distant_ip,str_distant_port));
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	try
	{
		machine_virtuelle.flux()->ouverture(nom_flux,flux);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxDejaOuvert& e)
	{
		return STREAM_ALREADY_EXISTS;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_close(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	try
	{
		machine_virtuelle.flux()->fermeture(nom_flux);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_close_read(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	try
	{
		machine_virtuelle.flux()->fermeture_lecture(nom_flux);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_close_write(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	try
	{
		machine_virtuelle.flux()->fermeture_ecriture(nom_flux);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_ReturnValue setlgg_machine_stream_read(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String format, const SetLgg_Machine_ReadMode mode, const size_t size)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not format._buffer)
	{
		throw EntreeInvalide("format",__func__);
	}
	SetLgg::Machine::Flux::DonneesLues donnees;
	SetLgg_Machine_ReturnValue retour;
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	try
	{
		switch(mode)
		{
			case SetLgg_Machine_ReadMode::AVAILABLE:
				donnees = machine_virtuelle.flux()->lecture(nom_flux,SetLgg::Machine::Flux::TypeLecture::Type::DISPONIBLE);
				break;
			case SetLgg_Machine_ReadMode::LINE:
				donnees = machine_virtuelle.flux()->lecture(nom_flux,SetLgg::Machine::Flux::TypeLecture::Type::LIGNE);
				break;
			case SetLgg_Machine_ReadMode::ALL:
				donnees = machine_virtuelle.flux()->lecture(nom_flux,SetLgg::Machine::Flux::TypeLecture::Type::TOUT);
				break;
			case SetLgg_Machine_ReadMode::SIZE:
				donnees = machine_virtuelle.flux()->lecture(nom_flux,SetLgg::Machine::Flux::TypeLecture(SetLgg::Machine::Flux::TypeLecture::Type::TAILLE,size));
				break;
			default:
				retour._return = INVALID_READMODE;
				return retour;
		}
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		retour._return = INVALID_STREAM;
		return retour;
	}
	catch(SetLgg::Machine::Flux::LectureFluxImpossible& e)
	{
		retour._return = UNREADABLE_STREAM;
		return retour;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		retour._return = INVALID_STREAM_OPERATION;
		return retour;
	}
	
	if(donnees)
	{
		retour._return = END_OF_STREAM;
		return retour;
	}

	SetLgg::Machine::Programme::Format formatteur(std::string(format._buffer,format._size));
	try
	{
		SetLgg::Machine::Memoire::ValeurSP valeur = formatteur(donnees);
		if(not valeur)
		{
			retour._return = INVALID_DATA;
			return retour;
		}
		retour._value = *valeur;
		retour._return = OK;
		return retour;
	}
	catch(SetLgg::Machine::Programme::FormatInvalide& e)
	{
		retour._return = INVALID_FORMAT;
		return retour;
	}
	catch(SetLgg::Machine::Programme::DonneesInvalidesPourFormat& e)
	{
		retour._return = INVALID_DATA_FORMAT;
		return retour;
	}
}

SetLgg_Machine_Return setlgg_machine_stream_write(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String format, const SetLgg_Machine_Value *value)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not format._buffer)
	{
		throw EntreeInvalide("format",__func__);
	}
	if(not value)
	{
		throw EntreeInvalide("value",__func__);
	}
	std::string donnees;
	try
	{
		SetLgg::Machine::Memoire::ValeurSP valeur = SetLgg::Machine::Plugin::Outils::convertit_valeur(__func__,machine_virtuelle.plugins(),*value);
		if(not valeur)
		{
			return INVALID_DATA;
		}
		SetLgg::Machine::Programme::Format formatteur(std::string(format._buffer,format._size));
		donnees = formatteur(valeur);
	}
	catch(SetLgg::Machine::Programme::FormatInvalide& e)
	{
		return INVALID_FORMAT;
	}
	catch(SetLgg::Machine::Programme::DonneesInvalidesPourFormat& e)
	{
		return INVALID_DATA_FORMAT;
	}
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		machine_virtuelle.flux()->ecriture(nom_flux,donnees);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::EcritureFluxImpossible& e)
	{
		return UNWRITABLE_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_wait(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String connector)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not connector._buffer)
	{
		throw EntreeInvalide("connector",__func__);
	}
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		SetLgg::Machine::Memoire::NomFlux nom_connecteur(std::string(connector._buffer,connector._size));
		machine_virtuelle.flux()->attente(nom_flux,nom_connecteur);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxDejaOuvert& e)
	{
		return STREAM_ALREADY_EXISTS;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_ReturnValue setlgg_machine_stream_seek(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_SeekMode mode, const long int position)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg_Machine_ReturnValue retour;
	retour._return = OK;
	retour._value._type._type = VALUE_INTEGER;
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		switch(mode)
		{
			case SetLgg_Machine_SeekMode::ABSOLUTE:
				retour._value._value._integer = machine_virtuelle.flux()->cherche(nom_flux,SetLgg::Machine::Flux::TypeCherche::ABSOLUE,position);
				break;
			case SetLgg_Machine_SeekMode::OFFSET:
				retour._value._value._integer = machine_virtuelle.flux()->cherche(nom_flux,SetLgg::Machine::Flux::TypeCherche::RELATIF,position);
				break;
			case SetLgg_Machine_SeekMode::END:
				retour._value._value._integer = machine_virtuelle.flux()->cherche(nom_flux,SetLgg::Machine::Flux::TypeCherche::FIN,position);
				break;
			default:
				retour._return = INVALID_SEEKMODE;
				return retour;
		}
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		retour._return = INVALID_STREAM;
		return retour;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		retour._return = INVALID_STREAM_OPERATION;
		return retour;
	}
	
	return retour;
}

SetLgg_Machine_Return setlgg_machine_stream_readable(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		if(not machine_virtuelle.flux()->lisible(nom_flux))
			return UNREADABLE_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_writable(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		if(not machine_virtuelle.flux()->inscriptible(nom_flux))
			return UNWRITABLE_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_seekable(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		if(not machine_virtuelle.flux()->cherchable(nom_flux))
			return UNSEEKABLE_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_send(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String name, const size_t address, const size_t size)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not name._buffer)
	{
		throw EntreeInvalide("block name",__func__);
	}
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		std::string nom_bloc(name._buffer,name._size);
		SetLgg::Machine::Memoire::Memoire::BlocMemoire bloc_memoire(address,size);
		SetLgg::Machine::Programme::InstructionEnvoi::execution_interne(machine_virtuelle._machine,nom_flux,nom_bloc,bloc_memoire);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	catch(SetLgg::Global::Exception::Execution& e)
	{
		return INVALID_DATA;
	}
	return OK;
}

SetLgg_Machine_Return setlgg_machine_stream_receive(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String name, const size_t address, const size_t size)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not name._buffer)
	{
		throw EntreeInvalide("block name",__func__);
	}
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		std::string nom_bloc(name._buffer,name._size);
		SetLgg::Machine::Memoire::Memoire::BlocMemoireSP bloc_memoire(new SetLgg::Machine::Memoire::Memoire::BlocMemoire(address,size));
		SetLgg::Machine::Memoire::ValeurSP retour = SetLgg::Machine::Programme::InstructionReception::execution_interne(machine_virtuelle._machine,nom_flux,nom_bloc,bloc_memoire);
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		return INVALID_STREAM;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		return INVALID_STREAM_OPERATION;
	}
	catch(SetLgg::Global::Exception::Execution& e)
	{
		return INVALID_DATA;
	}
	return OK;
}

SetLgg_Machine_ReturnAddress setlgg_machine_stream_receive_allocate(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String name)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not name._buffer)
	{
		throw EntreeInvalide("block name",__func__);
	}
	try
	{
		SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
		std::string nom_bloc(name._buffer,name._size);
		SetLgg::Machine::Memoire::Memoire::BlocMemoireSP bloc_memoire;
		SetLgg::Machine::Memoire::ValeurSP valeur = SetLgg::Machine::Programme::InstructionReception::execution_interne(machine_virtuelle._machine,nom_flux,nom_bloc,bloc_memoire);
		SetLgg::Machine::Memoire::AdresseSP adresse = std::dynamic_pointer_cast<SetLgg::Machine::Memoire::Adresse>(valeur);
		SetLgg_Machine_ReturnAddress retour;
		retour._return = OK;
		retour._address = *adresse;
		return retour;
	}
	catch(SetLgg::Machine::Flux::FluxInexistant& e)
	{
		SetLgg_Machine_ReturnAddress retour;
		retour._return = INVALID_STREAM;
		retour._address = 0;
		return retour;
	}
	catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
	{
		SetLgg_Machine_ReturnAddress retour;
		retour._return = INVALID_STREAM_OPERATION;
		retour._address = 0;
		return retour;
	}
	catch(SetLgg::Global::Exception::Execution& e)
	{
		SetLgg_Machine_ReturnAddress retour;
		retour._return = INVALID_DATA;
		retour._address = 0;
		return retour;
	}
}

SetLgg_Machine_Return setlgg_machine_internal_stream_register_file(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String name)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not name._buffer)
	{
		throw EntreeInvalide("file name",__func__);
	}
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux;
	switch(mode)
	{
		case READONLY:
		case WRITEONLY:
		case READWRITE:
			flux = SetLgg::Machine::Plugin::Interface::ouvre_flux_fichier(fd,name,mode);
			break;
		default:
			return INVALID_FILEMODE;
			break;
	}
	flux->descripteur_debug(nom_flux);
	if(not machine_virtuelle.flux()->ajoute(nom_flux,flux))
		return STREAM_ALREADY_EXISTS;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_internal_stream_register_tcp(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not local_ip._buffer)
	{
		throw EntreeInvalide("local ip address",__func__);
	}
	if(not local_port._buffer)
	{
		throw EntreeInvalide("local port",__func__);
	}
	if(not distant_ip._buffer)
	{
		throw EntreeInvalide("distant ip address",__func__);
	}
	if(not distant_port._buffer)
	{
		throw EntreeInvalide("distant port",__func__);
	}
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux;
	switch(mode)
	{
		case READONLY:
		case WRITEONLY:
			flux = SetLgg::Machine::Plugin::Interface::ouvre_flux_socketTCP(fd,mode,local_ip,local_port,distant_ip,distant_port);
			break;
		default:
			return INVALID_FILEMODE;
			break;
	}
	flux->descripteur_debug(nom_flux);
	if(not machine_virtuelle.flux()->ajoute(nom_flux,flux))
		return STREAM_ALREADY_EXISTS;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_internal_stream_register_udp(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not local_ip._buffer)
	{
		throw EntreeInvalide("local ip address",__func__);
	}
	if(not local_port._buffer)
	{
		throw EntreeInvalide("local port",__func__);
	}
	if(not distant_port._buffer)
	{
		throw EntreeInvalide("distant ip address",__func__);
	}
	if(not distant_port._buffer)
	{
		throw EntreeInvalide("distant port",__func__);
	}
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux(SetLgg::Machine::Plugin::Interface::ouvre_flux_socketUDP(fd,local_ip,local_port,distant_ip,distant_port));
	flux->descripteur_debug(nom_flux);
	if(not machine_virtuelle.flux()->ajoute(nom_flux,flux))
		return STREAM_ALREADY_EXISTS;
	return OK;
}

SetLgg_Machine_Return setlgg_machine_internal_stream_register_pipe(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd_read, const unsigned int fd_write)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux(new SetLgg::Machine::Flux::FluxPaireTubesAnonymes(fd_read,fd_write));
	flux->descripteur_debug(nom_flux);
	if(not machine_virtuelle.flux()->ajoute(nom_flux,flux))
		return STREAM_ALREADY_EXISTS;
	return OK;
}

SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_buffer_get(void *machine, const SetLgg_Machine_InOutReference stream, const size_t size)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux = (*machine_virtuelle.flux())[nom_flux];
	if(flux)
	{
		try
		{
			std::string chaine=SetLgg::Machine::Plugin::Interface::tampon_depile_flux(flux,size);
			SetLgg_Machine_ReturnValue retour;
			retour._return = OK;
			retour._value._initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
			retour._value._type._type = VALUE_STRING;
			retour._value._value._string = { SetLgg::Global::Chaine::duplique(chaine.c_str(),chaine.size()), chaine.size() };
			return retour;
		}
		catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
		{
			SetLgg_Machine_ReturnValue retour;
			retour._return = INVALID_STREAM_OPERATION;
			return retour;
		}
	}
	else
	{
		SetLgg_Machine_ReturnValue retour;
		retour._return = INVALID_STREAM;
		return retour;
	}
}

/*
SetLgg_Machine_Return setlgg_machine_internal_stream_buffer_put(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String string)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not string._buffer)
	{
		throw EntreeInvalide("string",__func__);
	}
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux = (*machine_virtuelle.flux())[nom_flux];
	if(flux)
	{
		std::string str(string._buffer,string._size);
		try
		{
			SetLgg::Machine::Plugin::Interface::tampon_empile_flux(flux,str);
			return OK;
		}
		catch(SetLgg::Machine::Flux::FluxOperationInvalide& e)
		{
			return INVALID_STREAM_OPERATION;
		}
	}
	else
	{
		return INVALID_STREAM;
	}
}
*/

SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_read_id(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux = (*machine_virtuelle.flux())[nom_flux];
	if(flux)
	{
		SetLgg_Machine_ReturnValue retour;
		retour._return = OK;
		retour._value._initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
		retour._value._type._type = VALUE_INTEGER;
		retour._value._value._integer = SetLgg::Machine::Plugin::Interface::descripteur_lecture(flux);
		return retour;
	}
	else
	{
		SetLgg_Machine_ReturnValue retour;
		retour._return = INVALID_STREAM;
		return retour;
	}
}

SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_write_id(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux = (*machine_virtuelle.flux())[nom_flux];
	if(flux)
	{
		SetLgg_Machine_ReturnValue retour;
		retour._return = OK;
		retour._value._initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
		retour._value._type._type = VALUE_INTEGER;
		retour._value._value._integer = SetLgg::Machine::Plugin::Interface::descripteur_ecriture(flux);
		return retour;
	}
	else
	{
		SetLgg_Machine_ReturnValue retour;
		retour._return = INVALID_STREAM;
		return retour;
	}
}

SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_name(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux = (*machine_virtuelle.flux())[nom_flux];
	if(flux)
	{
		std::string nom = nom_flux;
		SetLgg_Machine_ReturnValue retour;
		retour._return = OK;
		retour._value._initialisation = SetLgg_Machine_ValueInitialisation::INITIALISED_VALUE;
		retour._value._type._type = VALUE_STRING;
		retour._value._value._string._buffer = SetLgg::Global::Chaine::duplique(nom.c_str(),nom.size()+1);
		retour._value._value._string._size=nom.size();
		return retour;
	}
	else
	{
		SetLgg_Machine_ReturnValue retour;
		retour._return = INVALID_STREAM;
		return retour;
	}
}

SetLgg_Machine_Return setlgg_machine_internal_stream_nature(void *machine, const SetLgg_Machine_InOutReference stream)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	SetLgg::Machine::Memoire::NomFlux nom_flux = SetLgg::Machine::Plugin::Outils::convertit_nom_flux(__func__,stream);
	SetLgg::Machine::Flux::FluxSP flux = (*machine_virtuelle.flux())[nom_flux];
	if(flux)
	{
		if(flux->media_actif())
		{
			return ACTIVE_STREAM;
		}
		else
		{
			return PASSIVE_STREAM;
		}
	}
	else
	{
		return INVALID_STREAM;
	}
}

SetLgg_Machine_ReturnRegex setlgg_machine_internal_regex_cache(void *machine, const SetLgg_Machine_String regex)
{
	SetLgg::Machine::Plugin::Interface machine_virtuelle(machine,__func__);
	if(not regex._buffer)
	{
		throw EntreeInvalide("regex",__func__);
	}
	SetLgg_Machine_ReturnRegex retour;
	try
	{
		std::string strregex(regex._buffer,regex._size);
		retour._regex = SetLgg::Machine::Memoire::Chaine::cache_regex(strregex);
		retour._return = OK;
		retour._error_message = 0; 
	}
	catch(std::string& erreur)
	{
		retour._regex = 0;
		retour._return = INVALID_REGEX;
		retour._error_message = ::setlgg_machine_value_new_string(erreur.c_str());
	}
	return retour;
}

}
