/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the Lesser GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef __SETLGG_MACHINE_PLUGIN_INTERFACE_H__
#define __SETLGG_MACHINE_PLUGIN_INTERFACE_H__

#include <regex.h>

/*
 * Setlgg machine is a virtual machine running a language merging assembly ideas and high level instructions.
 *
 * This file contains all callbacks to control the machine from the plugin code written in C or C++.
 *
 * The plugins are shared libraries that export some specific functions:
 * - void plugin_configuration(void *plugin_handler);
 *   This function shall be defined to configure the plugin. See the setlgg_machine_plugin_register function.
 *
 * - For each user type declared in the plugin, four functions need to be defined:
 *   	1) char *setlgg_machine_<plugin name>_type_<user type name>_serialise(const void *handler);
 *            This function takes a value from the specified user type name and generates a nul terminated
 *            string allocated with malloc (even in C++).
 *   	2) void *setlgg_machine_<plugin name>_type_<user type name>_unserialise(const char *stream);
 *   	      This function is the reverse operation of the serialize one.
 *   	3) void  setlgg_machine_<plugin name>_type_<user type name>_delete(void *handler);
 *   	      This function takes a value from the specified user type and free all allocated memory
 *   	      associated to it.
 *   	4) void *setlgg_machine_<plugin name>_type_<user type name>_copy(const void *handler);
 *   	      This last function only makes a copy of the value taken as a parameter.
 *
 * - For each instruction declared in the plugin, one function need to be defined:
 *   	1) SetLgg_Machine_Value* setlgg_machine_<plugin name>_instruction_<instruction name>(void *machine, unsigned long int argc, SetLgg_Machine_Value argv[]);
 *   	      This function takes a machine handler that need to be passed to the callbach functions described below. Then, it takes
 *   	      the arguments that are computed from the program.
 */

#if defined __cplusplus
extern "C"
{
#endif
	/*
	 * Those structures are defined to communicate values between the virtual machine and the plugin code.
	 *
	 * The virtual machine only supports six basic types:
	 * - integer (long int)
	 * - string (struct)
	 * - boolean (unsigned char)
	 * - address (size_t)
	 * - input/output reference (struct)
	 * - user defined type (void *)
	 */

	typedef struct string
	{
		const char *_buffer;
		size_t _size;	// The size correspond to the size of the string. The final \0 is not included if one is present.
		                // From the plugin instruction parameters, you can always considers that the \0 character is present.
				// The functions below and the virtual machine will never assume that the character \0 is present.
	} SetLgg_Machine_String;

	/*
	 * The two following structures represent the type of the value.
	 */
	typedef enum {VALUE_INTEGER, VALUE_STRING, VALUE_BOOLEAN, VALUE_POINTER, VALUE_INOUTREF, VALUE_USERTYPE, VALUE_ALL} SetLgg_Machine_ValueTypeCategory;

	typedef struct valuetype
	{
		SetLgg_Machine_ValueTypeCategory _type; // Type category
		SetLgg_Machine_String  _user_name;       // Name of the user type. Only used when _type is VALUE_USERTYPE and undefined otherwise.
	} SetLgg_Machine_ValueType;

	/*
	 * This structure is defined to convey values between the core of the virtual machine and the plugin functions.
	 */
	typedef enum {IOR_STDIN, IOR_STDOUT, IOR_STDERR, IOR_CUSTOM} SetLgg_Machine_InOutReferenceType;
	typedef struct inoutref
	{
		SetLgg_Machine_InOutReferenceType _type;
		SetLgg_Machine_String _name;
	} SetLgg_Machine_InOutReference;

	typedef enum {NULL_VALUE, INITIALISED_VALUE} SetLgg_Machine_ValueInitialisation;
	typedef struct value
	{
		SetLgg_Machine_ValueType _type;                     // the type of the value
		SetLgg_Machine_ValueInitialisation _initialisation; // flag indicating whether a value is present
		union
		{
			long int _integer;                          // valid only if the type is VALUE_INTEGER
			SetLgg_Machine_String _string;              // valid only if the type is VALUE_STRING
			unsigned char _boolean;                     // valid only if the type is VALUE_BOOLEAN
			size_t _pointer;                            // valid only if the type is VALUE_POINTER
			SetLgg_Machine_InOutReference _inoutref;    // valid only if the type is VALUE_INOUTREF
			void *_user_type;                           // valid only if the type is VALUE_USERTYPE
		} _value;                                           // valid only if _initialisation is equal to INITIALISED_VALUE
	} SetLgg_Machine_Value;

	/*
	 * This enumeration represents what is the general result of the following functions.
	 * The codes are categorised by the object in the virtual machine on which they refer.
	 * Most of the codes are error codes. The last one is returned on success.
	 */
	typedef enum {
		INEXISTANT_LABEL, INVALID_INTERRUPTION, INEXISTANT_CALLSTACK_FRAME, FRAME_NOT_FOUND, // processor
		INVALID_CODE, // programm
		INVALID_ADDRESS, INVALID_TYPE, INEXISTANT_ALIAS, UNDEFINED_MEMORY, UNINITIALISED_MEMORY, ALIAS_ALREADY_EXISTS, IS_A_POINTER, IS_A_VALUE, POINTER_LOOP, // memory
		INVALID_STREAM, INVALID_FILEMODE, INVALID_READMODE, INVALID_SEEKMODE, INVALID_STREAM_OPERATION, UNREADABLE_STREAM, UNWRITABLE_STREAM, UNSEEKABLE_STREAM, STREAM_ALREADY_EXISTS, INVALID_FORMAT, INVALID_DATA_FORMAT, INVALID_DATA, END_OF_STREAM, ACTIVE_STREAM, PASSIVE_STREAM, //stream
		INVALID_REGEX, CHECKPOINT_NOT_REACHED, // miscellaneous
		VOID, OK } SetLgg_Machine_Return;

	/*
	 * The following structures are designed to return a code and/or a value.
	 */
	typedef struct returntype
	{
		SetLgg_Machine_Return _return;
		SetLgg_Machine_ValueType _type;
	} SetLgg_Machine_ReturnType;

	typedef struct returnvalue
	{
		SetLgg_Machine_Return _return;
		SetLgg_Machine_Value _value;
	} SetLgg_Machine_ReturnValue;

	typedef struct returnaddress
	{
		SetLgg_Machine_Return _return;
		size_t _address;
	} SetLgg_Machine_ReturnAddress;

	typedef struct returnprogram
	{
		SetLgg_Machine_Return _return;
		void *_program;
	} SetLgg_Machine_ReturnProgram;

	typedef struct returnmemory
	{
		SetLgg_Machine_Return _return;
		void *_memory;
	} SetLgg_Machine_ReturnMemory;

	typedef struct returnregex
	{
		SetLgg_Machine_Return _return;
		SetLgg_Machine_Value *_error_message;
		regex_t *_regex;
	} SetLgg_Machine_ReturnRegex;

#define SETLGG_MACHINE_EXTRACT_USERTYPE(type,variable,pointer)	type variable = *reinterpret_cast< type *>(pointer)
#define SETLGG_MACHINE_EXTRACT_USERTYPE_PARAMETER(type,variable,param_list,index)	type variable = *reinterpret_cast< type *>(param_list [index]._value._user_type)
#define SETLGG_MACHINE_VALUE_TO_CPP_STRING(value) std::string(value._value._string._buffer,value._value._string._size)
#define SETLGG_MACHINE_VALUE_PTR_TO_CPP_STRING(value) std::string(value->_value._string._buffer,value->_value._string._size)
#define SETLGG_MACHINE_VARIABLE_VALUE_TO_CPP_STRING(variable,value) std::string variable(value._value._string._buffer,value._value._string._size)
#define SETLGG_MACHINE_VARIABLE_VALUE_PTR_TO_CPP_STRING(variable,value) std::string variable(value->_value._string._buffer,value->_value._string._size)

// plugin
	/*
	 * This function shall be called from the plugin_configuration function.
	 *
	 * The simplest plugin_configuration function should look like:

void plugin_configuration(void *plugin_handler)
{
	setlgg_machine_plugin_register(plugin_handler,
		"<plug in configuration>"
	);
}

	 * where the plugin configuration is a text following this syntax:
	 * PLUGIN <plugin name>
	 * DEPENDS ON 
	 * 	PLUGIN <another plugin name>
	 * 	TYPE <user type name>
	 * 	TYPE <user type name> IN <another plugin name>
	 *	INSTRUCTION <instruction name>
	 *	INSTRUCTION <instruction name> IN <another plugin name>
	 * END
	 * CONTAINS
	 * 	TYPE <user type name>
	 * 	INSTRUCTION <instruction name> INT STR BLN PTR USR(<user type name>) ADDRESS RETURN
	 * 	INSTRUCTION <instruction name> INT STR BLN PTR USR(<user type name>) ADDRESS NO RETURN
	 * END
	 * 
	 * The section DEPENDS ON ... END is optional and contains the dependencies of the plugin towards other definitions.
	 * All directives listed in the DEPENDS ON ... END and CONTAINS ... END can be used in all orders, and be present zero, one or several times.
	 *
	 * The dependencies directives:
	 * PLUGIN <another plugin name>: this plugin needs another plugin to be loaded
	 * TYPE <user type name>: this plugin needs this user type to be loaded
	 * TYPE <user type name> IN <another plugin name>: this plugin needs a specific implementation of a user type to be loaded
	 * INSTRUCTION <instruction name>: this plugin needs this instruction to be loaded
	 * INSTRUCTION <instruction name> IN <another plugin name>: this plugin needs a specific implementation of this instruction to be loaded
	 *
	 * The declaration directives:
	 * TYPE <user type name>: declare a new user type. The four functions handling the values of this type shall be defined
	 * INSTRUCTION <instruction name> INT STR BLN PTR USR(<user type name>) ADDRESS RETURN: declare a new instruction. The function that will be called by the virtual machine shall be defined.
	 *	This instruction is marked as returning a value, and shall return one. The SetLgg_Machine_Value* passed to the virtual machine must be valid.
	 * INSTRUCTION <instruction name> INT STR BLN PTR USR(<user type name>) ADDRESS NO RETURN: same as above, but in this case the function will never return a value. The SetLgg_Machine_Value* returned to
	 * 	the virtual machine must be 0.
	 * The list of parameters INT STR BLN PTR USR(<user type name>) ADDRESS can be empty, each type/address keyword can be used zero, one or several times. This list specifies the mandatory parameters
	 * checked by the virtual machine before calling the callback function of the plugin. Extra parameters are not checked.
	 */
	void setlgg_machine_plugin_register(void *plugin_handler, const char *plugin_configuration);


// machine generic
	/*
	 * This function raises to the virtual machine an error condition.
	 * The interruption (signal name extracted from the signal.h header) let the programm have a fallback code for this error.
	 */
	SetLgg_Machine_Return setlgg_machine_error(void *machine, const unsigned int interruption, const char *error_message);
	SetLgg_Machine_Return setlgg_machine_error_string(void *machine, const unsigned int interruption, const SetLgg_Machine_String error_message);

	/*
	 * This function shut the machine down.
	 * If the core_dump is not null and a core file has been specified, a file that represents the virtual machine state is generated.
	 * If the memory_dump is not null, the memory is dumped into the memory file provided in input.
	 */
	void setlgg_machine_machine_shutdown(void *machine, const unsigned char core_dump, const unsigned char memory_dump, const unsigned int return_code);

// value
	/* 
	 * Those functions are some recommended helpers to handle the SetLgg_Machine_Value structure.
	 */

	/*
	 * All the following ones are designed to create a value from all supported types, setting the value type correctly.
	 * When a pointer is provided to those functions, it has to be not NULL and set to a valid memory address.
	 * Please note that the handler of the setlgg_machine_value_new_usertype function need to be allocated on the heap, and no copy is done by the function.
	 */
	SetLgg_Machine_Value* setlgg_machine_value_new_null_integer();
	SetLgg_Machine_Value* setlgg_machine_value_new_null_string();
	SetLgg_Machine_Value* setlgg_machine_value_new_null_boolean();
	SetLgg_Machine_Value* setlgg_machine_value_new_null_pointer();
	SetLgg_Machine_Value* setlgg_machine_value_new_null_inoutref();
	SetLgg_Machine_Value* setlgg_machine_value_new_null_usertype(const char *name);
	SetLgg_Machine_Value* setlgg_machine_value_new_null_usertype_string(const SetLgg_Machine_String name);
	SetLgg_Machine_Value* setlgg_machine_value_new_integer(const long int integer);
	SetLgg_Machine_Value* setlgg_machine_value_new_string(const char *string);
	SetLgg_Machine_Value* setlgg_machine_value_new_sized_string(const char *string, const size_t size);
	SetLgg_Machine_Value* setlgg_machine_value_new_boolean(const unsigned char boolean);
	SetLgg_Machine_Value* setlgg_machine_value_new_pointer(const size_t address);
	SetLgg_Machine_Value* setlgg_machine_value_new_inoutref(const SetLgg_Machine_InOutReferenceType type, const char *name);
	SetLgg_Machine_Value* setlgg_machine_value_new_inoutref_string(const SetLgg_Machine_InOutReferenceType type, const SetLgg_Machine_String name);
	SetLgg_Machine_Value* setlgg_machine_value_new_usertype(const char *name, void *handler);
	SetLgg_Machine_Value* setlgg_machine_value_new_usertype_string(const SetLgg_Machine_String name, void *handler);
	/*
	 * This function delete a value created from the functions above.
	 * Please note that this function shall not be used on instruction parameters!
	 * This function does not free the user type values, this pointer is under your responsability.
	 */
	void setlgg_machine_value_delete(SetLgg_Machine_Value *value);
	/*
	 * This function delete only the content of a value created from the functions above.
	 * Please note that this function shall not be used on instruction parameters!
	 * This function does not free the user type values, this pointer is under your responsability.
	 */
	void setlgg_machine_value_clear(SetLgg_Machine_Value *value);
	/*
	 * This function shall be called on any value returned by the functions below.
	 */
	void setlgg_machine_return_value_delete(SetLgg_Machine_ReturnValue *return_value);
	/*
	 * This function returns OK if the value is initialised or UNINITIALISED_MEMORY if the value is null.
	 * If the value itself is NULL, the function returns UNDEFINED_MEMORY.
	 */
	SetLgg_Machine_Return setlgg_machine_value_initialised(SetLgg_Machine_Value* value);


	/*
	 * All the functions below are virtual machine accessors.
	 * The first parameter void *machine is the one provided as first parameter to the
	 * instruction callbacks.
	 * All functions raise an error condition when a parameter is not valid, stopping the
	 * virtual machine.
	 */

// processor
	/*
	 * Those functions are high level accessors to the virtual machine processor.
	 */

	/*
	 * This function return OK whether the label exists.
	 * It returns INEXISTANT_LABEL otherwise.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_exist_label(void *machine, const SetLgg_Machine_String label);

	/*
	 * This function jumps to the specified label and returns OK.
	 * It returns INEXISTANT_LABEL when the label does not exist.
	 * This function does the action of the :goto instruction.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_jump_label(void *machine, const SetLgg_Machine_String label);

	/*
	 * This function saves the current processor state and jumps to the specified label
	 * and returns OK. It returns INEXISTANT_LABEL when the label does not exist.
	 * This function does the action of the :call instruction.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_call_label(void *machine, const SetLgg_Machine_String label, const size_t parameter_address);

	/*
	 * This function restores the last save processor state.
	 * This function does the action of the :return instruction without argument.
	 * If the delete_memory is not zero, the memory allocated at the current frame is deleted
	 */
	void setlgg_machine_processor_return(void *machine, const unsigned int delete_memory);

	/*
	 * This function stacks an interruption to be processed after the current instruction.
	 * SetLgg_Machine_Return OK if the interruption has been stacked, or INVALID_INTERRUPTION if the interruption number is incorrect.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_interrupt(void *machine, const unsigned int interruption);

	/*
	 * This function registers a subroutine for a specific interruption.
	 * The value returned is OK if the callback is registered, INEXISTANT_LABEL when the label does not exist and INVALID_INTERRUPTION
	 * if the interruption number is incorrect.
	 * This function does the action of the :interruption instruction with two parameters.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_interruption_register_label(void *machine, const unsigned int interruption, const SetLgg_Machine_String label);

	/*
	 * This function unregisters a subroutine for a specific interruption.
	 * The value returned is OK if the callback is registered and INVALID_INTERRUPTION if the interruption number is incorrect.
	 * This function does the action of the :interruption instruction with one parameter.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_interruption_unregister(void *machine, const unsigned int interruption);

	/*
	 * This function adds a checkpoint on the current frame.
	 * The value returned is OK if the checkpoint has been added correctly.
	 * This function does the same action of the :checkpoint with one parameter or :checkpoint with the boolean set to TRUE.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_checkpoint_add(void *machine, const SetLgg_Machine_String checkpoint);

	/*
	 * This function removes a checkpoint on the current frame.
	 * The value returned is OK if the checkpoint has been removed correctly.
	 * This function does the same action of the :checkpoint with the boolean set to FALSE.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_checkpoint_remove(void *machine, const SetLgg_Machine_String checkpoint);

	/*
	 * This function checks whether a checkpoint has been set on the current frame.
	 * The value returned is CHECKPOINT_NOT_REACHED when no checkpoint has been set, and OK otherwise.
	 * This function does the same action of the :reached condition.
	 */
	SetLgg_Machine_Return setlgg_machine_processor_checkpoint_reached(void *machine, const SetLgg_Machine_String checkpoint);

	/*
	 * Those functions are low level accessors to the virtual machine processor.
	 * It is not recommended to use them until you really know what you are doing!
	 */
	typedef struct returnframe
	{
		SetLgg_Machine_Return _return;
		unsigned int _frame;
	} SetLgg_Machine_ReturnFrame;

	/*
	 * This function returns the address of the next instruction executed by the processor. 
	 */
	size_t setlgg_machine_internal_processor_next_instruction(void *machine);

	/*
	 * This function jumps to a specific address in the program. If the address is invalid, a ILL interruption will be raised after returning
	 * the plugin function.
	 */
	void setlgg_machine_internal_processor_jump_address(void *machine, const size_t address);

	/*
	 * This function stacks the current state of the processor, and jumps to the function_address with the parameter_address.
	 * If the address is invalid, a ILL interruption will be raised after returning the plugin function.
	 */
	void setlgg_machine_internal_processor_call_address(void *machine, const size_t function_address, const size_t parameter_address);

	/*
	 * This function register a callback address for a specific interruption. The function returns OK if the callback is registered, INVALID_INTERRUPTION when the
	 * interruption is not a valid one,
	 */
	SetLgg_Machine_Return setlgg_machine_internal_processor_interruption_register_address(void *machine, const unsigned int interruption, const size_t address);

	/*
	 * This function duplicates the last stored state in the processor return stack. It returns INEXISTANT_CALLSTACK_FRAME when the stack is empty.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_dup_frame(void *machine);

	/*
	 * This function removes the last stored state in the processor return stack. It returns INEXISTANT_CALLSTACK_FRAME when the stack is empty.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_drop_frame(void *machine);

	/*
	 * This function swaps the two last stored states in the processor return stack. It returns INEXISTANT_CALLSTACK_FRAME when the stack is empty or contains only
	 * one state.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_swap_frame(void *machine);

	/*
	 * This function takes the frame_level state level and moves it to the top of the stack. The next :return instruction will restore this state.
	 * It returns INEXISTANT_CALLSTACK_FRAME when the stack does not contain at least frame_level states.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_roll_frame(void *machine, const unsigned int frame_level);

	/*
	 * This function takes the next state level and moves it to the frame_level level of the stack. This operation is the opposite of the previous one.
	 * It returns INEXISTANT_CALLSTACK_FRAME when the stack does not contain at least frame_level states.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_processor_callstack_unroll_frame(void *machine, const unsigned int frame_level);

	/*
	 * This function pushes the current processor state on the top of the return stack. The next :return instruction will jump to the next instruction.
	 */
	void setlgg_machine_internal_processor_callstack_push_current_frame(void *machine);

	/*
	 * This function reads the return stack until the program address used for the :return instruction matches the address associated to the label.
	 * The function returns INEXISTANT_LABEL if the label does not exist, FRAME_NOT_FOUND if no frame matches the condition above, and OK plus a frame
	 * level if one exists.
	 */
	SetLgg_Machine_ReturnFrame setlgg_machine_internal_processor_callstack_find_frame_label(void *machine, const SetLgg_Machine_String label);

	/*
	 * This function reads the return stack until the program address used for the :return instruction matches the address given in input.
	 * The function returns INEXISTANT_LABEL if the label does not exist, FRAME_NOT_FOUND if no frame matches the condition above, and OK plus a frame
	 * level if one exists.
	 */
	SetLgg_Machine_ReturnFrame setlgg_machine_internal_processor_callstack_find_frame_address(void *machine, const size_t address);

	/*
	 * This function only returns the number of frames in the processor return stack.
	 */
	unsigned int setlgg_machine_internal_processor_callstack_size(void *machine);

	/*
	 * This function marks a bloc of memory to be deleted by a :return :delete instruction.
	 * It may let the :return instruction raise an error if this instruction is misused.
	 */
	void setlgg_machine_internal_processor_register_allocation(void *machine, const size_t address, const size_t size);

	/*
	 * This function prevents a bloc of memory to be deleted by a :return :delete instruction.
	 * It may let the :return instruction raise an error if this instruction is misused.
	 */
	void setlgg_machine_internal_processor_unregister_allocation(void *machine, const size_t address, const size_t size);


	/*
	 * This function returns a handler on the current processor of the machine.
	 */
	void* setlgg_machine_internal_processor_save(void *machine);

	/*
	 * This function restores a processor from a previous handler.
	 * If you use load after save without reset between them, the load will do nothing.
	 */
	void setlgg_machine_internal_processor_load(void *machine, void *processor);

	/*
	 * This function loads an empty processor in the machine.
	 */
	void setlgg_machine_internal_processor_reset(void *machine);

	/*
	 * This function destroys a reference on a processor.
	 * This function shall be called on all values returned by setlgg_machine_internal_processor_save().
	 */
	void setlgg_machine_internal_processor_destruct(void *processor);

// program

// program low level (not recommanded)
	/*
	 * This function takes a string representing a program and compiles it. The function returns OK and a handler on the compiled program, or replies
	 * INVALID_CODE if the code is not valid.
	 */
	SetLgg_Machine_ReturnProgram setlgg_machine_internal_program_compile(void *machine, const SetLgg_Machine_String code);

	/*
	 * This function destroys a program returned by the functions setlgg_machine_internal_program_compile and setlgg_machine_internal_program_save.
	 */
	void setlgg_machine_internal_program_destruct(void *program);

	/*
	 * This function returns the current program runned by the machine.
	 */
	void* setlgg_machine_internal_program_save(void *machine);

	/*
	 * This function adds the code from the program given in input to the program currently loaded into the machine. All handlers to the current program
	 * are modified.
	 */
	void setlgg_machine_internal_program_append(void *machine, const void *program);

	/*
	 * This function loads the program in place of the current program, without modifying the machine state.
	 */
	void setlgg_machine_internal_program_load(void *machine, void *program);
	
	/*
	 * This function returns the number of instructions of the given program.
	 */
	size_t setlgg_machine_internal_program_size(void *program);

	/*
	 * This function returns the address of a label inside the given program.
	 * If the label exists, the function returns OK plus the address, or INEXISTANT_LABEL otherwise.
	 */
	SetLgg_Machine_ReturnAddress setlgg_machine_internal_program_find_label(void *program, const SetLgg_Machine_String label);

// memory
	typedef struct memorytypedbloc
	{
		SetLgg_Machine_ValueType _type;
		size_t _size;
	} SetLgg_Machine_MemoryTypedBloc;

	typedef struct memorybloc
	{
		size_t _address;
		size_t _size;
	} SetLgg_Machine_MemoryBloc;

	/*
	 * This function allocates a bloc of memory. Each typed bloc corresponds to the arguments of the :new instruction. The function returns the address of
	 * the first allocated memory element.
	 *
	 * As this function is a bit tricky to use, let see a simple example, equivalent to the instruction :new INT*3 , STR:

SetLgg_Machine_ValueType integer;
integer._type = VALUE_INTEGER;
SetLgg_Machine_ValueType string;
string._type = VALUE_STRING;
SetLgg_Machine_MemoryTypedBloc blocs[] = {  { integer , 3 } , { string , 1 } };
size_t address = setlgg_machine_memory_new(machine,2, blocs);

	 */
	size_t setlgg_machine_memory_new(void *machine, const size_t nb_of_blocs, const SetLgg_Machine_MemoryTypedBloc blocs[]);

	/*
	 * This function frees several blocs of memory. Each bloc is represented by the lowest address of the bloc, followed by the size of the bloc.
	 * The function returns OK if all blocs has been freed, of INVALID_ADDRESS if one address was not allocated.
	 *
	 * As this function is not the simpliest one to call, let see an example, equivalent to :delete &2*3,&10:

SetLgg_Machine_MemoryBloc blocs[] = { { 2 , 3 } , { 10 , 1 } };
if(setlgg_machine_memory_delete(machine,2,blocs)==INVALID_ADDRESS)
{
	setlgg_machine_error(machine,SIGSEGV,"Invalid address to be freed");
}
	 */
	SetLgg_Machine_Return setlgg_machine_memory_delete(void *machine,const size_t nb_of_blocs, const SetLgg_Machine_MemoryBloc blocs[]);

	/*
	 * This function reads a value from the memory by address. The wanted type of the value is passed as parameter. If the type of the value does not matter,
	 * the type VALUE_ALL shall be used to disable the test of the type.
	 * The function returns OK and the value if the read is a success, or returns UNDEFINED_MEMORY when the address is not defined - allocated - and
	 * UNINITIALISED_MEMORY when not initialised,
	 * or returns INVALID_TYPE if the type of the stored value does not match the wanted one.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_memory_read_address(void *machine, const size_t address, const SetLgg_Machine_ValueType asked_type);

	/*
	 * This function reads a value from the memory by alias. The resolution of the alias is performed before calling the previous function. It behaves exactly like
	 * the setlgg_machine_memory_read_address one, except that it returns also the error INEXISTANT_ALIAS if the alias does not exist.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_memory_read_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType asked_type);

	/*
	 * This function writes a value into the memory at the given address. This function returns OK if the value is written, or returns UNDEFINED_MEMORY if the memory
	 * address is not defined - allocated -, or returns INVALID_TYPE if the value is not of the same as the one used when the memory address has been allocated.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_write_address(void *machine, const size_t address, const SetLgg_Machine_Value value);

	/*
	 * This function writes a value into the memory at the given alias. The resolution of the alias is performed before calling the previous function.
	 * It behaves like the setlgg_machine_memory_write_address function, except tha it returns also the error INEXISTANT_ALIAS when the alias does not exist.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_write_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_Value value);

	/*
	 * This function removes the value at a given address without changing the type of the value. It returns OK if it succed, or UNDEFINED_MEMORY if the address is not defined.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_uninitialise_address(void *machine, const size_t address);

	/*
	 * This function removes the value at the linked address of the given alias. It behaves like the previous one, except that it can return INEXISTANT_ALIAS if the alias is not
	 * linekd to any address.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_uninitialise_alias(void *machine, const SetLgg_Machine_String alias);

	/*
	 * This function returns the state of a memory address. The function returns OK when the memory address is defined and the value has been initialised.
	 * It returns UNINITIALISED_MEMORY when the address has been allocated but no value have been stored into it.
	 * It returns UNDEFINED_MEMORY when the address has not been allocated.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_state_address(void *machine, const size_t address);

	/*
	 * This function returns the state of a memory address by an alias. This fuction behaves like the previous one, except that it can return INEXISTANT_ALIAS when
	 * the alias does not exists.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_state_alias(void *machine, const SetLgg_Machine_String alias);

	/*
	 * This function creates a memory alias on a specific address. It returns OK if the alias is linked to the new address, or returns ALIAS_ALREADY_EXISTS if the
	 * alias is already linked to another address.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_link_alias(void *machine, const SetLgg_Machine_String alias, const size_t address);

	/*
	 * This function removes a memory alias. It returns OK if the alias was linked to an address, or INEXISTANT_ALIAS if the alias was unknown.
	 */
	SetLgg_Machine_Return setlgg_machine_memory_unlink_alias(void *machine, const SetLgg_Machine_String alias);

	/*
	 * This function checks the existence of an alias. The function returns OK and the linked address when the alias exists, or INEXISTANT_ALIAS if the alias is unknown.
	 */
	SetLgg_Machine_ReturnAddress setlgg_machine_memory_exist_alias(void *machine, const SetLgg_Machine_String alias);

	/*
	 * This function returns the current address, like the :current keyword.
	 */
	size_t setlgg_machine_memory_current(void *machine);
// memory low level (not recommanded)

	/*
	 * This function changes the current address, returned by the :current keyword.
	 */
	void setlgg_machine_internal_memory_current(void *machine, const size_t address);
	/*
	 * This function only reduces the size of the memory container, to allow the virtual machine process to have a smaller memory usage on the system.
	 */
	void setlgg_machine_internal_memory_truncate(void *machine);

	/*
	 * This function reads the type associated to a memory address. The function returns OK and the type when the address is defined, or UNDEFINED_MEMORY when the address
	 * is not defined.
	 */
	SetLgg_Machine_ReturnType setlgg_machine_internal_memory_read_type_address(void *machine, const size_t address);

	/*
	 * This function reads the type associated to the memory address linked to an alias. This function behaves like the previous one, except that it returns INEXISTANT_ALIAS
	 * when the alias does not exist.
	 */
	SetLgg_Machine_ReturnType setlgg_machine_internal_memory_read_type_alias(void *machine, const SetLgg_Machine_String alias);

	/*
	 * This function changes the type of the value at the given memory address. The function lets the value uninitialised if the type change is successful.
	 * The function returns OK when it succeed, or UNDEFINED_MEMORY when the address is not defined.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_memory_write_type_address(void *machine, const size_t address, const SetLgg_Machine_ValueType type);

	/*
	 * This function changes the type of the value at the address linked by the given alias. It behaves like the previous function, except that it can returns INEXISTANT_ALIAS
	 * when the alias is unknown.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_memory_write_type_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType type);

	/*
	 * This function solves an address. It only means that the function will follow memory pointers until it find an exit condition. On success, the function returns IS_A_VALUE and the found value.
	 * On error, it can return POINTER_LOOP if a loop of pointer has been detected, IS_A_POINTER if the last found address does not contain a valid address, INVALID_TYPE if the found value is not 
	 * of the waited type, UNDEFINED_MEMORY if a pointer in the list of looked ones refers to an undefined address, UNINITIALISED_MEMORY if a looked pointer refers to an uninitialised address.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_solve_address(void *machine, const size_t address, const SetLgg_Machine_ValueType asked_type);

	/*
	 * This function solves an address from a given alias. It behaves like the previous one, except that it can return INEXISTANT_ALIAS when the alias is not linked to an address.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_solve_alias(void *machine, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType asked_type);

	/*
	 * This function solves an address from a given address. It follows the memory pointers until it reaches a value or the max hops. It returns IS_A_VALUE and the found value or IS_A_POINTER and
	 * the pointer if something has been found, or UNDEFINED_MEMORY if a pointer found during the solve process refers to an undefined address, UNINITIALISED_MEMORY if a looked pointer refers to an
	 * uninitialised address.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_limited_solve_address(void *machine, const unsigned int max, const size_t address, const SetLgg_Machine_ValueType asked_type);

	/*
	 * This function solves an address from the address linked to the given alias. It behaves like the previous function, except it can return INEXISTANT_ALIAS if the alias is not linked to an
	 * address.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_internal_memory_limited_solve_alias(void *machine, const unsigned int max, const SetLgg_Machine_String alias, const SetLgg_Machine_ValueType asked_type);

	/*
	 * This function returns the nature of an address. It returns UNDEFINED_MEMORY when the address is not defined, UNINITIALISED_MEMORY when no value is associated to the address, IS_A_POINTER
	 * if the value has the type VALUE_POINTER or IS_A_VALUE otherwise.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_memory_pointer_address(void *machine, const size_t address);

	/*
	 * This function returns the nature of the address linked by an alias. It behaves like the previous one, except that it can return INEXISTANT_ALIAS if the alias is not linked to an address.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_memory_pointer_alias(void *machine, const SetLgg_Machine_String alias);

	/*
	 * This function takes a string representing a memory file and compiles it. The function returns OK and a handler on the compiled memory, or replies
	 * INVALID_CODE if the code is not valid.
	 */
	SetLgg_Machine_ReturnMemory setlgg_machine_internal_memory_compile(void *machine, const SetLgg_Machine_String code);

	/*
	 * This function returns a handler on the current memory of the machine.
	 */
	void* setlgg_machine_internal_memory_save(void *machine);

	/*
	 * This function restores a memory from a previous handler.
	 * If you use load after save without reset between them, the load will do nothing.
	 */
	void setlgg_machine_internal_memory_load(void *machine, void *memory);

	/*
	 * This function loads an empty memory in the machine.
	 */
	void setlgg_machine_internal_memory_reset(void *machine);

	/*
	 * This function destroys a reference on a memory.
	 * This function shall be called on all values returned by setlgg_machine_internal_memory_save().
	 */
	void setlgg_machine_internal_memory_destruct(void *memory);

// streams
	typedef enum {READONLY,WRITEONLY,READWRITE} SetLgg_Machine_FileMode;
	typedef enum {AVAILABLE,LINE,ALL,SIZE} SetLgg_Machine_ReadMode;
	typedef enum {OFFSET,ABSOLUTE,END} SetLgg_Machine_SeekMode;

	/*
	 * This function opens a stream using a file as media. This function behaves like the :open instruction.
	 * It returns OK if the stream is opened, INVALID_FILEMODE if the file mode is not correct, INVALID_STREAM if the system is unable to open the stream, STREAM_ALREADY_EXISTS if another stream
	 * with the same name exists, INVALID_STREAM_OPERATION if the operation is not supported on this stream type.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_open_file(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String name);

	/*
	 * This function opens a stream using a socket configured to work with the TCP protocol. This function behaves like the :open instruction.
	 * It returns OK if the stream is opened, INVALID_FILEMODE if the file is incorrect. Please note that the READWRITE mode is not available here.
	 * If READONLY is used, the socket is configured to wait connections (server socket), and if WRITEONLY is used, the socket is configured to make requests to a server (client socket).
	 * The function returns also INVALID_STREAM if the system can open the media, STREAM_ALREADY_EXISTS is a stream with the same name already exist, INVALID_STREAM_OPERATION if the operation
	 * is not supported on this stream type.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_open_tcp(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port);

	/*
	 * This function opens a stream using a socket configured to work with the UDP protocol. This function behaves like the :open instruction.
	 * It returns OK if the stream is opened, INVALID_STREAM if the system can open the media, STREAM_ALREADY_EXISTS is a stream with the same name already exist, INVALID_STREAM_OPERATION if the operation
	 * is not supported on this stream type.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_open_udp(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port);

	/*
	 * This function closes a stream, like the :close instruction.
	 * It returns OK if the stream is closed, INVALID_STREAM if the stream does not exist, or INVALID_STREAM_OPERATION if the stream type does not support his operation.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_close(void *machine, const SetLgg_Machine_InOutReference stream);

	/*
	 * This function closes a stream, like the :close < instruction.
	 * It returns OK if the stream is closed, INVALID_STREAM if the stream does not exist, or INVALID_STREAM_OPERATION if the stream type does not support his operation.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_close_read(void *machine, const SetLgg_Machine_InOutReference stream);

	/*
	 * This function closes a stream, like the :close < instruction.
	 * It returns OK if the stream is closed, INVALID_STREAM if the stream does not exist, or INVALID_STREAM_OPERATION if the stream type does not support his operation.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_close_write(void *machine, const SetLgg_Machine_InOutReference stream);

	/*
	 * This function reads a value from the given stream like the :read instruction.
	 * The size value is used only when the SetLgg_Machine_ReadMode is equal to SIZE.
	 * It returns OK and the read value on success, INVALID_READMODE if the read mode is invalid, INVALID_STREAM if the stream does not exist, UNREADABLE_STREAM if the system can not perform a read on the
	 * media, INVALID_STREAM_OPERATION if the stream type does not support the read operation, INVALID_DATA if no data can be generated by the formatter, INVALID_FORMAT if the format string is invalid,
	 * INVALID_DATA_FORMAT if the data read on the media can not be tranformed to the given format,
	 * END_OF_STREAM if no more data are available on the stream.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_stream_read(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String format, const SetLgg_Machine_ReadMode mode, const size_t size);

	/*
	 * This function write a value to the given stream like the :write instruction.
	 * It returns OK and the read value on success, INVALID_STREAM if the stream does not exist, UNWRITABLE_STREAM if the system can not perform a write on the media, INVALID_STREAM_OPERATION if the stream
	 * type does not support the write operation, INVALID_DATA if no data can be generated by the formatter, INVALID_FORMAT if the format string is invalid, INVALID_DATA_FORMAT if the data can not be
	 * tranformed to the given format. 
	 */
	SetLgg_Machine_Return setlgg_machine_stream_write(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String format, const SetLgg_Machine_Value *value);

	/*
	 * This function performs an accept on a TCP socket opened by the setlgg_machine_stream_open_tcp with READONLY as file mode.
	 * On success, it returns OK and open a new stream linked to the connector stream name given in argument.
	 * Otherwise, it returns INVALID_STREAM if the stream does not exist, INVALID_STREAM_OPERATION if the operation is not alloawed on the stream type, STREAM_ALREADY_EXISTS if the name given in argument
	 * already exist.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_wait(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String connector);

	/*
	 * This function changes the read/write cursor on a stream, like the :seek instruction.
	 * It returns OK and the new cursor position on success, INVALID_SEEKMODE if the seek mode is invalid, INVALID_STREAM if the stream does not exist, INVALID_STREAM_OPERATION if the stream type does not
	 * support this operation.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_stream_seek(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_SeekMode mode, const long int position);

	/*
	 * Those three functions tell if some operations are possible on the given stream. All functions returns OK when the operation is possible, INVALID_STREAM if the stream does not exist,
	 * INVALID_STREAM_OPERATION if the operation is not supported by the stream.
	 * The first function returns UNREADABLE_STREAM if the stream is not readable, the second one returns UNWRITABLE_STREAM if the stream is not writable, the last one UNSEEKABLE_STREAM if the cursor
	 * position can not be changed on the stream.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_readable(void *machine, const SetLgg_Machine_InOutReference stream);
	SetLgg_Machine_Return setlgg_machine_stream_writable(void *machine, const SetLgg_Machine_InOutReference stream);
	SetLgg_Machine_Return setlgg_machine_stream_seekable(void *machine, const SetLgg_Machine_InOutReference stream);

	/*
	 * This function acts as the :send instruction.
	 * It returns OK when the data is transmitted, INVALID_STREAM_OPERATION when the selected stream does not support the transfert, or INVALID_DATA if another error occurs.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_send(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String name, const size_t address, const size_t size);

	/*
	 * Those functions act as the :receive instruction.
	 * It returns OK when the data is transmitted, INVALID_STREAM_OPERATION when the selected stream does not support the transfert, or INVALID_DATA if another error occurs.
	 */
	SetLgg_Machine_Return setlgg_machine_stream_receive(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String name, const size_t address, const size_t size);
	SetLgg_Machine_ReturnAddress setlgg_machine_stream_receive_allocate(void *machine, const SetLgg_Machine_InOutReference stream, const SetLgg_Machine_String name);

// streams low level (not recommanded)
	/*
	 * Those three functions create an entry in the stream list on an already opened media. The media is given as fd argument. The functions return OK when the stream is registered, INVALID_FILEMODE
	 * if the file mode is invalid, STREAM_ALREADY_EXISTS if the stream name already exists.
	 * Please note that the parameters given after the file descriptor are used to perform the read/write operations.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_stream_register_file(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String name);
	SetLgg_Machine_Return setlgg_machine_internal_stream_register_tcp(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd, const SetLgg_Machine_FileMode mode, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port);
	SetLgg_Machine_Return setlgg_machine_internal_stream_register_udp(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd, const SetLgg_Machine_String local_ip, const SetLgg_Machine_String local_port, const SetLgg_Machine_String distant_ip, const SetLgg_Machine_String distant_port);

	/*
	 * This function register two file descriptors under a stream name. It can be used to give an access to unix pipes/sockets.
	 * The function returns OK if the stream is registered or STREAM_ALREADY_EXISTS if the stream name already exist. 
	 */
	SetLgg_Machine_Return setlgg_machine_internal_stream_register_pipe(void *machine, const SetLgg_Machine_InOutReference stream, const unsigned int fd_read, const unsigned int fd_write);

	/*
	 * This function manipulates the internal buffer of a stream used in read operations.
	 * It gets at most a given number of bytes and returns OK and the read SetLgg_Machine_String in case of success, or INVALID_STREAM if the stream does not exist or INVALID_STREAM_OPERATION if the stream is not readable.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_buffer_get(void *machine, const SetLgg_Machine_InOutReference stream, const size_t size);

	/*
	 * Those three functions return respectively the read and the write file descriptors to perform low level input/output operations on streams, the third one returning a string containing the name of the stream.
	 * The functions return OK and the file descriptor value on success, or INVALID_STREAM if the stream does not exist.
	 */
	SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_read_id(void *machine, const SetLgg_Machine_InOutReference stream);
	SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_write_id(void *machine, const SetLgg_Machine_InOutReference stream);
	SetLgg_Machine_ReturnValue setlgg_machine_internal_stream_name(void *machine, const SetLgg_Machine_InOutReference stream);

	/*
	 * This function returns the nature of the stream.
	 * If the stream is connected to a process, the function returns ACTIVE_STREAM, and PASSIVE_STREAM otherwise.
	 * The functions returns INVALID_STREAM if the stream does not exist.
	 */
	SetLgg_Machine_Return setlgg_machine_internal_stream_nature(void *machine, const SetLgg_Machine_InOutReference stream);

// miscellaneous helpers and internal accessors
	/*
	 * The STR/STR qnd STR%STR operators uses a cache to keep compiled regular expressions.
	 * It is a valuable optimisation and this function gives access to this cache to plugins.
	 * The function takes the regular expression text and returns:
	 * - when the regular expression is correct:
	 *   - OK
	 *   - NULL
	 *   - A pointer to the compiled expression. You shall not call regfree on this pointer.
	 * - when the regular expression is invalid:
	 *   - INVALID_REGEX
	 *   - A pointer to a string value containing the error message. You shall call setlgg_machine_value_delete on this pointer
	 *   - A null pointer.
	 */
	SetLgg_Machine_ReturnRegex setlgg_machine_internal_regex_cache(void *machine, const SetLgg_Machine_String regex);

#if defined __cplusplus
}
#endif

#endif
