/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <signal.h>
#include <string.h>
#include <sstream>
#include <iostream>
#include <src/machine/plugin/officiels/real.h>

extern "C"
{
void plugin_configuration(void *plugin_handler)
{
	::setlgg_machine_plugin_register(plugin_handler,
		"PLUGIN real "
		"CONTAINS "
			"TYPE real "
			"INSTRUCTION real_addition USR(real) USR(real) RETURN "
			"INSTRUCTION real_substraction USR(real) USR(real) RETURN "
			"INSTRUCTION real_multiplication USR(real) USR(real) RETURN "
			"INSTRUCTION real_division USR(real) USR(real) RETURN "
			"INSTRUCTION real_negation USR(real) RETURN "
			"INSTRUCTION real_zero USR(real) STR NO RETURN "
			"INSTRUCTION real_positive USR(real) STR NO RETURN "
			"INSTRUCTION int_to_real INT RETURN "
			"INSTRUCTION str_to_real STR RETURN "
			"INSTRUCTION real_to_int USR(real) RETURN "
			"INSTRUCTION real_to_str USR(real) RETURN "
		"END"
	);
}

char* setlgg_machine_real_type_real_serialise(const void* valeur)
{
	std::ostringstream str;
	str << *(reinterpret_cast<const double*>(valeur));
	char *ser;
	ser = ::strdup(str.str().c_str());
	return ser;
}

void* setlgg_machine_real_type_real_unserialise(const char *valeur)
{
	std::istringstream str;
	str.str(std::string(valeur));
	double *r = new double;
	str >> *r;
	return reinterpret_cast<void*>(r);
}

void setlgg_machine_real_type_real_delete(void* valeur)
{
	delete reinterpret_cast<double*>(valeur);
}

void* setlgg_machine_real_type_real_copy(const void* valeur)
{
	const double *r = reinterpret_cast<const double*>(valeur);
	double *c = new double;
	*c=*r;
	return reinterpret_cast<void*>(c);
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_addition(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double gauche = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	double droite = *reinterpret_cast<double*>(parametres[1]._value._user_type);
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_usertype("real",new double);
	*(reinterpret_cast<double*>(resultat->_value._user_type)) = gauche+droite;
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_substraction(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double gauche = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	double droite = *reinterpret_cast<double*>(parametres[1]._value._user_type);
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_usertype("real",new double);
	*(reinterpret_cast<double*>(resultat->_value._user_type)) = gauche-droite;
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_multiplication(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double gauche = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	double droite = *reinterpret_cast<double*>(parametres[1]._value._user_type);
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_usertype("real",new double);
	*(reinterpret_cast<double*>(resultat->_value._user_type)) = gauche*droite;
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_division(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double gauche = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	double droite = *reinterpret_cast<double*>(parametres[1]._value._user_type);
	if((droite>-1e-5) and (droite<1e-5))
	{
		::setlgg_machine_error(machine,SIGFPE,"Division by zero");
	}
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_usertype("real",new double);
	*(reinterpret_cast<double*>(resultat->_value._user_type)) = gauche/droite;
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_negation(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double gauche = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_usertype("real",new double);
	*(reinterpret_cast<double*>(resultat->_value._user_type)) = -gauche;
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_zero(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double reel = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	if((reel>-1e-5) and (reel<1e-5))
	{
		::setlgg_machine_processor_jump_label(machine,parametres[1]._value._string);
	}
	return 0;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_positive(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double reel = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	if(reel>0)
	{
		::setlgg_machine_processor_jump_label(machine,parametres[1]._value._string);
	}
	return 0;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_int_to_real(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	long int entier = parametres[0]._value._integer;
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_usertype("real",new double);
	*(reinterpret_cast<double*>(resultat->_value._user_type)) = entier;
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_str_to_real(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	const char *chaine = parametres[0]._value._string._buffer;
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_usertype("real",new double);
	::sscanf(chaine,"%lf",(reinterpret_cast<double*>(resultat->_value._user_type)));
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_to_int(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double reel = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_integer(reel);
	return resultat;
}

SetLgg_Machine_Value* setlgg_machine_real_instruction_real_to_str(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	double reel = *reinterpret_cast<double*>(parametres[0]._value._user_type);
	char buffer[256];
	::snprintf(buffer,255,"%f",reel);
	SetLgg_Machine_Value *resultat = ::setlgg_machine_value_new_string(buffer);
	return resultat;
}

}
