/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <signal.h>
#include <string.h>
#include <sstream>
#include <iostream>
#include <vector>
#include <src/machine/plugin/officiels/strings.h>
#include <regex.h> // regex non supporte in g++4.4

extern "C"
{
void plugin_configuration(void *plugin_handler)
{
	setlgg_machine_plugin_register(plugin_handler,
		"PLUGIN strings "
		"CONTAINS "
			"INSTRUCTION strings_regex STR STR INT RETURN "
			"INSTRUCTION strings_substitute STR STR INT STR RETURN "
			"INSTRUCTION strings_split STR STR RETURN "
			"INSTRUCTION strings_token STR STR ADDRESS RETURN "
		"END"
	);
}

SetLgg_Machine_Value* setlgg_machine_strings_instruction_strings_regex(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	std::string resultat;
	SETLGG_MACHINE_VARIABLE_VALUE_TO_CPP_STRING(chaine,parametres[0]);
	long int nombre_sous_motifs = parametres[2]._value._integer;
	if(nombre_sous_motifs<0)
	{
		::setlgg_machine_error(machine,SIGILL,"Number of maximum submatches shall be positive");
	}
	++nombre_sous_motifs;
	SetLgg_Machine_ReturnRegex re = ::setlgg_machine_internal_regex_cache(machine,parametres[1]._value._string);
	if(re._return!=SetLgg_Machine_Return::OK)
	{
		std::ostringstream details;
		details << "Invalid regular expression" << std::endl << "Details:" << std::endl << "regcomp: " << SETLGG_MACHINE_VALUE_PTR_TO_CPP_STRING(re._error_message);
		::setlgg_machine_value_delete(re._error_message);
		::setlgg_machine_error(machine,SIGABRT,details.str().c_str());
	}
	::regmatch_t *match = new ::regmatch_t[nombre_sous_motifs];
	std::vector<std::vector<std::string> > motifs_trouves;
	while(::regexec(re._regex,chaine.c_str(),nombre_sous_motifs,match,0)==0)
	{
		std::vector<std::string> sous_motifs_trouves;
		regmatch_t *indice;
		for(indice = match ; indice<(match+nombre_sous_motifs) ; ++indice)
		{
			if(indice->rm_so<0)
			{
				break;
			}
			sous_motifs_trouves.push_back(std::string(chaine,indice->rm_so,indice->rm_eo-indice->rm_so));
		}
		chaine=chaine.substr(match->rm_eo);
		motifs_trouves.push_back(sous_motifs_trouves);
	}

	SetLgg_Machine_ValueType integer;
	integer._type = VALUE_INTEGER;
	SetLgg_Machine_ValueType string;
	string._type = VALUE_STRING;
	SetLgg_Machine_ValueType pointer;
	pointer._type = VALUE_POINTER;
	SetLgg_Machine_MemoryTypedBloc blocs[] = {  { integer , 1 } , { pointer, motifs_trouves.size() }};
	unsigned int adresse = ::setlgg_machine_memory_new(machine,2, blocs);
 	SetLgg_Machine_Value* retour = ::setlgg_machine_value_new_pointer(adresse);
	SetLgg_Machine_Value valeur;
	valeur._type._type = VALUE_INTEGER;
	valeur._value._integer = motifs_trouves.size();
	::setlgg_machine_memory_write_address(machine,adresse,valeur);
	for(auto it=motifs_trouves.cbegin() ; it!=motifs_trouves.cend() ; ++it)
	{
		++adresse;
		SetLgg_Machine_MemoryTypedBloc sous_blocs[] = {  { integer , 1 } , { string, it->size() }};
		unsigned int sousadresse = ::setlgg_machine_memory_new(machine,2, sous_blocs);
		valeur._type._type = VALUE_POINTER;
		valeur._value._pointer = sousadresse;
		::setlgg_machine_memory_write_address(machine,adresse,valeur);
		valeur._type._type = VALUE_INTEGER;
		valeur._value._integer = it->size();
		::setlgg_machine_memory_write_address(machine,sousadresse,valeur);
		++sousadresse;
		for(auto sousit=it->cbegin() ; sousit!=it->cend() ; ++sousit)
		{
			valeur._type._type = VALUE_STRING;
			valeur._value._string._buffer = const_cast<char*>(sousit->c_str());
			valeur._value._string._size = sousit->size();
			::setlgg_machine_memory_write_address(machine,sousadresse++,valeur);
		}
	}
	delete [] match;
	return retour;	
}

SetLgg_Machine_Value* setlgg_machine_strings_instruction_strings_substitute(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	std::string resultat;
	SETLGG_MACHINE_VARIABLE_VALUE_TO_CPP_STRING(chaine,parametres[0]);
	long int indice_sous_motifs = parametres[2]._value._integer;
	SETLGG_MACHINE_VARIABLE_VALUE_TO_CPP_STRING(remplacement,parametres[3]);
	SetLgg_Machine_ReturnRegex re = ::setlgg_machine_internal_regex_cache(machine,parametres[1]._value._string);
	if(re._return!=SetLgg_Machine_Return::OK)
	{
		std::ostringstream details;
		details << "Invalid regular expression" << std::endl << "Details:" << std::endl << "regcomp: " << SETLGG_MACHINE_VALUE_PTR_TO_CPP_STRING(re._error_message);
		::setlgg_machine_value_delete(re._error_message);
		::setlgg_machine_error(machine,SIGABRT,details.str().c_str());
	}
	if(indice_sous_motifs<0)
	{
		::setlgg_machine_error(machine,SIGILL,"Index of replaced submatch shall be positive");
	}
	::regmatch_t *match = new ::regmatch_t[indice_sous_motifs+1];
	while(not chaine.empty())
	{
		if(::regexec(re._regex,chaine.c_str(),indice_sous_motifs+1,match,0)==0)
		{
			if(match[indice_sous_motifs].rm_so!=-1)
			{
				resultat += chaine.substr(0,match[indice_sous_motifs].rm_so);
				resultat += remplacement;
				chaine = chaine.substr(match[indice_sous_motifs].rm_eo);
			}
			else
			{
				delete [] match;
				::setlgg_machine_error(machine,SIGILL,"Index of replaced submatch does not exist");
			}
		}
		else
		{
			resultat += chaine;
			chaine.clear();
		}
	}
	delete [] match;
	return ::setlgg_machine_value_new_string(resultat.c_str());
}

SetLgg_Machine_Value* setlgg_machine_strings_instruction_strings_split(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	std::string resultat;
	std::string chaine(parametres[0]._value._string._buffer,parametres[0]._value._string._size);
	::regmatch_t match;
	SetLgg_Machine_ReturnRegex re = ::setlgg_machine_internal_regex_cache(machine,parametres[1]._value._string);
	if(re._return!=SetLgg_Machine_Return::OK)
	{
		std::ostringstream details;
		details << "Invalid regular expression" << std::endl << "Details:" << std::endl << "regcomp: " << SETLGG_MACHINE_VALUE_PTR_TO_CPP_STRING(re._error_message);
		::setlgg_machine_value_delete(re._error_message);
		::setlgg_machine_error(machine,SIGABRT,details.str().c_str());
	}
	std::vector<std::string > motifs_trouves;
	while(::regexec(re._regex,chaine.c_str(),1,&match,0)==0)
	{
		motifs_trouves.push_back(std::string(chaine,0,match.rm_so));
		chaine=chaine.substr(match.rm_eo);
	}
	motifs_trouves.push_back(chaine);

	SetLgg_Machine_ValueType integer;
	integer._type = VALUE_INTEGER;
	SetLgg_Machine_ValueType string;
	string._type = VALUE_STRING;
	SetLgg_Machine_MemoryTypedBloc blocs[] = {  { integer , 1 } , { string, motifs_trouves.size() }};
	unsigned int adresse = ::setlgg_machine_memory_new(machine,2, blocs);
 	SetLgg_Machine_Value* retour = ::setlgg_machine_value_new_pointer(adresse);
	SetLgg_Machine_Value* valeur = ::setlgg_machine_value_new_integer(motifs_trouves.size());
	::setlgg_machine_memory_write_address(machine,adresse,*valeur);
	::setlgg_machine_value_delete(valeur);
	for(auto it=motifs_trouves.cbegin() ; it!=motifs_trouves.cend() ; ++it)
	{
		++adresse;
		valeur = ::setlgg_machine_value_new_sized_string(it->c_str(),it->size());
		::setlgg_machine_memory_write_address(machine,adresse,*valeur);
		::setlgg_machine_value_delete(valeur);
	}
	return retour;	
}

SetLgg_Machine_Value* setlgg_machine_strings_instruction_strings_token(void *machine, unsigned long int nb_parametres, SetLgg_Machine_Value parametres[])
{
	std::string resultat;
	auto adresse_chaine_depart = parametres[2]._value._pointer;
	SetLgg_Machine_ValueType string;
	string._type = VALUE_STRING;
	SetLgg_Machine_ReturnValue valeur_chaine_depart = ::setlgg_machine_memory_read_address(machine,adresse_chaine_depart,string);
	switch(valeur_chaine_depart._return)
	{
		case UNDEFINED_MEMORY:
		case UNINITIALISED_MEMORY:
			::setlgg_machine_error(machine,SIGSEGV,"Invalid address to string");
			break;
		case INVALID_TYPE:
			::setlgg_machine_error(machine,SIGSEGV,"String should be stored at given address");
			break;
		default:
			break;
	}
	std::string chaine_depart(valeur_chaine_depart._value._value._string._buffer,valeur_chaine_depart._value._value._string._size);
	::setlgg_machine_value_clear(&valeur_chaine_depart._value);
	std::ostringstream motif;
	motif << "^(" << SETLGG_MACHINE_VALUE_TO_CPP_STRING(parametres[0]) << ")" << SETLGG_MACHINE_VALUE_TO_CPP_STRING(parametres[1]);
	SetLgg_Machine_String motif_str = { motif.str().c_str(), motif.str().size()};
	SetLgg_Machine_ReturnRegex re = ::setlgg_machine_internal_regex_cache(machine,motif_str);
	if(re._return!=OK)
	{
		std::ostringstream details;
		details << "Invalid regular expression" << std::endl << "Details:" << std::endl << "regcomp: " << SETLGG_MACHINE_VALUE_PTR_TO_CPP_STRING(re._error_message);
		::setlgg_machine_value_delete(re._error_message);
		::setlgg_machine_error(machine,SIGABRT,details.str().c_str());
	}
	SetLgg_Machine_Value *retour;
	::regmatch_t match[2];
	if(::regexec(re._regex,chaine_depart.c_str(),2,match,0)==0)
	{
		std::string token_trouve(chaine_depart,0,match[1].rm_eo);
		std::string chaine_restante(chaine_depart.substr(match[0].rm_eo));
		retour = ::setlgg_machine_value_new_string(token_trouve.c_str());
		SetLgg_Machine_Value valeur;
		valeur._type._type = VALUE_STRING;
		valeur._value._string._buffer = const_cast<char*>(chaine_restante.c_str());
		valeur._value._string._size = chaine_restante.size();
		::setlgg_machine_memory_write_address(machine,adresse_chaine_depart,valeur);
	}
	else
	{
		retour = ::setlgg_machine_value_new_string("");
	}
	return retour;
}

}
