/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <csignal>
#include <sys/wait.h>
#include <cerrno>

#include <src/machine/processeur/gestionnaireinterruptions.h>
#include <src/machine/machine/machine.h>
using namespace SetLgg::Machine::Processeur;
bool GestionnaireInterruptions::interruption(const size_t& interruption)
{
	if(not valide_interruption(interruption))
		return false;
	::sigset_t blocage;
	::sigfillset(&blocage);
	if(::sigprocmask(SIG_BLOCK,&blocage,NULL))
	{
		std::cerr << "Writing to interruption handler without signal protection." << std::endl;
	}
	_interruptions_en_attente.ajout(interruption);
	if(::sigprocmask(SIG_UNBLOCK,&blocage,NULL))
	{
		std::cerr << "Unable to remove interruption handler signal protection." << std::endl;
	}
	return true;
}

void GestionnaireInterruptions::enregistrement_signaux(SetLgg::Machine::Processeur::ProcesseurSP& processeur)
{
	GestionnaireInterruptionsResident::instance()->_processeur = processeur;
	auto interruptions = interruptions_supportees();
	for(auto it = interruptions.cbegin() ; it!=interruptions.cend() ; ++it)
	{
		struct sigaction action;
		::sigemptyset(&(action.sa_mask));
		action.sa_flags = 0;
		if(processeur->existe_interruption(*it))
		{
			action.sa_handler=GestionnaireInterruptionsResident::gestion_signal;
		}
		else
		{
			if((*it)==SIGCHLD)
			{
				action.sa_flags = SA_RESTART;
				action.sa_handler=GestionnaireInterruptionsResident::consomme_enfants;
			}
			else
			{
				action.sa_handler=SIG_DFL;
			}
		}
		if(::sigaction(*it,&action,NULL))
		{
			std::cerr << "Unable to add/remove signal handler on interruption " << (*it) << std::endl;
		}
	}
}

void GestionnaireInterruptions::GestionnaireInterruptionsResident::gestion_signal(int signal)
{
	SetLgg::Machine::Processeur::ProcesseurSP processeur = GestionnaireInterruptionsResident::instance()->_processeur.lock();
	if(processeur)
	{
		processeur->interruption(signal);
	}
}

void GestionnaireInterruptions::GestionnaireInterruptionsResident::consomme_enfants(int signal)
{
	::wait(NULL);
}


bool GestionnaireInterruptions::modifier_interruption(const size_t& interruption, const SetLgg::Machine::Programme::AdresseSP& adresse_interruption)
{
	if(not valide_interruption(interruption))
		return false;
	struct sigaction action;
	::sigemptyset(&(action.sa_mask));
	action.sa_flags = 0;
	bool interruption_active = true;
	if(adresse_interruption)
	{
		_interruptions.ajoute(interruption,*adresse_interruption);
	}
	else
	{
		interruption_active = not _interruptions.enleve(interruption);
	}

	if(interruption_active)
	{
		action.sa_handler=GestionnaireInterruptionsResident::gestion_signal;
		if(::sigaction(interruption,&action,NULL))
		{
			std::cerr << "Unable to add signal handler on interruption " << interruption << std::endl;
		}
	}
	else
	{
		if(interruption==SIGCHLD)
		{
			action.sa_handler=GestionnaireInterruptionsResident::consomme_enfants;
			action.sa_flags=SA_RESTART;
		}
		else
		{
			action.sa_handler=SIG_DFL;
		}
		if(::sigaction(interruption,&action,NULL))
		{
			std::cerr << "Unable to add signal handler on interruption " << interruption << std::endl;
		}
	}
	return true;
}

bool GestionnaireInterruptions::valide_interruption(const size_t interruption)
{
	switch(interruption)
	{
		case SIGHUP:
		case SIGINT:
		case SIGQUIT:
		case SIGILL:
		case SIGABRT:
		case SIGFPE:
		case SIGSEGV:
		case SIGPIPE:
		case SIGALRM:
		case SIGTERM:
		case SIGUSR1:
		case SIGUSR2:
		case SIGCHLD:
		case SIGCONT:
		case SIGTSTP:
		case SIGTTIN:
		case SIGTTOU:
		case SIGBUS:
		case SIGPROF:
		case SIGURG:
		case SIGIO:
			return true;
		default:
			return false;
	}
}

std::set<size_t> GestionnaireInterruptions::interruptions_supportees()
{
	return { SIGHUP, SIGINT, SIGQUIT, SIGILL, SIGABRT, SIGFPE, SIGSEGV, SIGPIPE, SIGALRM, SIGTERM, SIGUSR1, SIGUSR2, SIGCHLD, SIGCONT, SIGTSTP, SIGTTIN, SIGTTOU, SIGBUS, SIGPROF, SIGURG, SIGIO };
}
