/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 
#include <unistd.h>

#include <src/machine/processeur/processeur.h>
using namespace SetLgg::Machine::Processeur;
#include <src/machine/machine/machine.h>
#include <src/machine/debogueur/donnees/debogueur.h>

template<Processeur::TypeExecution type>
void Processeur::execution(SetLgg::Machine::Machine::MachineSP& machine)
{
	for(;;)
	{
		InterruptionsEnAttente::ListeInterruptions interruptions = machine->_processeur->_interruptions.interruptions_en_attente();
		if(not interruptions.empty())
		{
			while(not interruptions.empty())
			{
				machine->_processeur->appel_interruption(interruptions.top());
				interruptions.pop();
			}
			if(type==Processeur::TypeExecution::DEBUG)
			{
				if(machine->_debogueur->execution_pointdarret())
					return;
			}
		}
		::sleep(0); // pour permettre aux signaux de se reveiller
		SetLgg::Machine::Programme::InstructionSP instruction = machine->_processeur->prochaine_instruction(machine->_programme);
		try
		{
			instruction->execution(machine);
		}
		catch(SetLgg::Global::Exception::Execution& e)
		{
			if(machine->_processeur->existe_interruption(e))
			{
				machine->_processeur->interruption(e);
			}
			else
			{
				e.change_position(*instruction);
				std::ostringstream oss;
				e.format(oss);
				std::cerr << oss.str();
				machine->arret(true,false,ERREUR_GENERIQUE,oss.str());
			}
		}
		if(type==Processeur::TypeExecution::DEBUG)
		{
			if(machine->_debogueur->execution_pointdarret())
				return;
		}
	}
}

// On a que deux valeurs possibles, c'est acceptable de les instancier explicitement
template void Processeur::execution<Processeur::TypeExecution::NORMAL>(SetLgg::Machine::Machine::MachineSP& machine);
template void Processeur::execution<Processeur::TypeExecution::DEBUG>(SetLgg::Machine::Machine::MachineSP& machine);

bool Processeur::interruption(const size_t& interruption)
{
	return _interruptions.interruption(interruption);
}

bool Processeur::appel_interruption(const size_t& interruption)
{
	SetLgg::Machine::Programme::AdresseSP adresse_interruption = _interruptions[interruption];
	if(adresse_interruption)
	{
		appel(*adresse_interruption,_etat_courant);
		return true;
	}
	return false;
}

bool Processeur::modifier_interruption(const size_t& interruption, const SetLgg::Machine::Programme::AdresseSP& adresse)
{
	return _interruptions.modifier_interruption(interruption,adresse);
}

void Processeur::appel(const SetLgg::Machine::Programme::Adresse& fonction, const SetLgg::Machine::Memoire::Adresse& parametres)
{
	_pile_etats.empile(_etat_courant);
	_etat_courant.prochaine_instruction(fonction,parametres);
	_etat_courant.nettoie();
}

void Processeur::retour()
{
	_etat_courant = _pile_etats.depile();
}

void Processeur::saut(const SetLgg::Machine::Programme::Adresse& label)
{
	_etat_courant.prochaine_instruction(label);
}

SetLgg::Machine::Programme::InstructionSP Processeur::prochaine_instruction(const SetLgg::Machine::Programme::ProgrammeSP& programme)
{
	auto instr_suivante = (*programme)[_etat_courant];
	_etat_courant.prochaine_instruction();
	return instr_suivante;
}

void Processeur::ajoute_marqueur(const std::string& marqueur)
{
	_etat_courant._marqueurs.ajoute(marqueur);
}

void Processeur::enleve_marqueur(const std::string& marqueur)
{
	_etat_courant._marqueurs.enleve(marqueur);
}

bool Processeur::presence_marqueur(const std::string& marqueur)
{
	return _etat_courant._marqueurs(marqueur);
}

void Processeur::ajoute_allocation_memoire(const SetLgg::Machine::Memoire::Memoire::BlocMemoire& bloc)
{
	_etat_courant._allocation_memoire.allocation(bloc);
}

void Processeur::ajoute_liberation_memoire(const SetLgg::Machine::Memoire::Memoire::ListeBlocsMemoire& blocs)
{
	_etat_courant._allocation_memoire.liberation(blocs);
}
