/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_CONDITIONOBJET_H_
#define _MACHINE_PROGRAMME_DONNEES_CONDITIONOBJET_H_

#include <src/global/global.h>
#include <src/global/exceptions.h>

#include <src/machine/machine/machine.h>
#include <src/machine/programme/donnees/adressememoire.h>
#include <src/machine/memoire/donnees/booleen.h>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/programme/donnees/condition.h>
#include <src/machine/programme/donnees/nomflux.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			template<typename Objet>
			class ConditionObjet : public Condition
			{
				public:
					ConditionObjet(const SHARED_PTR(Objet)& objet)
					:_objet(objet) {};
					virtual ~ConditionObjet() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const ConditionObjet<Objet>& condition)
					{
						condition.clause(os);
						os << *(condition._objet) << " :" << condition.typecondition();
						return os;
					}
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
					virtual std::string typecondition() const =0;
				protected:
					SHARED_PTR(Objet) _objet;
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const =0;
			};


			class ConditionAdresseDefinie : public ConditionObjet<AdresseMemoire>
			{
				public:
					ConditionAdresseDefinie(const AdresseMemoireSP& adresse)
					:ConditionObjet<AdresseMemoire>(adresse) {};
					virtual ~ConditionAdresseDefinie() {};
					virtual std::string typecondition() const
					{
						return "defined";
					}
				private:
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::Adresse adresse;
						try
						{
							adresse = _objet->resolution(machine);
						}
						catch(SetLgg::Machine::Memoire::AliasInconnu e)
						{
							return false;
						}
						return machine->_memoire->adresse_valide(adresse);
					};
			};

			class ConditionAdresseInitialisee : public ConditionObjet<AdresseMemoire>
			{
				public:
					ConditionAdresseInitialisee(const AdresseMemoireSP& adresse)
					:ConditionObjet<AdresseMemoire>(adresse) {};
					virtual ~ConditionAdresseInitialisee() {};
					virtual std::string typecondition() const
					{
						return "initialised";
					}
				private:
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::Adresse adresse = _objet->resolution(machine);
						return machine->_memoire->adresse_initialisee(adresse);
					};
			};

			class ConditionBooleenne : public ConditionObjet<SetLgg::Machine::Programme::Valeur>
			{
				public:
					ConditionBooleenne(const SetLgg::Machine::Programme::ValeurSP& booleen)
					:ConditionObjet<SetLgg::Machine::Programme::Valeur>(booleen) {};
					virtual ~ConditionBooleenne() {};
					virtual std::string typecondition() const
					{
						return "true";
					}
				private:
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::ValeurCSP valeur = _objet->evaluation(machine);
						SetLgg::Machine::Memoire::BooleenCSP booleen = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Booleen>(valeur);
						return *booleen;
					};
			};

			class ConditionChaineVide : public ConditionObjet<SetLgg::Machine::Programme::Valeur>
			{
				public:
					ConditionChaineVide(const SetLgg::Machine::Programme::ValeurSP& chaine)
					:ConditionObjet<SetLgg::Machine::Programme::Valeur>(chaine) {};
					virtual ~ConditionChaineVide() {};
					virtual std::string typecondition() const
					{
						return "empty";
					}
				private:
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::ValeurCSP valeur = _objet->evaluation(machine);
						SetLgg::Machine::Memoire::ChaineCSP chaine = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Chaine>(valeur);
						return (chaine->taille()==SetLgg::Machine::Memoire::Entier(0));
					};
			};

			class ConditionDescripteurLisible : public ConditionObjet<SetLgg::Machine::Programme::NomFlux>
			{
				public:
					ConditionDescripteurLisible(const SetLgg::Machine::Programme::NomFluxSP& nom)
					:ConditionObjet<SetLgg::Machine::Programme::NomFlux>(nom) {};
					virtual ~ConditionDescripteurLisible() {};
					virtual std::string typecondition() const
					{
						return "readable";
					}
				private:
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const;
			};

			class ConditionDescripteurInscriptible : public ConditionObjet<SetLgg::Machine::Programme::NomFlux>
			{
				public:
					ConditionDescripteurInscriptible(const SetLgg::Machine::Programme::NomFluxSP& nom)
					:ConditionObjet<SetLgg::Machine::Programme::NomFlux>(nom) {};
					virtual ~ConditionDescripteurInscriptible() {};
					virtual std::string typecondition() const
					{
						return "writable";
					}
				private:
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const;
			};

			class ConditionDescripteurCherchable : public ConditionObjet<SetLgg::Machine::Programme::NomFlux>
			{
				public:
					ConditionDescripteurCherchable(const SetLgg::Machine::Programme::NomFluxSP& nom)
					:ConditionObjet<SetLgg::Machine::Programme::NomFlux>(nom) {};
					virtual ~ConditionDescripteurCherchable() {};
					virtual std::string typecondition() const
					{
						return "seekable";
					}
				private:
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const;
			};
		}
	}
}
#endif
