/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_CONDITIONVALEUR_H_
#define _MACHINE_PROGRAMME_DONNEES_CONDITIONVALEUR_H_

#include <src/global/global.h>
#include <src/global/exceptions.h>

#include <src/machine/programme/donnees/valeur.h>
#include <src/machine/programme/donnees/condition.h>
#include <src/machine/memoire/donnees/entier.h>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/memoire/donnees/nomflux.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			template <typename Comparateur>
			class ConditionValeur : public Condition
			{
				public:
					ConditionValeur(ValeurSP gauche, ValeurSP droite)
					:_gauche(gauche),_droite(droite) {};
					virtual ~ConditionValeur() {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const ConditionValeur<Comparateur>& condition)
					{
						condition.clause(os);
						return os << *(condition._gauche) << " " << Comparateur::symbole() << " " << *(condition._droite);
					};

					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
				private:
					ValeurSP _gauche;
					ValeurSP _droite;
					virtual const bool evaluation_booleene_interne(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::ValeurCSP gauche = _gauche->evaluation(machine);
						SetLgg::Machine::Memoire::ValeurCSP droite = _droite->evaluation(machine);
						return (Comparateur::evaluation(machine,gauche,droite));
					};
			};

			namespace Comparaison
			{
				class TypesOperandesIncorrects : public SetLgg::Global::Exception::Execution
				{
					public:
						TypesOperandesIncorrects(const std::string& symbole, const SetLgg::Machine::Memoire::Type& gauche, const SetLgg::Machine::Memoire::Type& droite)
						:Execution(SIGABRT,std::string("Expression ")+std::string(gauche)+" "+symbole+" "+std::string(droite)+" has incorrect type operands") {};
				};

				struct ResultatComparaison
				{
					ResultatComparaison()
					:_valide(false),_resultat(false) {};
					ResultatComparaison(const bool resultat)
					:_valide(true),_resultat(resultat) {};
					bool _valide;
					bool _resultat;
				};

				template<typename Gauche, typename Operateur, typename Droite, typename Resultat>
				struct Evaluateur
				{
					static const ResultatComparaison evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						const SHARED_PTR(const Gauche) agauche = std::dynamic_pointer_cast<const Gauche>(gauche);
						if(not agauche)
							return ResultatComparaison();
						const SHARED_PTR(const Droite) adroite = std::dynamic_pointer_cast<const Droite>(droite);
						if(not adroite)
							return ResultatComparaison();
						return ResultatComparaison(Operateur::evaluation_operateur(machine,*agauche,*adroite));
					}
				};

				template<typename Operateur>
				struct PatronEntierChaineBooleenPointeurNomflux
				{
					static const bool evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						ResultatComparaison resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Entier,Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Chaine,Operateur,SetLgg::Machine::Memoire::Chaine,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Adresse,Operateur,SetLgg::Machine::Memoire::Adresse,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Booleen,Operateur,SetLgg::Machine::Memoire::Booleen,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::NomFlux,Operateur,SetLgg::Machine::Memoire::NomFlux,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						throw Comparaison::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*droite);
					};
				};

				template<typename Operateur>
				struct PatronEntierChaineBooleen
				{
					static const bool evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						ResultatComparaison resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Entier,Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Chaine,Operateur,SetLgg::Machine::Memoire::Chaine,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Booleen,Operateur,SetLgg::Machine::Memoire::Booleen,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						throw Comparaison::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*droite);
					};
				};

				template<typename Operateur>
				struct PatronEntierChaine
				{
					static const bool evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						ResultatComparaison resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Entier,Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						resultat = Comparaison::Evaluateur<SetLgg::Machine::Memoire::Chaine,Operateur,SetLgg::Machine::Memoire::Chaine,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat._valide)
							return resultat._resultat;
						throw Comparaison::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*droite);
					};
				};

			}
			struct Egal: public Comparaison::PatronEntierChaineBooleenPointeurNomflux<Egal>
			{
				static std::string symbole() { return "="; };
				template<typename Operande>
					static const bool evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche==droite;
					};
			};

			struct Different: public Comparaison::PatronEntierChaineBooleenPointeurNomflux<Different>
			{
				static std::string symbole() { return "<>"; };
				template<typename Operande>
					static const bool evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche!=droite;
					};
			};

			struct Inferieur: public Comparaison::PatronEntierChaine<Inferieur>
			{
				static std::string symbole() { return "<"; };
				template<typename Operande>
					static const bool evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche<droite;
					};
			};

			struct InferieurOuEgal: public Comparaison::PatronEntierChaine<InferieurOuEgal>
			{
				static std::string symbole() { return "<="; };
				template<typename Operande>
					static const bool evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche<=droite;
					};
			};

			struct Superieur: public Comparaison::PatronEntierChaine<Superieur>
			{
				static std::string symbole() { return ">"; };
				template<typename Operande>
					static const bool evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche>droite;
					};
			};

			struct SuperieurOuEgal: public Comparaison::PatronEntierChaine<SuperieurOuEgal>
			{
				static std::string symbole() { return ">="; };
				template<typename Operande>
					static const bool evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche>=droite;
					};
			};
		}
	}
}
#endif
