/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_EXPRESSIONOPERATEURBINAIRE_H_
#define _MACHINE_PROGRAMME_DONNEES_EXPRESSIONOPERATEURBINAIRE_H_

#include <src/global/global.h>

#include <src/machine/programme/donnees/expression.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			template<typename OperateurBinaire, typename Operande>
			class ExpressionOperateurBinaire : public Expression
			{
				public:
					ExpressionOperateurBinaire(SHARED_PTR(Operande) gauche, SHARED_PTR(Operande) droite)
					:_gauche(gauche), _droite(droite) {}
					virtual ~ExpressionOperateurBinaire() {};
					virtual const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::ValeurCSP gauche = _gauche->evaluation(machine);
						SetLgg::Machine::Memoire::ValeurCSP droite = _droite->evaluation(machine);
						return OperateurBinaire::evaluation(machine,gauche,droite);
					};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const ExpressionOperateurBinaire<OperateurBinaire,Operande>& expression)
					{
						return os << *(expression._gauche) << " " << OperateurBinaire::symbole() << " " << *(expression._droite);
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
				private:
					SHARED_PTR(Operande) _gauche;
					SHARED_PTR(Operande) _droite;
			};

			namespace OperateurBinaire
			{
				class TypesOperandesIncorrects : public SetLgg::Global::Exception::Execution
				{
					public:
						TypesOperandesIncorrects(const std::string& symbole, const SetLgg::Machine::Memoire::Type& gauche, const SetLgg::Machine::Memoire::Type& droite)
						:Execution(SIGSEGV,std::string("Expression ")+std::string(gauche)+" "+symbole+" "+std::string(droite)+" has incorrect type operands") {};
				};

				template<typename Gauche, typename Operateur, typename Droite, typename Resultat>
				struct Evaluateur
				{
					static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						const SHARED_PTR(const Gauche) agauche = std::dynamic_pointer_cast<const Gauche>(gauche);
						if(not agauche)
							return SetLgg::Machine::Memoire::ValeurCSP();
						const SHARED_PTR(const Droite) adroite = std::dynamic_pointer_cast<const Droite>(droite);
						if(not adroite)
							return SetLgg::Machine::Memoire::ValeurCSP();
						SetLgg::Machine::Memoire::ValeurCSP resultat(new Resultat(Operateur::evaluation_operateur(machine,*agauche,*adroite)));
						return resultat;
					}
				};

				template<typename Operateur>
				struct PatronEntierChaine
				{
					static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurBinaire::Evaluateur<SetLgg::Machine::Memoire::Entier,Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Entier>::evaluation(machine,gauche,droite);
						if(resultat)
							return resultat;
						resultat = OperateurBinaire::Evaluateur<SetLgg::Machine::Memoire::Chaine,Operateur,SetLgg::Machine::Memoire::Chaine,SetLgg::Machine::Memoire::Chaine>::evaluation(machine,gauche,droite);
						if(resultat)
							return resultat;
						throw OperateurBinaire::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*droite);
					};
				};

				template<typename Operateur>
				struct PatronBooleen
				{
					static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurBinaire::Evaluateur<SetLgg::Machine::Memoire::Booleen,Operateur,SetLgg::Machine::Memoire::Booleen,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,gauche,droite);
						if(resultat)
							return resultat;
						throw OperateurBinaire::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*droite);
					};
				};
			}
			struct Addition : public OperateurBinaire::PatronEntierChaine<Addition>
			{
				static std::string symbole() { return "+"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche+droite;
					};
			};

			struct Soustraction
			{
				typedef Soustraction Operateur;
				static std::string symbole() { return "-"; };
				static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
				{
					SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurBinaire::Evaluateur<SetLgg::Machine::Memoire::Entier,Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Entier>::evaluation(machine,gauche,droite);
					if(resultat)
						return resultat;
					resultat = OperateurBinaire::Evaluateur<SetLgg::Machine::Memoire::Chaine,Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Entier>::evaluation(machine,gauche,droite);
					if(resultat)
						return resultat;
					resultat = OperateurBinaire::Evaluateur<SetLgg::Machine::Memoire::Adresse,Operateur,SetLgg::Machine::Memoire::Adresse,SetLgg::Machine::Memoire::Entier>::evaluation(machine,gauche,droite);
					if(resultat)
						return resultat;
					throw OperateurBinaire::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*droite);
				};
				template<typename Operande>
					static SetLgg::Machine::Memoire::Entier evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, Operande gauche, const SetLgg::Machine::Memoire::Entier& droite)
					{
						return gauche-droite;
					};
				static SetLgg::Machine::Memoire::Entier evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::Adresse& gauche, const SetLgg::Machine::Memoire::Adresse& droite)
				{
					return SetLgg::Machine::Memoire::Entier(gauche-droite);
				};
			};

			struct Multiplication
			{
				typedef Multiplication Operateur;
				static std::string symbole() { return "*"; };
				static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& droite)
				{
					SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurBinaire::Evaluateur<SetLgg::Machine::Memoire::Entier,Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Entier>::evaluation(machine,gauche,droite);
					if(resultat)
						return resultat;
					throw OperateurBinaire::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*droite);
				};
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche*droite;
					};
			};

			struct Division : public OperateurBinaire::PatronEntierChaine<Division>
			{
				static std::string symbole() { return "/"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche/droite;
					};
			};

			struct Reste : public OperateurBinaire::PatronEntierChaine<Reste>
			{
				static std::string symbole() { return "%"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche%droite;
					};
			};

			struct Minimum : public OperateurBinaire::PatronEntierChaine<Minimum>
			{
				static std::string symbole() { return ":min"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						if(gauche<droite)
							return gauche;
						else
							return droite;
					};
			};

			struct Maximum : public OperateurBinaire::PatronEntierChaine<Maximum>
			{
				static std::string symbole() { return ":max"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						if(gauche<droite)
							return droite;
						else
							return gauche;
					};
			};

			struct EtBooleen : public OperateurBinaire::PatronBooleen<EtBooleen>
			{
				static std::string symbole() { return ":and"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche and droite;
					};
			};

			struct OuBooleen : public OperateurBinaire::PatronBooleen<OuBooleen>
			{
				static std::string symbole() { return ":or"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche or droite;
					};
			};

			struct OuExclusifBooleen : public OperateurBinaire::PatronBooleen<OuExclusifBooleen>
			{
				static std::string symbole() { return ":xor"; };
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& gauche, const Operande& droite)
					{
						return gauche xor droite;
					};
			};
		}

	}
}
#endif
