/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_EXPRESSIONOPERATEURTERNAIRE_H_
#define _MACHINE_PROGRAMME_DONNEES_EXPRESSIONOPERATEURTERNAIRE_H_

#include <src/global/global.h>

#include <src/machine/programme/donnees/expression.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			enum class ExpressionOperateurTernaireOrigine {DEBUT, FIN};
			template<typename OperateurTernaire, typename Operande>
			class ExpressionOperateurTernaire : public Expression
			{
				public:
					ExpressionOperateurTernaire(SHARED_PTR(Operande) gauche, SHARED_PTR(Operande) milieu, SHARED_PTR(Operande) droite)
					:_gauche(gauche), _milieu(milieu), _droite(droite) {}
					virtual ~ExpressionOperateurTernaire() {};
					virtual const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::ValeurCSP gauche = _gauche->evaluation(machine);
						SetLgg::Machine::Memoire::ValeurCSP milieu = _milieu->evaluation(machine);
						SetLgg::Machine::Memoire::ValeurCSP droite = _droite->evaluation(machine);
						return OperateurTernaire::evaluation(machine,gauche,milieu,droite);
					};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const ExpressionOperateurTernaire<OperateurTernaire,Operande>& expression)
					{
						os << OperateurTernaire::symbole() << " " << *(expression._gauche) << " " << *(expression._milieu) << " " << *(expression._droite);
						OperateurTernaire::affiche(os);
						return os;
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
				private:
					SHARED_PTR(Operande) _gauche;
					SHARED_PTR(Operande) _milieu;
					SHARED_PTR(Operande) _droite;
			};

			namespace OperateurTernaire
			{
				class TypesOperandesIncorrects : public SetLgg::Global::Exception::Execution
				{
					public:
						TypesOperandesIncorrects(const std::string& symbole, const SetLgg::Machine::Memoire::Type& gauche, const SetLgg::Machine::Memoire::Type& milieu, const SetLgg::Machine::Memoire::Type& droite)
						:Execution(SIGSEGV,std::string("Expression ")+symbole+" "+std::string(gauche)+" "+std::string(milieu)+" "+std::string(droite)+" has incorrect type operands") {};
				};

				template<typename Operateur, typename Gauche, typename Milieu, typename Droite, typename Resultat>
				struct Evaluateur
				{
					static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& milieu, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						const SHARED_PTR(const Gauche) agauche = std::dynamic_pointer_cast<const Gauche>(gauche);
						if(not agauche)
							return SetLgg::Machine::Memoire::ValeurCSP();
						const SHARED_PTR(const Milieu) aumilieu = std::dynamic_pointer_cast<const Milieu>(milieu);
						if(not aumilieu)
							return SetLgg::Machine::Memoire::ValeurCSP();
						const SHARED_PTR(const Droite) adroite = std::dynamic_pointer_cast<const Droite>(droite);
						if(not adroite)
							return SetLgg::Machine::Memoire::ValeurCSP();
						SetLgg::Machine::Memoire::ValeurCSP resultat(new Resultat(Operateur::evaluation_operateur(machine,*agauche,*aumilieu,*adroite)));
						return resultat;
					}
				};

				template<typename Operateur>
				struct PatronChaineEntierEntier
				{
					static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& gauche, const SetLgg::Machine::Memoire::ValeurCSP& milieu, const SetLgg::Machine::Memoire::ValeurCSP& droite)
					{
						SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurTernaire::Evaluateur<Operateur,SetLgg::Machine::Memoire::Chaine,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Chaine>::evaluation(machine,gauche,milieu,droite);
						if(resultat)
							return resultat;
						throw OperateurTernaire::TypesOperandesIncorrects(Operateur::symbole(),*gauche,*milieu,*droite);
					};
				};

			}
			template <ExpressionOperateurTernaireOrigine origine=ExpressionOperateurTernaireOrigine::DEBUT>
			struct SousChaine : public OperateurTernaire::PatronChaineEntierEntier<SousChaine<origine> >
			{
				static const ExpressionOperateurTernaireOrigine _origine = origine;
				static std::string symbole() { return ":substring"; };
				template<typename oStream>
				static void affiche(oStream& os)
				{
					if(origine==ExpressionOperateurTernaireOrigine::FIN)
					{
						os << " END";
					}
				};
				template<typename OperandeGauche, typename OperandeMilieu, typename OperandeDroite>
					static OperandeGauche evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const OperandeGauche& gauche, const OperandeMilieu& milieu, const OperandeDroite& droite)
					{
						SetLgg::Machine::Memoire::Chaine::_Chaine chaine = gauche;
						SetLgg::Machine::Memoire::Entier::_Entier position = milieu;
						SetLgg::Machine::Memoire::Entier::_Entier taille = droite;
						SetLgg::Machine::Memoire::Entier::_Entier debut;
						SetLgg::Machine::Memoire::Entier::_Entier taillechaine = chaine.length();
						if(origine==SetLgg::Machine::Programme::ExpressionOperateurTernaireOrigine::FIN)
						{
							position+=taillechaine;
						}
						if(taille<0)
						{
							debut = position+taille;
							taille *= -1;
						}
						else
						{
							debut = position;
						}
						if(debut<0)
						{
							taille += debut;
							debut = 0;
						}
						if(debut>taillechaine)
						{
							taille += (debut-taillechaine);
							debut = taillechaine;
						}
						if((debut+taille)>taillechaine)
						{
							taille = taillechaine-debut;
						}
						if(taille<0)
						{
							taille = 0;
						}
						return chaine.substr(debut,taille);
					};
			};

		}
	}
}
#endif
