/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_EXPRESSIONOPERATEURUNAIRE_H_
#define _MACHINE_PROGRAMME_DONNEES_EXPRESSIONOPERATEURUNAIRE_H_

#include <src/global/global.h>

#include <src/machine/programme/donnees/expression.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			template<typename OperateurUnaire, typename Operande>
			class ExpressionOperateurUnaire : public Expression
			{
				public:
					ExpressionOperateurUnaire(SHARED_PTR(Operande) operande)
					:_operande(operande) {}
					virtual ~ExpressionOperateurUnaire() {};
					virtual const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						SetLgg::Machine::Memoire::ValeurCSP valeur = _operande->evaluation(machine);
						return OperateurUnaire::evaluation(machine,valeur);
					};
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const ExpressionOperateurUnaire<OperateurUnaire,Operande>& expression)
					{
						return os << OperateurUnaire::symbole() << *(expression._operande);
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
				private:
					SHARED_PTR(Operande) _operande;
			};

			namespace OperateurUnaire
			{
				class TypeOperandeIncorrect : public SetLgg::Global::Exception::Execution
				{
					public:
						TypeOperandeIncorrect(const std::string& symbole, const SetLgg::Machine::Memoire::Type& valeur)
						:Execution(SIGSEGV,std::string("Expression ")+symbole+" "+std::string(valeur)+" has incorrect type operand") {};
				};

				template<typename Operateur, typename Valeur, typename Resultat>
				struct Evaluateur
				{
					static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& valeur)
					{
						const SHARED_PTR(const Valeur) avaleur = std::dynamic_pointer_cast<const Valeur>(valeur);
						if(not avaleur)
							return SetLgg::Machine::Memoire::ValeurCSP();
						SetLgg::Machine::Memoire::ValeurCSP resultat(new Resultat(Operateur::evaluation_operateur(machine,*avaleur)));
						return resultat;
					}
				};
			}

			struct NegationBooleene
			{
				typedef NegationBooleene Operateur;
				static std::string symbole() { return ":not "; };
				static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& valeur)
				{
					SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurUnaire::Evaluateur<Operateur,SetLgg::Machine::Memoire::Booleen,SetLgg::Machine::Memoire::Booleen>::evaluation(machine,valeur);
					if(resultat)
						return resultat;
					throw OperateurUnaire::TypeOperandeIncorrect(Operateur::symbole(),*valeur);
				};
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& valeur)
					{
						return not valeur;
					};
			};

			struct NegationEntiere
			{
				typedef NegationEntiere Operateur;
				static std::string symbole() { return "-"; };
				static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& valeur)
				{
					SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurUnaire::Evaluateur<Operateur,SetLgg::Machine::Memoire::Entier,SetLgg::Machine::Memoire::Entier>::evaluation(machine,valeur);
					if(resultat)
						return resultat;
					throw OperateurUnaire::TypeOperandeIncorrect(Operateur::symbole(),*valeur);
				};
				template<typename Operande>
					static Operande evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const Operande& valeur)
					{
						return Operande(-valeur);
					};
			};

			struct TailleChaine
			{
				typedef TailleChaine Operateur;
				static std::string symbole() { return ":size "; };
				static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& valeur)
				{
					SetLgg::Machine::Memoire::ValeurCSP resultat = OperateurUnaire::Evaluateur<Operateur,SetLgg::Machine::Memoire::Chaine,SetLgg::Machine::Memoire::Entier>::evaluation(machine,valeur);
					if(resultat)
						return resultat;
					throw OperateurUnaire::TypeOperandeIncorrect(Operateur::symbole(),*valeur);
				};
				static SetLgg::Machine::Memoire::Entier evaluation_operateur(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::Chaine& valeur)
				{
					return valeur.taille();
				};
			};

			struct Copie
			{
				typedef Copie Operateur;
				static std::string symbole() { return ""; };
				static const SetLgg::Machine::Memoire::ValeurCSP evaluation(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::ValeurCSP& valeur)
				{
					SetLgg::Machine::Memoire::ValeurCSP resultat = valeur->copie();
					return resultat;
				};
			};
		}
	}
}
#endif
