/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <src/machine/programme/donnees/format.h>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/memoire/donnees/entier.h>
#include <stdio.h>
#include <regex.h>
using namespace SetLgg::Machine::Programme;

std::string Format::operator() (const SetLgg::Machine::Memoire::ValeurCSP& valeur) const
{
	if(_format=="s")
	{
		SetLgg::Machine::Memoire::ChaineCSP chaine = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Chaine>(valeur);
		if(not chaine)
		{
			std::ostringstream oss;
			valeur->format(oss);
			throw DonneesInvalidesPourFormat(oss.str(),_format);
		}
		return *chaine;
	}
	SetLgg::Machine::Memoire::EntierCSP entier = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Entier>(valeur);
	if(not entier)
	{
		std::ostringstream oss;
		valeur->format(oss);
		throw DonneesInvalidesPourFormat(oss.str(),_format);
	}
	long int entier_base = static_cast<long int>(*entier);
	if(_format=="c")
	{
		if(not ((entier_base>=0) and (entier_base<=255)))
		{
			std::ostringstream oss;
			valeur->format(oss);
			throw DonneesInvalidesPourFormat(oss.str(),_format);
		}
		unsigned char caractere = entier_base;
		return std::string(1,caractere);
	}
	char buffer[256];
	::snprintf(buffer,255,(std::string("%")+_format).c_str(),entier_base);
	return std::string(buffer);
}

bool Format::represente_un_entier(std::string chaine)
{
	regex_t re;
	regmatch_t match;
	::regcomp(&re,"^[+-]?([0-9]+|0[0-7]+|0[xX][0-9a-fA-F]+)$",REG_EXTENDED);
	if(::regexec(&re,chaine.c_str(),1,&match,0)==0)
	{
		::regfree(&re);
		return match.rm_so==0 and (match.rm_eo)==static_cast<int>(chaine.size());
	}
	::regfree(&re);
	return false;
}

std::string Format::conversion_format(const std::string& format)
{
	return std::string("%l")+format.substr(format.size()-1);
}

SetLgg::Machine::Memoire::ValeurSP Format::operator() (const std::string& valeur) const
{
	if(_format=="s")
	{
		SetLgg::Machine::Memoire::ValeurSP chaine(new SetLgg::Machine::Memoire::Chaine(valeur));
		return chaine;
	}
	if(_format=="c")
	{
		if(not (valeur.size()==1))
		{
			throw DonneesInvalidesPourFormat(valeur,_format);
		}
		unsigned char caractere = valeur[0];
		long int entierdiscret = caractere;
		SetLgg::Machine::Memoire::ValeurSP entier(new SetLgg::Machine::Memoire::Entier(entierdiscret));
		return entier;
	}
	if(not represente_un_entier(valeur))
	{
		throw DonneesInvalidesPourFormat(valeur,_format);
	}
	long int entierdiscret;
	std::string format = conversion_format(_format);
	::sscanf(valeur.c_str(),format.c_str(),&entierdiscret);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << valeur.c_str() << " " <<  (std::string("%")+_format).c_str() << ": " << entierdiscret << std::endl;
#endif
	SetLgg::Machine::Memoire::ValeurSP entier(new SetLgg::Machine::Memoire::Entier(entierdiscret));
	return entier;
}

SetLgg::Machine::Memoire::ValeurSP Format::convertit(const SetLgg::Machine::Memoire::ValeurCSP& valeur) const
{
	if(_format=="s")
	{
		throw FormatInvalide();
	}
	else
	{
		SetLgg::Machine::Memoire::EntierCSP entier = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Entier>(valeur);
		if(entier)
		{
			SetLgg::Machine::Memoire::ValeurSP resultat(new SetLgg::Machine::Memoire::Chaine((*this)(valeur)));
			return resultat;
		}
		else
		{
			SetLgg::Machine::Memoire::ChaineCSP chaine = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Chaine>(valeur);
			return (*this)(*chaine);
		}
	}
}

SetLgg::Machine::Memoire::ValeurSP Format::valeur_nulle() const
{
	if(_format=="s")
	{
		SetLgg::Machine::Memoire::ChaineSP chaine_nulle(new SetLgg::Machine::Memoire::ChaineNulle());
		return chaine_nulle;
	}
	else
	{
		SetLgg::Machine::Memoire::EntierSP entier_nul(new SetLgg::Machine::Memoire::EntierNul());
		return entier_nul;
	}
}
