/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <fstream>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <src/machine/programme/donnees/instructionexecution.h>
#include <src/machine/machine/machine.h>
#include <src/machine/flux/flux.h>
#include <src/machine/memoire/donnees/chaine.h>
using namespace SetLgg::Machine::Programme;
#define LECTURE 0
#define ECRITURE 1
namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			namespace Outil
			{
				struct Memoire
				{
					Memoire()
						:fichier(0),parametres(0),taille(0)
					{
						tube_lecture[LECTURE]=-1;
						tube_lecture[ECRITURE]=-1;
						tube_ecriture[LECTURE]=-1;
						tube_ecriture[ECRITURE]=-1;
						tube_erreur[LECTURE]=-1;
						tube_erreur[ECRITURE]=-1;
					};
					~Memoire()
					{
						if(tube_lecture[LECTURE]>=0)	{ ::close(tube_lecture[LECTURE]); }
						if(tube_lecture[ECRITURE]>=0)	{ ::close(tube_lecture[ECRITURE]); }
						if(tube_ecriture[LECTURE]>=0)	{ ::close(tube_ecriture[LECTURE]); }
						if(tube_ecriture[ECRITURE]>=0)	{ ::close(tube_ecriture[ECRITURE]); }
						if(tube_erreur[LECTURE]>=0)	{ ::close(tube_erreur[LECTURE]); }
						if(tube_erreur[ECRITURE]>=0)	{ ::close(tube_erreur[ECRITURE]); }
						if(fichier) ::free(fichier);
						for(int i=0 ; i<taille ; ++i)
						{
							if(parametres[i]) ::free(parametres[i]);
						}
						delete [] parametres;
					}
					static void ferme(int &descripteur)
					{
						::close(descripteur);
						descripteur = -1;
					}

					int tube_lecture[2];
					int tube_ecriture[2];
					int tube_erreur[2];
					char *fichier;
					char **parametres;
					int taille;
				};
			}
		}
	}
}

void InstructionExecution::execution(SetLgg::Machine::Machine::MachineSP& machine) const
{
	SetLgg::Machine::Programme::Outil::Memoire memoire;
	SetLgg::Machine::Memoire::NomFlux nom_flux = _descripteur->evaluation_basique(machine);

	if(::pipe(memoire.tube_lecture)<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "pipe: " << ::strerror(errno);
		throw ImpossibleDExecuterCommande(details.str());
	}
	if(::pipe(memoire.tube_ecriture)<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "pipe: " << ::strerror(errno);
		throw ImpossibleDExecuterCommande(details.str());
	}

	SetLgg::Machine::Memoire::NomFluxSP nom_flux_erreur;
	if(_erreur)
	{
		nom_flux_erreur = SetLgg::Machine::Memoire::NomFluxSP(new SetLgg::Machine::Memoire::NomFlux(_erreur->evaluation_basique(machine)));
		if(::pipe(memoire.tube_erreur)<0)
		{
			std::ostringstream details;
			details << SetLgg::Global::Exception::details << std::endl << "pipe: " << ::strerror(errno);
			throw ImpossibleDExecuterCommande(details.str());
		}
	}

	memoire.parametres = new char*[_valeurs.size()+1];
	
	for(auto it = _valeurs.cbegin() ; it!=_valeurs.cend() ; ++it)
	{
		SetLgg::Machine::Memoire::ValeurCSP valeur = (*it)->evaluation(machine);
		SetLgg::Machine::Memoire::ChaineCSP chaine = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Chaine>(valeur);
		if(not chaine)
		{
			std::ostringstream oss;
			oss << valeur;
			throw ParametreIncorrect(oss.str());
		}
		std::string parametre = *chaine;
		memoire.parametres[memoire.taille]=::strdup(parametre.c_str());
		if(memoire.taille==0)
		{
			memoire.fichier=::strdup(parametre.c_str());
		}
		++memoire.taille;
	}
	memoire.parametres[memoire.taille]=NULL;

	pid_t pid = ::fork();
	if(pid<0)
	{
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "fork: " << ::strerror(errno);
		throw ImpossibleDExecuterCommande(details.str());
	}
	if(pid)
	{ // pere
		Outil::Memoire::ferme(memoire.tube_lecture[ECRITURE]);
		Outil::Memoire::ferme(memoire.tube_ecriture[LECTURE]);
		SetLgg::Machine::Flux::FluxSP tubes(new SetLgg::Machine::Flux::FluxPaireTubesAnonymes(memoire.tube_lecture[LECTURE],memoire.tube_ecriture[ECRITURE]));
		if(not machine->_flux->ajoute(nom_flux,tubes))
		{
			throw ImpossibleDeCommuniquerAvecLaCommande(nom_flux);
		}

		if(_erreur)
		{
			SetLgg::Machine::Flux::FluxSP erreur(new SetLgg::Machine::Flux::FluxPaireTubesAnonymes(memoire.tube_erreur[LECTURE],memoire.tube_erreur[ECRITURE]));
			erreur->fermeture_ecriture();
			memoire.tube_erreur[ECRITURE]=-1;
			if(not machine->_flux->ajoute(*nom_flux_erreur,erreur))
			{
				machine->_flux->fermeture(nom_flux);
				throw ImpossibleDeCommuniquerAvecLaCommande(*nom_flux_erreur);
			}
		}
		memoire.tube_lecture[LECTURE]=-1;
		memoire.tube_ecriture[ECRITURE]=-1;
		memoire.tube_erreur[LECTURE]=-1;
	}
	else
	{
		Outil::Memoire::ferme(memoire.tube_lecture[LECTURE]);
		Outil::Memoire::ferme(memoire.tube_ecriture[ECRITURE]);
		if(_erreur)
		{
			Outil::Memoire::ferme(memoire.tube_erreur[LECTURE]);
		}
		{
#ifdef SETLGG_DEBUG
			std::ofstream os("/tmp/exec.log");
#endif
			auto descripteurs_non_systeme = machine->_flux->descripteurs_non_systeme();
			for(auto descripteur: descripteurs_non_systeme)
			{
#ifdef SETLGG_DEBUG
				os << "close " << descripteur << std::endl;
#endif
				Outil::Memoire::ferme(descripteur);
			}
#ifdef SETLGG_DEBUG
			os << memoire.tube_lecture[LECTURE] << " *" << memoire.tube_lecture[ECRITURE] << " / *" << memoire.tube_ecriture[LECTURE] << " " << memoire.tube_ecriture[ECRITURE] << " / " << memoire.tube_erreur[LECTURE] << " *" << memoire.tube_erreur[ECRITURE] << std::endl;
#endif
			int lecture[4] = {-1, -1, -1, -1};
			lecture[0] = ::dup(memoire.tube_lecture[ECRITURE]);
			lecture[1] = ::dup(lecture[0]);
			lecture[2] = ::dup(lecture[1]);
			lecture[3] = ::dup(lecture[2]);
#ifdef SETLGG_DEBUG
			os << lecture[0] << " " << lecture[1] << " " << lecture[2] << " " << lecture[3] << std::endl;
#endif
			int ecriture = ::dup(memoire.tube_ecriture[LECTURE]);
			int err;
			if(_erreur)
			{
				err = ::dup(memoire.tube_erreur[ECRITURE]);
			}
			Outil::Memoire::ferme(lecture[0]);
			Outil::Memoire::ferme(lecture[1]);
			Outil::Memoire::ferme(lecture[2]);
			Outil::Memoire::ferme(memoire.tube_lecture[ECRITURE]);
			Outil::Memoire::ferme(memoire.tube_ecriture[LECTURE]);
			if(_erreur)
			{
				Outil::Memoire::ferme(memoire.tube_erreur[ECRITURE]);
			}
			memoire.tube_lecture[ECRITURE] = lecture[3];
			memoire.tube_ecriture[LECTURE] = ecriture;
			if(_erreur)
			{
				memoire.tube_erreur[ECRITURE] = err;
			}
#ifdef SETLGG_DEBUG
			os << memoire.tube_lecture[LECTURE] << " *" << memoire.tube_lecture[ECRITURE] << " / *" << memoire.tube_ecriture[LECTURE] << " " << memoire.tube_ecriture[ECRITURE] << " / " << memoire.tube_erreur[LECTURE] << " *" << memoire.tube_erreur[ECRITURE] << std::endl;
#endif
		}
		if(::dup2(memoire.tube_lecture[ECRITURE],STDOUT_FILENO)<0)
		{
			std::ostringstream details;
			details << SetLgg::Global::Exception::details << std::endl << "dup2: " << ::strerror(errno);
			throw ImpossibleDExecuterCommande(details.str());
		}
		if(::dup2(memoire.tube_ecriture[LECTURE],STDIN_FILENO)<0)
		{
			std::ostringstream details;
			details << SetLgg::Global::Exception::details << std::endl << "dup2: " << ::strerror(errno);
			throw ImpossibleDExecuterCommande(details.str());
		}
		if(_erreur)
		{
			if(::dup2(memoire.tube_erreur[ECRITURE],STDERR_FILENO)<0)
			{
				std::ostringstream details;
				details << SetLgg::Global::Exception::details << std::endl << "dup2: " << ::strerror(errno);
				throw ImpossibleDExecuterCommande(details.str());
			}
		}
		Outil::Memoire::ferme(memoire.tube_lecture[ECRITURE]);
		Outil::Memoire::ferme(memoire.tube_ecriture[LECTURE]);

		::execv(memoire.fichier,memoire.parametres);
		std::ostringstream details;
		details << SetLgg::Global::Exception::details << std::endl << "execv: " << ::strerror(errno);
		throw ImpossibleDExecuterCommande(details.str());
	}
}
