/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_INSTRUCTIONEXECUTION_H_
#define _MACHINE_PROGRAMME_DONNEES_INSTRUCTIONEXECUTION_H_

#include <src/global/global.h>

namespace SetLgg { namespace Machine { namespace Programme { DECL_SHARED_PTR(InstructionExecution); } } }

#include <src/machine/programme/donnees/instruction.h>
#include <src/machine/programme/donnees/valeur.h>
#include <src/machine/programme/donnees/nomflux.h>
#include <src/machine/flux/gestionnaireflux.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			typedef std::vector<ValeurSP> ListeValeurs;
			class InstructionExecution: public Instruction
			{
				public:
					InstructionExecution(const SetLgg::Machine::Programme::NomFluxSP& descripteur, const ListeValeurs& valeurs, const SetLgg::Global::Source::PositionSP& position)
					:Instruction(position), _descripteur(descripteur), _valeurs(valeurs) {};
					InstructionExecution(const SetLgg::Machine::Programme::NomFluxSP& descripteur, const SetLgg::Machine::Programme::NomFluxSP& erreur, const ListeValeurs& valeurs, const SetLgg::Global::Source::PositionSP& position)
					:Instruction(position), _descripteur(descripteur), _erreur(erreur), _valeurs(valeurs) {};
					virtual ~InstructionExecution() {};
					virtual void execution(SetLgg::Machine::Machine::MachineSP& machine) const;
					template<typename oStream>
					friend oStream& operator<<(oStream&,const InstructionExecution&);
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
				private:
					SetLgg::Machine::Programme::NomFluxSP _descripteur;
					SetLgg::Machine::Programme::NomFluxSP _erreur;
					ListeValeurs _valeurs;
			};
			template<typename oStream>
			oStream& operator<<(oStream& os,const InstructionExecution& instruction)
			{
				os << ":exec " << *(instruction._descripteur) << " ";
				if(instruction._erreur)
				{
					os << *(instruction._erreur) << " ";
				}
				for(auto it = instruction._valeurs.cbegin() ; it!=instruction._valeurs.cend() ; ++it)
				{
					if(it!=instruction._valeurs.cbegin())
					{
						os << " , ";
					}
					os << **it;
				}
				return os;
			};

			class ParametreIncorrect : public SetLgg::Global::Exception::Execution
			{
				public:
					ParametreIncorrect(const std::string& parametre)
					:Execution(SIGILL,std::string("Invalid parameter ")+parametre+" in command execution") {};
			};

			class ImpossibleDExecuterCommande : public SetLgg::Global::Exception::Execution
			{
				public:
					ImpossibleDExecuterCommande(const std::string& details = std::string())
					:Execution(SIGABRT,std::string("Unable to execute command")+details) {};
			};

			class ImpossibleDeCommuniquerAvecLaCommande : public SetLgg::Global::Exception::Execution
			{
				public:
					ImpossibleDeCommuniquerAvecLaCommande(const std::string& descripteur, const std::string& commande = std::string(""))
					:Execution(SIGABRT,std::string("Unable to communicate with command, as stream ")+descripteur+" already exist"+commande) {};
			};
		}
	}
}
#endif
