/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_INSTRUCTIONOUVERTURE_H_
#define _MACHINE_PROGRAMME_DONNEES_INSTRUCTIONOUVERTURE_H_

#include <src/global/global.h>

namespace SetLgg { namespace Machine { namespace Programme { DECL_SHARED_PTR(InstructionOuverture); } } } 

#include <src/machine/programme/donnees/instruction.h>
#include <src/machine/programme/donnees/nomflux.h>
#include <src/machine/flux/gestionnaireflux.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{

			DECL_SHARED_PTR(InstructionOuvertureFlux);

			enum class Acces {LECTURE, ECRITURE, LECTUREECRITURE, ECRITUREAJOUT};

			template<typename oStream>
			oStream& operator<<(oStream& os, const Acces& acces)
			{
				switch(acces)
				{
					case Acces::LECTURE:
						return os << "<";
					case Acces::ECRITURE:
						return os << ">";
					case Acces::LECTUREECRITURE:
						return os << "< >";
					case Acces::ECRITUREAJOUT:
						return os << "> >";
				}
				return os;
			}

			enum class Protocole {TCP,UDP};

			template<typename oStream>
			oStream& operator<<(oStream& os, const Protocole& protocole)
			{
				switch(protocole)
				{
					case Protocole::TCP:
						return os << "TCP";
					case Protocole::UDP:
						return os << "UDP";
				}
				return os;
			}

			class InstructionOuvertureFlux
			{
				public:
					virtual std::ostream& format(std::ostream& os) const = 0;
					virtual SetLgg::Machine::Flux::FluxSP operator() (SetLgg::Machine::Machine::MachineSP& machine) const = 0;
			};

			class InstructionOuvertureFluxFichier : public InstructionOuvertureFlux
			{
				public:
					InstructionOuvertureFluxFichier(const Acces& acces, const ValeurSP& nom)
					:_acces(acces),_nom(nom) {};

					virtual std::ostream& format(std::ostream& os) const
					{
						return os << _acces << " " << *_nom;
					};
					virtual SetLgg::Machine::Flux::FluxSP operator() (SetLgg::Machine::Machine::MachineSP& machine) const;
				private:
					Acces _acces;
					SetLgg::Machine::Programme::ValeurSP _nom;
			};

			class InstructionOuvertureFluxSocket : public InstructionOuvertureFlux
			{
				public:
					InstructionOuvertureFluxSocket(const Protocole& protocole, const ValeurSP& ip_locale, const ValeurSP& port_local, const Acces& acces, const ValeurSP& ip_distante, const ValeurSP& port_distant)
					:_protocole(protocole), _ip_locale(ip_locale), _port_local(port_local), _acces(acces), _ip_distante(ip_distante), _port_distant(port_distant) {};

					virtual std::ostream& format(std::ostream& os) const
					{
						if(_ip_locale and not _ip_distante)
						{
							return os << _protocole << " " << _acces << " " << *_ip_locale << " " << *_port_local;
						}
						if(_ip_distante and not _ip_locale)
						{
							return os << _protocole << " " << _acces << " " << *_ip_distante << " " << *_port_distant;
						}
						return os << _protocole << " " << _acces << " " << *_ip_locale << " " << *_port_local << " " << *_ip_distante << " " << *_port_distant;
					};
					virtual SetLgg::Machine::Flux::FluxSP operator() (SetLgg::Machine::Machine::MachineSP& machine) const;
				private:
					Protocole _protocole;
					ValeurSP _ip_locale;
					ValeurSP _port_local;
					Acces _acces;
					ValeurSP _ip_distante;
					ValeurSP _port_distant;
			};

			class InstructionOuverture: public Instruction
			{
				public:
					InstructionOuverture(const SetLgg::Machine::Programme::NomFluxSP& descripteur, const InstructionOuvertureFluxSP& flux, const SetLgg::Global::Source::PositionSP& position)
					:Instruction(position), _descripteur(descripteur), _flux(flux) {};
					virtual ~InstructionOuverture() {};
					virtual void execution(SetLgg::Machine::Machine::MachineSP& machine) const;
					template<typename oStream>
					friend oStream& operator<<(oStream& os,const InstructionOuverture& instruction)
					{
						os << ":open " << *(instruction._descripteur) << " ";
						return instruction._flux->format(os);
					};
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
				private:
					SetLgg::Machine::Programme::NomFluxSP _descripteur;
					InstructionOuvertureFluxSP _flux;
					
			};
		}
	}
}
#endif
