/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <src/machine/programme/donnees/instructionreception.h>
#include <src/machine/machine/machine.h>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/programme/donnees/instructionenvoi.h>
#include <src/machine/programme/transfert/analyseur.h>
#include <src/machine/memoire/analyseur/analyseur.h>
using namespace SetLgg::Machine::Programme;

SetLgg::Machine::Memoire::ValeurSP InstructionReception::execution_avec_resultat(SetLgg::Machine::Machine::MachineSP& machine) const
{
	SetLgg::Machine::Memoire::NomFlux nom_flux = _nom_flux->evaluation_basique(machine);
	SetLgg::Machine::Memoire::ValeurCSP valeur_nom_bloc = _nom_bloc->evaluation(machine);
	SetLgg::Machine::Memoire::ChaineCSP nom_bloc = std::dynamic_pointer_cast<const SetLgg::Machine::Memoire::Chaine>(valeur_nom_bloc);
	SetLgg::Machine::Memoire::Memoire::BlocMemoireSP bloc;
	if(_bloc_memoire)
	{
		bloc.reset(new SetLgg::Machine::Memoire::Memoire::BlocMemoire((*_bloc_memoire)(machine)));
	}
	return execution_interne(machine,nom_flux,*nom_bloc,bloc);
}

SetLgg::Machine::Memoire::ValeurSP InstructionReception::execution_interne(SetLgg::Machine::Machine::MachineSP& machine, const SetLgg::Machine::Memoire::NomFlux& nom_flux, const std::string nom_bloc, SetLgg::Machine::Memoire::Memoire::BlocMemoireSP& bloc)
{
	SetLgg::Machine::Flux::FluxSP flux=(*(machine->_flux))[nom_flux];
	if(not flux)
	{
		std::ostringstream oss;
		nom_flux.format(oss);
		throw SetLgg::Machine::Flux::FluxInexistant(oss.str());
	}
	if(not flux->lisible())
	{
		std::ostringstream oss;
		nom_flux.format(oss);
		throw SetLgg::Machine::Flux::FluxOperationInvalide(oss.str());
	}
	if(flux->media_actif() and not flux->inscriptible())
	{
		std::ostringstream oss;
		nom_flux.format(oss);
		throw SetLgg::Machine::Flux::FluxOperationInvalide(oss.str());
	}
	try
	{
		SetLgg::Machine::Programme::Transfert::TransfertSP requete = reception(flux);
		if(not requete)
		{
			throw MessageInvalide();
		}
		SetLgg::Machine::Programme::Transfert::RequeteBlocMemoireSP bloc_memoire = std::dynamic_pointer_cast<SetLgg::Machine::Programme::Transfert::RequeteBlocMemoire>(requete);
		if(not bloc_memoire)
		{
			throw MessageInvalide();
		}
		if(bloc_memoire->_nom!=nom_bloc)
		{
			throw BlocIncorrect(bloc_memoire->_nom,nom_bloc);
		}
		if(bloc)
		{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << bloc_memoire->_taille << " " << bloc->_taille << std::endl;
#endif
			if(bloc->_taille!=bloc_memoire->_taille)
			{
				std::stringstream oss_recu;
				oss_recu << bloc_memoire->_taille ;
				std::stringstream oss_attendu;
				oss_attendu << bloc->_taille;
				throw StructureBlocIncorrecte(bloc_memoire->_nom,oss_recu.str(),oss_attendu.str());
			}
		}
		SetLgg::Machine::Memoire::MemoireSP memoire;
		try
		{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << bloc_memoire->_bloc << std::endl;
#endif
			memoire = SetLgg::Machine::Memoire::Analyseur::analyse_chaine(bloc_memoire->_bloc,machine->_plugins);
		}
		catch(SetLgg::Global::Exception::Compilation& erreur)
		{
			throw SyntaxeIncorrecte();
		}
		if(not memoire->verification_taille(bloc_memoire->_taille))
		{
			std::stringstream oss_taille;
			oss_taille << bloc_memoire->_taille ;
			throw StructureBlocIncorrecte(bloc_memoire->_nom,oss_taille.str());
		}
		if(bloc)
		{
			if(not machine->_memoire->importe(*bloc,memoire))
			{
				throw TypesIncorrects();
			}
		}
		else
		{
			bloc.reset(new SetLgg::Machine::Memoire::Memoire::BlocMemoire(machine->_memoire->importe(memoire)));
			machine->_processeur->ajoute_allocation_memoire(*bloc);
		}
		machine->_memoire->decalage(*bloc,*bloc_memoire);
		if(flux->media_actif())
		{
			SetLgg::Machine::Programme::Transfert::TransfertSP reponse(new SetLgg::Machine::Programme::Transfert::ReponseOK());
			InstructionEnvoi::envoi(flux,reponse);
		}
		SetLgg::Machine::Memoire::ValeurSP retour(new SetLgg::Machine::Memoire::Adresse(bloc->_adresse));
		return retour;
	}
	catch(SetLgg::Global::Exception::Execution& erreur)
	{
		if(flux->media_actif())
		{
			SetLgg::Machine::Programme::Transfert::TransfertSP reponse(new SetLgg::Machine::Programme::Transfert::ReponseErreur(erreur,erreur));
			InstructionEnvoi::envoi(flux,reponse);
		}
		throw;
	}
}

SetLgg::Machine::Programme::Transfert::TransfertSP InstructionReception::reception(SetLgg::Machine::Flux::FluxSP& flux)
{
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
	SetLgg::Machine::Flux::DonneesLues entete_lu = flux->lecture(SetLgg::Machine::Flux::TypeLecture::Type::LIGNE);
	if(entete_lu)
	{
		throw SetLgg::Machine::Programme::Transfert::EnteteInvalide();
	}
	std::string entete = entete_lu;
	size_t taille=0;
	try
	{
		taille=SetLgg::Machine::Programme::Transfert::Transfert::decode_entete(entete);
	}
	catch(SetLgg::Machine::Programme::Transfert::EnteteInvalide& e)
	{
		flux->tampon_empile(entete+"\n");
		throw;
	}
	SetLgg::Machine::Flux::DonneesLues requete_lue = flux->lecture(SetLgg::Machine::Flux::TypeLecture(SetLgg::Machine::Flux::TypeLecture::Type::TAILLE,taille));
	if(requete_lue)
	{
		flux->tampon_empile(entete+"\n");
		throw CorpsInvalide();
	}
	std::string requete = requete_lue;
#ifdef SETLGG_DEBUG
SETLGG_TRACE << requete.size() << " " << requete << std::endl;
#endif
	if(requete.size()<taille)
	{
		flux->tampon_empile(entete+"\n");
		flux->tampon_empile(requete);
		throw CorpsInvalide();
	}
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
	try
	{
		SetLgg::Machine::Programme::Transfert::TransfertSP message = SetLgg::Machine::Programme::Transfert::Analyseur::analyse_chaine(requete);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
		return message;
	}
	catch(SetLgg::Global::Exception::Compilation& erreur)
	{
		flux->tampon_empile(entete+"\n");
		flux->tampon_empile(requete);
#ifdef SETLGG_DEBUG
SETLGG_TRACE << std::endl;
#endif
		throw SyntaxeIncorrecte();
	}
}
