/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <src/machine/memoire/donnees/entier.h>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/memoire/donnees/booleen.h>
#include <src/machine/memoire/donnees/nomflux.h>
#include <src/machine/memoire/donnees/adresse.h>
#include <src/machine/memoire/donnees/utilisateur.h>
#include <src/machine/plugin/donnees/gestionnaireplugins.h>
#include <src/machine/plugin/donnees/contenu.h>
#include <src/machine/plugin/interface/setlgg_machine.h>
#include <src/machine/programme/donnees/instructionutilisateur.h>
#include <src/machine/machine/machine.h>
using namespace SetLgg::Machine::Programme;

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			namespace Outil
			{
				struct MemoireUtilisateur
				{
					MemoireUtilisateur(SetLgg::Machine::Plugin::GestionnairePluginSP plugins)
						:nombre_parametres(0), parametres(0), retour(0), _plugins(plugins)
					{
					};
					~MemoireUtilisateur()
					{
						if(parametres)
						{
							for(auto it = parametres ; it!=(parametres+nombre_parametres) ; ++it)
							{
								vide<false,true>(it);
							}
							delete [] parametres;
						}
						if(retour)
						{
							vide<true,false>(retour);
						}
					};

					size_t nombre_parametres;
					SetLgg_Machine_Value *parametres;
					SetLgg_Machine_Value *retour;
					SetLgg::Machine::Plugin::GestionnairePluginSP _plugins;
					private:
					template<bool supprime_structure, bool supprime_user>
					void vide(SetLgg_Machine_Value *value)
					{
						if(supprime_user and (value->_type._type==SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE))
						{
							SetLgg::Machine::Plugin::TypeUtilisateurSP type = _plugins->type(std::string(value->_type._user_name._buffer,value->_type._user_name._size));
							type->_destruction(value->_value._user_type);
						}
						if(supprime_structure)
							::setlgg_machine_value_delete(value);
						else
							::setlgg_machine_value_clear(value);
					};
				};
			}
		}
	}
}

SetLgg::Machine::Memoire::ValeurSP InstructionUtilisateur::execution_avec_resultat(SetLgg::Machine::Machine::MachineSP& machine) const
{
	Outil::MemoireUtilisateur memoire(_plugins);
	if(_retour_demande and not _instruction->_retour)
	{
		throw InstructionSansRetourDansAffectation(_nom_instruction);
	}
	if(_parametres.size()<_instruction->_parametres.size())
	{
		std::ostringstream oss;
		oss << _parametres.size();
		std::ostringstream oss_min;
		oss_min << _instruction->_parametres.size();
		throw InstructionNombreParametresIncorrect(_nom_instruction, oss.str(), oss_min.str());
	}
	SetLgg::Machine::Plugin::ListeParametres liste_parametres;
	{
		for(auto it=_parametres.cbegin() ; it!=_parametres.cend() ; ++it)
		{
			liste_parametres.push_back(it->evaluation(machine));
		}
	}
	{
		auto itval = liste_parametres.cbegin();
		auto ittype = _instruction->_parametres.cbegin();
		size_t indice = 1;
		for( ; ittype!=_instruction->_parametres.cend() ; ++itval, ++ittype, ++indice)
		{
			(*ittype)->conformite(*itval,_nom_instruction,indice);
		}
	}
	memoire.nombre_parametres = liste_parametres.size();
	memoire.parametres = new SetLgg_Machine_Value[memoire.nombre_parametres];
	SetLgg_Machine_Value *iterateur = memoire.parametres;
	for(auto it=liste_parametres.cbegin() ; it!=liste_parametres.cend() ; ++it,++iterateur)
	{
		*iterateur = *it;
	}
	
	memoire.retour = _instruction->_instruction(&machine,memoire.nombre_parametres,memoire.parametres);

	if(memoire.retour)
	{
		if(not _instruction->_retour)
		{
			throw InstructionDevraitPasRetournerUneValeur(_nom_instruction);
		}
		SetLgg::Machine::Memoire::ValeurSP valeur_retour;
		switch(memoire.retour->_type._type)
		{
			case SetLgg_Machine_ValueTypeCategory::VALUE_INTEGER:
				if(memoire.retour->_initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::EntierNul());
				else
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Entier(memoire.retour->_value._integer));
				break;
			case SetLgg_Machine_ValueTypeCategory::VALUE_STRING:
				if(memoire.retour->_initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::ChaineNulle());
				else
				{
					if(not memoire.retour->_value._string._buffer)
					{
						throw InstructionRetourChaineInexistante(_nom_instruction);
					}
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Chaine(std::string(memoire.retour->_value._string._buffer,memoire.retour->_value._string._size)));
				}
				break;
			case SetLgg_Machine_ValueTypeCategory::VALUE_BOOLEAN:
				if(memoire.retour->_initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::BooleenNul());
				else
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Booleen(memoire.retour->_value._boolean));
				break;
			case SetLgg_Machine_ValueTypeCategory::VALUE_POINTER:
				if(memoire.retour->_initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::AdresseNulle());
				else
					valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Adresse(memoire.retour->_value._pointer));
				break;
			case SetLgg_Machine_ValueTypeCategory::VALUE_INOUTREF:
				{
					if(memoire.retour->_initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
						valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFluxNul());
					else
					{
						switch(memoire.retour->_value._inoutref._type)
						{
							case SetLgg_Machine_InOutReferenceType::IOR_STDIN:
								{
									if(memoire.retour->_value._inoutref._name._buffer)
									{
										throw InstructionRetourNomPresent(_nom_instruction);
									}
									valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDIN));
									break;
								}
							case SetLgg_Machine_InOutReferenceType::IOR_STDOUT:
								{
									if(memoire.retour->_value._inoutref._name._buffer)
									{
										throw InstructionRetourNomPresent(_nom_instruction);
									}
									valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDOUT));
									break;
								}
							case SetLgg_Machine_InOutReferenceType::IOR_STDERR:
								{
									if(memoire.retour->_value._inoutref._name._buffer)
									{
										throw InstructionRetourNomPresent(_nom_instruction);
									}
									valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(SetLgg::Machine::Memoire::NomFlux::Type::STDERR));
									break;
								}
							case SetLgg_Machine_InOutReferenceType::IOR_CUSTOM:
								{
									if(not memoire.retour->_value._inoutref._name._buffer)
									{
										throw InstructionRetourNomAbsent(_nom_instruction);
									}
									valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::NomFlux(std::string(memoire.retour->_value._inoutref._name._buffer,memoire.retour->_value._inoutref._name._size)));
									break;
								}
							default:
								throw InstructionRetourNomInvalide(_nom_instruction);

						}
					}
				}
				break;
			case SetLgg_Machine_ValueTypeCategory::VALUE_USERTYPE:
				{
					if(not memoire.retour->_type._user_name._buffer)
					{
						throw InstructionRetourTypeUtilisateurNonSpecifie(_nom_instruction);
					}
					std::string str_type(memoire.retour->_type._user_name._buffer,memoire.retour->_type._user_name._size);
					if(memoire.retour->_initialisation==SetLgg_Machine_ValueInitialisation::NULL_VALUE)
						valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::UtilisateurNul(_plugins->type(str_type,SetLgg::Global::Source::PositionSP())));
					else
					{
						if(not memoire.retour->_value._user_type)
						{
							throw InstructionRetourTypeUtilisateurInexistant(_nom_instruction);
						}
						{
							valeur_retour = SetLgg::Machine::Memoire::ValeurSP(new SetLgg::Machine::Memoire::Utilisateur(_plugins->type(str_type,SetLgg::Global::Source::PositionSP()),memoire.retour->_value._user_type));
						}
					}
				}
				break;
			default:
				{
					throw InstructionRetourTypeInconnu(_nom_instruction);
				}
				break;
		}
		return valeur_retour;
	}
	else
	{
		if(_instruction->_retour)
		{
			throw InstructionDevraitRetournerUneValeur(_nom_instruction);
		}
		return SetLgg::Machine::Memoire::ValeurSP();
	}
}

SetLgg::Machine::Plugin::Parametre Parametre::evaluation(SetLgg::Machine::Machine::MachineSP& machine) const
{
	if(_valeur)
	{
		return SetLgg::Machine::Plugin::Parametre(_valeur->evaluation(machine));
	}
	if(_adresse)
	{
		return SetLgg::Machine::Plugin::Parametre(_adresse->evaluation(machine));
	}
	throw;
};
