/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_INSTRUCTIONUTILISATEUR_H_
#define _MACHINE_PROGRAMME_DONNEES_INSTRUCTIONUTILISATEUR_H_

#include <src/global/global.h>

namespace SetLgg { namespace Machine { namespace Programme { DECL_SHARED_PTR(InstructionUtilisateur); } } }

#include <src/machine/programme/donnees/instruction.h>
#include <src/machine/programme/donnees/valeur.h>
#include <src/machine/programme/donnees/adressememoire.h>
#include <src/machine/plugin/donnees/instructionutilisateur.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Plugin
		{
			class Parametre;
			DECL_SHARED_PTR(GestionnairePlugin);
		}
		namespace Programme
		{
			class Parametre
			{
				public:
					Parametre(const ValeurSP& valeur)
					:_valeur(valeur) {};
					Parametre(const AdresseMemoireSP& adresse)
					:_adresse(adresse) {};
					template<typename oStream>
					friend oStream& operator<<(oStream& os, const Parametre& parametre)
					{
						if(parametre._valeur)
						{
							os << *(parametre._valeur);
						}
						if(parametre._adresse)
						{
							os << *(parametre._adresse);
						}
						return os;
					};
					SetLgg::Machine::Plugin::Parametre evaluation(SetLgg::Machine::Machine::MachineSP& machine) const;
				private:
					ValeurSP _valeur;
					AdresseMemoireSP _adresse;
			};

			typedef std::vector<Parametre> ListeParametres;
			class InstructionUtilisateur: public Instruction
			{
				public:
					InstructionUtilisateur(const std::string& nom_instruction, const ListeParametres& parametres, const SetLgg::Machine::Plugin::InstructionUtilisateurSP& instruction, const SetLgg::Machine::Plugin::GestionnairePluginSP& plugins, const SetLgg::Global::Source::PositionSP& position)
					:Instruction(position), _nom_instruction(nom_instruction), _parametres(parametres), _instruction(instruction), _plugins(plugins), _retour_demande(false) { }
					virtual ~InstructionUtilisateur() {};
					virtual void execution(SetLgg::Machine::Machine::MachineSP& machine) const
					{
						execution_avec_resultat(machine);
					}
					virtual SetLgg::Machine::Memoire::ValeurSP execution_avec_resultat(SetLgg::Machine::Machine::MachineSP& machine) const;
					template<typename oStream>
					friend oStream& operator<<(oStream&,const InstructionUtilisateur&);
					virtual std::ostream& format(std::ostream& os) const
					{
						return os << (*this);
					};
					void retour_demande()
					{
						_retour_demande=true;
					}
				private:
					std::string _nom_instruction;
					ListeParametres _parametres;
					SetLgg::Machine::Plugin::InstructionUtilisateurSP _instruction;
					SetLgg::Machine::Plugin::GestionnairePluginSP _plugins;
					bool _retour_demande;
			};
			template<typename oStream>
			oStream& operator<<(oStream& os,const InstructionUtilisateur& instruction)
			{
				os << ":" << (instruction._nom_instruction);
				for(auto it = instruction._parametres.cbegin() ; it!=instruction._parametres.cend() ; ++it)
				{
					os << " " << *it;
				}
				return os;
			};

			class InstructionSansRetourDansAffectation : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionSansRetourDansAffectation(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" without a return value used in affectation") {};
			};

			class InstructionNombreParametresIncorrect : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionNombreParametresIncorrect(const std::string& nom_instruction, const std::string& nb_params, const std::string& nb_params_min)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" is called with "+nb_params+" parameters but requires at least "+nb_params_min+" parameters") {};
			};

			class  InstructionDevraitPasRetournerUneValeur : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionDevraitPasRetournerUneValeur(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" should not return a value") {};
			};

			class  InstructionDevraitRetournerUneValeur : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionDevraitRetournerUneValeur(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" should return a value") {};
			};

			class  InstructionRetourChaineInexistante : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionRetourChaineInexistante(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" returned a null string") {};
			};

			class  InstructionRetourNomPresent : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionRetourNomPresent(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" returned a in out reference with a name") {};
			};

			class  InstructionRetourNomAbsent : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionRetourNomAbsent(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" returned a in out reference without a name") {};
			};

			class  InstructionRetourNomInvalide : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionRetourNomInvalide(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" returned a in out reference with an invalid type") {};
			};

			class  InstructionRetourTypeUtilisateurNonSpecifie : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionRetourTypeUtilisateurNonSpecifie(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" returned a user type without naming it") {};
			};

			class  InstructionRetourTypeUtilisateurInexistant : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionRetourTypeUtilisateurInexistant(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" returned a null user type") {};
			};

			class  InstructionRetourTypeInconnu : public SetLgg::Global::Exception::Execution
			{
				public:
					InstructionRetourTypeInconnu(const std::string& nom_instruction)
					:Execution(SIGILL,std::string("Instruction ")+nom_instruction+" returned a unkown type") {};
			};
		}
	}
}
#endif
