/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <src/machine/programme/donnees/programme.h>
#include <src/machine/programme/donnees/instructionlabel.h>
#include <src/machine/programme/donnees/instructionarret.h>
#include <src/machine/programme/donnees/condition.h>
#include <src/machine/memoire/donnees/entier.h>
using namespace SetLgg::Machine::Programme;

void Programme::ajout_instruction(const InstructionSP instruction)
{
	SETLGG_TEST(instruction);
	const InstructionLabelSP instructionlabel = std::dynamic_pointer_cast<InstructionLabel>(instruction);
	if(instructionlabel)
	{
		auto resultat = _labels.insert(std::pair<Label,Adresse>(*instructionlabel,_adresse_terminale));
		if(not resultat.second)
		{
			throw LabelEnDouble(*instructionlabel,*instructionlabel);
		}
	}
	else
	{
		_instructions.push_back(instruction);
		++_adresse_terminale;
	}
}

void Programme::ajout_programme(const ProgrammeSP programme)
{
	for(auto it=programme->_labels.cbegin() ; it!=programme->_labels.cend() ; ++it)
	{
		Adresse adresse = _adresse_terminale;
		adresse += it->second;
		_labels[it->first] = adresse;
	}
	_instructions.insert(_instructions.end(),programme->_instructions.cbegin(),programme->_instructions.cend());
	_adresse_terminale += Adresse(programme->_instructions.size());
}

InstructionSP Programme::operator[] (const Adresse& adresse) const
{
	if(adresse<_adresse_terminale)
	{
		return _instructions[adresse];
	}
	if(adresse==_adresse_terminale)
	{
		ValeurSP valeur(new SetLgg::Machine::Memoire::Entier(0));
		ConditionSP condition(new Condition());
		SetLgg::Global::Source::PositionSP position;
		InstructionSP exit(new InstructionArret(valeur,condition,position));
		return exit;
	}
	std::ostringstream oss;
	oss << adresse;
	throw AdresseProgrammeIncorrecte(oss.str());
}

Adresse Programme::lien_statique(const Label& label, const SetLgg::Global::Source::PositionSP& position) const
{
	auto adresse = _labels.find(label);
	if(adresse == _labels.end())
	{
		throw LabelStatiqueInexistant(label,position);
	}
	return adresse->second;
}

Adresse Programme::lien_dynamique(const Label& label) const
{
	auto adresse = _labels.find(label);
	if(adresse == _labels.end())
	{
		throw LabelDynamiqueInexistant(label);
	}
	return adresse->second;
}

void Programme::edition_liens()
{
	for(Adresse adresse = Adresse() ; adresse<_adresse_terminale ; ++adresse)
	{
		(*this)[adresse]->edition_liens(this->shared_from_this());
	}
}

std::pair<bool,Adresse> Programme::lien(const std::string& label) const
{
	auto it = _labels.find(Label(label));
	if(it==_labels.end())
	{
		return std::pair<bool,Adresse>(false,_adresse_terminale);
	}
	return std::pair<bool,Adresse>(true,it->second);
}
