/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_DONNEES_PROGRAMME_H_
#define _MACHINE_PROGRAMME_DONNEES_PROGRAMME_H_

#include <vector>
#include <map>

#include <src/global/global.h>
#include <src/global/exceptions.h>
#include <src/global/sources/position.h>

namespace SetLgg { namespace Machine { namespace Programme { DECL_SHARED_PTR(Programme); } } }

#include <src/machine/programme/donnees/instruction.h>
#include <src/machine/programme/donnees/label.h>
#include <src/machine/programme/donnees/adresse.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Processeur { class Processeur; }
		namespace Debogueur { class Debogueur; }
		namespace Plugin { class Interface; }
		namespace Programme
		{
			class InstructionSaut;
			class InstructionAppel;
			class InstructionInterruption;

			class Programme : public std::enable_shared_from_this<Programme>
			{
				friend class InstructionSaut;
				friend class InstructionAppel;
				friend class InstructionInterruption;
				friend class Processeur::Processeur;
				friend class Debogueur::Debogueur;
				friend class Plugin::Interface;
				public:
					Programme() = default;
					Programme(const Programme&) = delete;
					void ajout_instruction(const InstructionSP instruction);
					void ajout_programme(const ProgrammeSP programme);
					void edition_liens();
					template<typename oStream>
					friend oStream& operator<<(oStream&,const Programme&);
					std::pair<bool,Adresse> lien(const std::string& label) const;
					size_t taille() const { return _instructions.size(); };
				private:
					std::vector<InstructionSP> _instructions;
					std::map<Label,Adresse> _labels;
					Adresse _adresse_terminale;
					InstructionSP operator[] (const Adresse& adresse) const;
					Adresse lien_statique(const Label& label, const SetLgg::Global::Source::PositionSP& position) const;
					Adresse lien_dynamique(const Label& label) const;
			};

			template<typename oStream>
			oStream& operator<<(oStream& os, const Programme& programme)
			{
				std::multimap<Adresse,Label> labels_inverses;
				for(auto it = programme._labels.cbegin() ; it!=programme._labels.cend() ; ++it)
				{
					labels_inverses.insert(std::pair<Adresse,Label>(it->second,it->first));
				}
				os << "# Program:" << std::endl;
				for(Adresse adresse = Adresse() ; adresse<=programme._adresse_terminale ; ++adresse)
				{
					auto premier_label = labels_inverses.lower_bound(adresse);
					auto apres_dernier_label = labels_inverses.upper_bound(adresse);
					for(auto label = premier_label ; label!=apres_dernier_label ; ++label)
					{
						os << ":label " << (label->second) << std::endl;
					}
					if(adresse==programme._adresse_terminale)
						break;
					os << "\t" << *(programme[adresse]) << "\t# " << adresse;
					os << std::endl;
				}
				os << "# End of program" << std::endl;
				return os;
			}

			class LabelEnDouble : public SetLgg::Global::Exception::Compilation
			{
				public:
					LabelEnDouble(const Label& label, const SetLgg::Global::Source::PositionSP& position)
					:Compilation(position,std::string("Duplicate label ")+std::string(label)+std::string(" found")) {};
			};

			class LabelStatiqueInexistant : public SetLgg::Global::Exception::Compilation
			{
				public:
					LabelStatiqueInexistant(const Label& label, const SetLgg::Global::Source::PositionSP& position)
					:Compilation(position,std::string("Inexistant label ")+std::string(label)+std::string(" to jump to")) {};
			};

			class LabelDynamiqueInexistant : public SetLgg::Global::Exception::Execution
			{
				public:
					LabelDynamiqueInexistant(const Label& label)
					:Execution(SIGILL,std::string("Inexistant label ")+std::string(label)+std::string(" to jump to")) {};
			};

			class AdresseProgrammeIncorrecte : public SetLgg::Global::Exception::Execution
			{
				public:
					AdresseProgrammeIncorrecte(const std::string& adresse)
					:Execution(SIGABRT,std::string("Impossible to jump to address ")+adresse) {};
			};
		}
	}
}

#endif
