/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_PROGRAMME_TRANSFERT_TRANSFERT_H_
#define _MACHINE_PROGRAMME_TRANSFERT_TRANSFERT_H_

#include <src/global/global.h>
#include <src/global/exceptions.h>

namespace SetLgg { namespace Machine { namespace Programme { namespace Transfert { DECL_SHARED_PTR(Transfert); } } } }

#include <src/machine/memoire/donnees/memoire.h>

namespace SetLgg
{
	namespace Machine
	{
		namespace Programme
		{
			class InstructionEnvoi;
			class InstructionReception;
			namespace Transfert
			{
				class Transfert
				{
					public:
						virtual ~Transfert() {};
						template<typename oStream>
							friend oStream& operator<<(oStream&,const Transfert&);
						virtual operator bool () const { return true; };
						virtual std::ostream& format(std::ostream& os) const =0;
						static std::string encode_entete(const std::string& requete);
						static size_t decode_entete(const std::string& entete);
				};
				template<typename oStream>
					oStream& operator<<(oStream& os,const Transfert& transfert)
					{
						os << "VERSION 1" << std::endl;
						transfert.format(os);
						return os;
					};

				DECL_SHARED_PTR(RequeteBlocMemoire);
				class RequeteBlocMemoire : public Transfert
				{
					friend class SetLgg::Machine::Programme::InstructionReception;
					public:
						RequeteBlocMemoire(const std::string& nom, const SetLgg::Machine::Memoire::Adresse& debut, const size_t taille, const std::string& bloc_memoire)
						:_nom(SetLgg::Global::Chaine::desechappe(nom)), _debut(debut), _taille(taille), _bloc(bloc_memoire) {}
						RequeteBlocMemoire(const std::string& nom, const SetLgg::Machine::Memoire::Memoire::BlocMemoire& bloc, const std::string& bloc_memoire)
						:RequeteBlocMemoire(nom, bloc._adresse, bloc._taille, bloc_memoire) {}

						virtual ~RequeteBlocMemoire() {};
						virtual operator bool () const { return false; };
						virtual std::ostream& format(std::ostream& os) const
						{
							os	<< "BLOCK \"" << SetLgg::Global::Chaine::echappe(_nom) << "\"" << std::endl
								<< "ADDRESS " << _debut << std::endl
								<< "SIZE " << _taille << std::endl
								<< "MEMORY" << std::endl
								<< _bloc;
							return os;
						}
						operator SetLgg::Machine::Memoire::Memoire::BlocMemoire () const
						{
							return SetLgg::Machine::Memoire::Memoire::BlocMemoire(_debut,_taille);
						}
					private:
						std::string _nom;
						SetLgg::Machine::Memoire::Adresse _debut;
						size_t _taille;
						std::string _bloc;
				};

				class ReponseOK : public Transfert
				{
					public:
						virtual ~ReponseOK() {};
						virtual std::ostream& format(std::ostream& os) const
						{
							os << "OK" << std::endl;
							return os;
						};
				};

				DECL_SHARED_PTR(ReponseErreur);
				class ReponseErreur : public Transfert
				{
					friend class SetLgg::Machine::Programme::InstructionEnvoi;
					public:
						ReponseErreur(const size_t interruption, const std::string& message)
						:_interruption(interruption), _message(SetLgg::Global::Chaine::desechappe(message)) {}
						virtual ~ReponseErreur() {};
						virtual std::ostream& format(std::ostream& os) const
						{
							os << "ERROR " << SetLgg::Global::Exception::Execution::nom_interruption(_interruption) << " \"" << SetLgg::Global::Chaine::echappe(_message) << "\"" << std::endl;
							return os;
						};
					private:
						size_t _interruption;
						std::string _message;
				};

				struct EnteteInvalide : public SetLgg::Global::Exception::Execution
				{
					EnteteInvalide()
						:SetLgg::Global::Exception::Execution(SIGABRT,"Invalid message header") {}
				};

			}
		}
	}
}
#endif
