/* 
 *  This program is a non relational database language running on a small
 *  virtual machine.
 *  Copyright (C) 2012 Julien Bruguier.
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#ifndef _MACHINE_SETLGGMACHINE_H_
#define _MACHINE_SETLGGMACHINE_H_
#include <iostream>
#include <unistd.h>

#include <config.h>

#include <src/global/exceptions.h>

#include <src/machine/machine/machine.h>
#include <src/machine/memoire/donnees/entier.h>
#include <src/machine/memoire/donnees/chaine.h>
#include <src/machine/memoire/donnees/booleen.h>
#include <src/machine/memoire/donnees/adresse.h>
#include <src/machine/plugin/analyseur/analyseur.h>

using namespace SetLgg::Machine;

void affiche_aide(std::ostream& os, const std::string commande)
{
	os	<< commande << " [options] program_file memory_file"
		<< std::endl << "where program_file contains the program to run, memory_file contains the memory from the last run." << std::endl
		<< "If the memory file does not exist, it will be created." << std::endl
		<< "Options:" << std::endl
		<< "-d            : boots the virtual machine in debug mode. Enter help in the debugger command line for more details." << std::endl
		<< "-D commands   : same as -d, but runs the commands in the debugger at startup." << std::endl
		<< "-c core_file  : optional file name where the state of the virtual machine is dumped. The name will be suffixed by the pid." << std::endl
		<< "-p plugin_file: optional file name containing a valid .so file to load at startup. This option can be repeated." << std::endl
		<< "-h            : display this help and exit." << std::endl
		<< "-v            : display the software version." << std::endl
		<< std::endl;
}

int main(int argc, char *argv[])
{
	std::string fichier_programme;
	std::string fichier_memoire;
	std::string fichier_coredump;
	std::vector<std::string> liste_plugins;
	SetLgg::Machine::Machine::Machine::TypeExecution mode_debogage = SetLgg::Machine::Machine::Machine::TypeExecution::NORMAL;
	std::string commandes_debogueur;
	opterr = 0;
	for(;;)
	{
		int c;
		c = ::getopt(argc,argv,"dD:c:p:hv");
		if(c==EOF)
		{
			break;
		}
		switch(c)
		{ 
			case 'd':
				mode_debogage=SetLgg::Machine::Machine::Machine::TypeExecution::DEBUG;
				break;
			case 'D':
				commandes_debogueur = optarg;
				mode_debogage=SetLgg::Machine::Machine::Machine::TypeExecution::DEBUG;
				break;
			case 'c':
				if(fichier_coredump.empty())
				{
					fichier_coredump = optarg;
				}
				else
				{
					std::cerr << "Option -c can not be specified several times." << std::endl;
					return ERREUR_LIGNEDECOMMANDE;
				}
				break;
			case 'p':
				liste_plugins.push_back(optarg);
				break;
			case 'h':
				affiche_aide(std::cerr,argv[0]);
				return 0;
				break;
			case 'v':
				std::cerr << argv[0] << " version " << PACKAGE_VERSION << "." << std::endl
					<< std::endl
					<< argv[0] << " Copyright (C) 2012  Julien Bruguier" << std::endl
					<< "This program comes with ABSOLUTELY NO WARRANTY. " << std::endl
					<< "This is free software, and you are welcome to redistribute it" << std::endl
					<< "under certain conditions. Please refer to the licence GPLv3" << std::endl
					<< "for details." << std::endl;

				return 0;
				break;
			default:
				std::cerr << "Invalid option -" << static_cast<char>(optopt) << ". Try -h option." << std::endl;
				return ERREUR_LIGNEDECOMMANDE;
				break;
		}
	}
	if(optind+2>argc)
	{
		affiche_aide(std::cerr,argv[0]);
		return ERREUR_LIGNEDECOMMANDE;
	}
	fichier_programme = argv[optind];
	fichier_memoire = argv[optind+1];
	try
	{
		SetLgg::Machine::Machine::MachineSP machine(new SetLgg::Machine::Machine::Machine(fichier_programme,fichier_memoire, fichier_coredump, liste_plugins, mode_debogage, commandes_debogueur));
		return SetLgg::Machine::Machine::Machine::execution(machine);
	}
	catch(SetLgg::Global::Exception::Compilation& e)
	{
		e.format(std::cerr);
	}
	catch(SetLgg::Global::Exception::Execution& e)
	{
		e.format(std::cerr);
	}
	catch(SetLgg::Global::Exception::Integration& e)
	{
		e.format(std::cerr);
	}
	return ERREUR_GENERIQUE;
}

#endif
