/*
 * Simple Virtual Machine - A versatile and robust architecture to
 * easily write applications.
 * Copyright (C) 2021  Julien BRUGUIER
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <src/global/global.h>
#include <src/global/installation.h>
#include <src/outils/analyseur/analyseur.h>

struct ElementDatabase
{
	ElementDatabase() = default;
	explicit ElementDatabase(const std::string& s)
	{
		if(s.empty()) return;
		if(s[0]=='(')
		{
			auto f = s.find(')');
			if(f==std::string::npos) throw 1;
			_nom = s.substr(1,f-1);
			_valeur = s.substr(f+1);
		}
		else
		{
			_nom = s;
			_valeur = s;
		}
	}
	std::string _nom;
	std::string _valeur;
	operator const std::string& () const { return _valeur; }
	template<typename oStream>
	friend oStream& operator<<(oStream& os, const ElementDatabase& e)
	{
		os << e._valeur;
		return os;
	}
	bool operator<(const ElementDatabase& e) const { return _nom<e._nom; }
	bool operator!=(const ElementDatabase& e) const { return _valeur!=e._valeur; }
};

struct Modification
{
	static Modification ajout(const ElementDatabase& a)
	{
		return { Operation::AJOUT , a , ElementDatabase() };
	}
	static Modification enleve(const ElementDatabase& e)
	{
		return { Operation::ENLEVE , e , ElementDatabase() };
	}
	static Modification change(const ElementDatabase& avant, const ElementDatabase& apres)
	{
		return { Operation::CHANGE , avant , apres };
	}
	enum class Operation { AJOUT, ENLEVE, CHANGE };
	Operation _operation;
	ElementDatabase _primaire;
	ElementDatabase _secondaire;
	bool operator<(const Modification& m) const
	{
		if(_primaire<m._primaire) return true;
		if(m._primaire<_primaire) return false;
		if(_operation<m._operation) return true;
		if(m._operation<_operation) return false;
		if(_secondaire<m._secondaire) return true;
		if(m._secondaire<_secondaire) return false;
		return false;
	}
	template<typename oStream>
	friend oStream& operator<<(oStream& os, const Modification& m)
	{
		switch(m._operation)
		{
			case Operation::AJOUT:
			{
				os << "Added  : " << m._primaire << std::endl;
			}
			break;
			case Operation::ENLEVE:
			{
				os << "Removed: " << m._primaire << std::endl;
			}
			break;
			case Operation::CHANGE:
			{
				os << "Changed: " << std::endl
				   << "         From: " << m._primaire << std::endl
				   << "         To  : " << m._secondaire << std::endl;
			}
			break;
		}
		return os;
	}
	private:
		Modification() = default;
};

typedef std::map<Modification,size_t> Changements;

void ajout_changement(Changements& changements, const Modification& m)
{
	auto it = changements.insert(std::make_pair(m,1));
	if(not it.second)
	{
		++(it.first->second);
	}
}

std::map<std::string,ElementDatabase> tokens(const std::string& chaine, const char separateur, const char sep)
{
	std::map<std::string,ElementDatabase> elements;
	std::stringstream ss;
	ss.str(chaine);
	for(std::string item ; std::getline(ss, item, separateur) ; )
	{
		if(item.empty()) continue;
		auto s = item.find(sep);
		if(s==std::string::npos) throw 1;
		elements.insert(std::make_pair(item.substr(0,s),ElementDatabase(item.substr(s+1))));
	}
	return elements;
}

ElementDatabase valeur(const std::map<std::string,ElementDatabase>& database, const std::string& clef)
{
	auto it = database.find(clef);
	if(it==database.end()) return ElementDatabase("<unknown>");
	return it->second;
}

void analyse_signature(std::istream& signature, std::string& svm_api_signature, std::string& svm_api_database, std::string& svm_api_version, const std::string& type)
{
	bool presence_signature = false;
	bool presence_version = false;
	svm_api_signature = "";
	svm_api_database = "";
	svm_api_version = "";
	std::string ligne;
	while(signature.good())
	{
		std::getline(signature,ligne);
		size_t indice = ligne.find("#define");
		if(indice==std::string::npos) continue;
		indice = ligne.find("SVM_API_SIGNATURE");
		if(indice!=std::string::npos)
		{
			indice = ligne.find('"');
			ligne = ligne.substr(indice+1);
			indice = ligne.find('"');
			svm_api_signature = ligne.substr(0,indice);
			presence_signature = true;
			continue;
		}
		indice = ligne.find("SVM_API_DATABASE");
		if(indice!=std::string::npos)
		{
			indice = ligne.find('"');
			ligne = ligne.substr(indice+1);
			indice = ligne.find('"');
			svm_api_database = ligne.substr(0,indice);
			continue;
		}
		indice = ligne.find("SVM_VERSION");
		if(indice!=std::string::npos)
		{
			indice = ligne.find('"');
			ligne = ligne.substr(indice+1);
			indice = ligne.find('"');
			svm_api_version = ligne.substr(0,indice);
			presence_version = true;
			continue;
		}
	}
	if(not presence_signature)
	{
		std::cerr << "Invalid " << type << " API signature content: Missing SVM_API_SIGNATURE definition." << std::endl;
		throw 0;
	}
	if(not presence_version)
	{
		std::cerr << "Invalid " << type << " API signature content: Missing SVM_VERSION definition." << std::endl;
		throw 0;
	}
}

void affiche(const size_t mesure, const std::string& texte)
{
	std::string p = ((mesure==1)?"":"s");
	std::cout << "  " << mesure << " C/C++ type" << p << " or function" << p << " " << texte << "." << std::endl;
}

void aide(std::ostream& os, const std::string& binaire, const std::string& reference_base)
{
	os	<< binaire << " [options]" << std::endl
		<< std::endl
		<< "This command produces elements of the SVM installation." << std::endl 
		<< std::endl
		<< "Options:" << std::endl
		<< "\t-c <name>: display value associated to one configuration element of this SVM" << std::endl
		<< "\t-l       : display all configuration elements of this SVM" << std::endl
		<< std::endl
		<< std::endl
		<< "\t-s       : display the API signature of this SVM (extracted from " << reference_base << ")" << std::endl
		<< std::endl
		<< "\t-B       : check the API signature of this SVM backward compatibility against a reference API signature (given on stdin)" << std::endl
		<< "\t           (Returns 0 when compatible, 1 when incompatible, 2 when API signature is invalid)" << std::endl
		<< std::endl
		<< "\t-F       : check the API signature of this SVM forward compatibility against a reference API signature (given on stdin)" << std::endl
		<< "\t           (Returns 0 when compatible, 1 when incompatible, 2 when API signature is invalid)" << std::endl
		<< std::endl
		<< "\t-R <file>: use the file content as reference signature for compatibility check instead of stdin" << std::endl
		<< std::endl
		<< "\t-S <file>: use the file content as signature for compatibility check instead the API signature of this SVM" << std::endl
		<< std::endl
		<< std::endl
		<< "\t-p <name>: display value associated to one configuration element of the plugin file (given on stdin)" << std::endl
		<< "\t           when name is \":list\", the list of possible keys is displayed instead" << std::endl
		<< std::endl
		<< std::endl
		<< "\t-h       : display this help" << std::endl
		<< "\t-v       : display svm_config version and license" << std::endl
		<< std::endl;
}

void version(std::ostream& os, const std::string& binaire)
{
	os	<< binaire << " version " << VERSION << "  Copyright (C) 2021  Julien BRUGUIER" << std::endl
		<< std::endl
		<< "This program comes with ABSOLUTELY NO WARRANTY." << std::endl
		<< "This is free software, and you are welcome to redistribute it under certain conditions." << std::endl
		<< "See GNU GPLv3 terms for details." << std::endl
		<< std::endl;
}

int main(int argc, char *argv[])
{
	bool liste = false;
	bool configuration = false;
	std::string clef_configuration;
	bool signature = false;
	bool reference = false;
	std::string signature_reference;
	bool signature_fichier = false;
	std::string signature_entree;
	bool compatibilite = false;
	bool avant = false;
	bool extension = false;
	std::string clef_extension;
	std::string reference_base = SVM::Global::Installation::valeur("includedir")+"/svm_compatibility.h";

	::opterr = 0;
	int option_trouvee;
	for( ; ; )
	{
		option_trouvee = ::getopt(argc,argv,"lc:sBFR:S:p:hv");
		if(option_trouvee==EOF)
			break;
		switch(option_trouvee)
		{
			case 'h':
				std::cerr << std::endl;
				::aide(std::cerr,argv[0],reference_base);
				return 0;
				break;
			case 'v':
				std::cerr << std::endl;
				::version(std::cerr,argv[0]);
				return 0;
				break;
			case 'c':
				configuration = true;
				clef_configuration = optarg;
				break;
			case 'l':
				liste = true;
				break;
			case 's':
				signature = true;
				break;
			case 'B':
				avant = true;
			case 'F':
				compatibilite = true;
				break;
			case 'S':
				reference = true;
				signature_reference = optarg;
				break;
			case 'R':
				signature_fichier = true;
				signature_entree = optarg;
				break;
			case 'p':
				extension = true;
				clef_extension = optarg;
				break;
			case '?':
			default:
				std::cerr << "invalid option -" << std::string(1,optopt) << std::endl << std::endl;
				::aide(std::cerr,argv[0],reference_base);
				return 2;
				break;
		}
	}
	if(configuration)
	{
		try
		{
			std::cout << SVM::Global::Installation::valeur(clef_configuration) << std::endl;
		}
		catch(SVM::Global::ClefInvalide& e)
		{
			std::cerr << "Invalid key " << clef_configuration << "." << std::endl;
			return 2;
		}
		return 0;
	}
	if(liste)
	{
		size_t t = 0;
		for(const auto& c : SVM::Global::Installation::clefs())
		{
			auto ct = c.size();
			if(ct>t)
			{
				t=ct;
			}
		}
		for(const auto& c : SVM::Global::Installation::variables())
		{
			std::cout << c.first << std::string(t-c.first.size(),' ') << ": " << c.second << std::endl;
		}
		return 0;
	}
	if(signature)
	{
		std::string svm_api_signature;
		std::string svm_api_database;
		std::string svm_version;
		std::fstream ref(reference_base.c_str(),std::ios_base::in);
		if(not ref.good())
		{
			std::cerr << "Invalid signature file " << reference_base << std::endl;
			return 2;
		}
		::analyse_signature(ref,svm_api_signature,svm_api_database,svm_version,"created");
		std::cout << "#define SVM_API_SIGNATURE \"" << svm_api_signature << "\"" << std::endl
			<< "#define SVM_API_DATABASE \"" << svm_api_database << "\"" << std::endl
			<< "#define SVM_VERSION \"" << svm_version << "\"" << std::endl;
		return 0;
	}
	if(compatibilite)
	{
		try
		{
			std::string svm_api_signature_reference;
			std::string svm_api_database_reference;
			std::string svm_api_version_reference;
			if(reference)
			{
				std::fstream ref(signature_reference.c_str(),std::ios_base::in);
				if(not ref.good())
				{
					std::cerr << "Invalid signature file " << signature_reference << std::endl;
					return 2;
				}
				::analyse_signature(ref,svm_api_signature_reference,svm_api_database_reference,svm_api_version_reference,"reference");
			}
			else
			{
				std::fstream ref(reference_base.c_str(),std::ios_base::in);
				if(not ref.good())
				{
					std::cerr << "Invalid signature file " << reference_base << std::endl;
					return 2;
				}
				::analyse_signature(ref,svm_api_signature_reference,svm_api_database_reference,svm_api_version_reference,"reference");
			}
			std::string svm_api_signature;
			std::string svm_api_database;
			std::string svm_api_version;
			if(signature_fichier)
			{
				std::fstream sig(signature_entree.c_str(),std::ios_base::in);
				if(not sig.good())
				{
					std::cerr << "Invalid signature file " << signature_entree << std::endl;
					return 2;
				}
				::analyse_signature(sig,svm_api_signature,svm_api_database,svm_api_version,"input");
			}
			else
			{
				::analyse_signature(std::cin,svm_api_signature,svm_api_database,svm_api_version,"input");
			}
			auto signature_api_machine_set = ::tokens(svm_api_signature_reference,'_',':');
			auto signature_api_parametre_set = ::tokens(svm_api_signature,'_',':');
			auto db = ::tokens(svm_api_database_reference,':','_');
			std::map<std::string,ElementDatabase> database_api_machine_set = { db.begin(),db.end() };
			decltype(database_api_machine_set) database_api_parametre_set;
			if(not svm_api_database.empty())
			{
				db = ::tokens(svm_api_database,':','_');
				database_api_parametre_set = { db.begin(),db.end() };
			}
			if(avant)
			{
				std::swap(signature_api_parametre_set,signature_api_machine_set);
				std::swap(database_api_parametre_set,database_api_machine_set);
				std::swap(svm_api_version,svm_api_version_reference);
			}
			auto itm = signature_api_machine_set.begin();
			auto itp = signature_api_parametre_set.begin();
			size_t supprimes = 0;
			size_t changes = 0;
			size_t ajoutes = 0;
			size_t im = 0;
			size_t ip = 0;
			Changements changements;
			for( ; ; )
			{
				if(itm==signature_api_machine_set.end())
				{
					ajoutes += signature_api_parametre_set.size()-ip;
					for( ; itp!=signature_api_parametre_set.end() ; ++itp)
					{
						ajout_changement(changements,Modification::ajout(valeur(database_api_parametre_set,itp->first)));
					}
					break;
				}
				if(itp==signature_api_parametre_set.end())
				{
					supprimes += signature_api_machine_set.size()-im;
					for( ; itm!=signature_api_machine_set.end() ; ++itm)
					{
						ajout_changement(changements,Modification::enleve(valeur(database_api_machine_set,itm->first)));
					}
					break;
				}
				if(itm->first==itp->first)
				{
					if(itm->second!=itp->second)
					{
						ajout_changement(changements,Modification::change(valeur(database_api_machine_set,itm->first),valeur(database_api_parametre_set,itp->first)));
						++changes;
					}
					++itm; ++im;
					++itp; ++ip;
					continue;
				}
				if(itm->first<itp->first)
				{
					ajout_changement(changements,Modification::enleve(valeur(database_api_machine_set,itm->first)));
					++supprimes;
					++itm; ++im;
				}
				else
				{
					ajout_changement(changements,Modification::ajout(valeur(database_api_parametre_set,itp->first)));
					++ajoutes;
					++itp; ++ip;
				}
			}
			if(not changements.empty())
			{
				std::cout << "##### Details #####" << std::endl;
				for(const auto& c: changements)
				{
					std::cout << c.first;
					if(c.second>1)
					{
						std::cout << "         Found: " << c.second << " times." << std::endl;
					}
				}
				std::cout << std::endl;
			}
			std::cout << "##### Summary #####" << std::endl;
			affiche(supprimes,"removed");
			affiche(changes,"changed");
			affiche(ajoutes,"added");
			std::cout << std::endl;
			bool compatible = (supprimes==0) and (changes==0);

			std::cout << "Compatibility: " << (compatible?"YES":"NO") << std::endl
				<< std::endl
				<< "  From old SVM API version " << svm_api_version_reference << std::endl
				<< "  To   new SVM API version " << svm_api_version << std::endl;
			return compatible?0:1;
		}
		catch(...)
		{
			std::cerr << "Invalid SVM API signature or database!" << std::endl;
			return 2;
		}
		return 0;
	}
	if(extension)
	{
		try
		{
			std::string fichier;
			for(std::string ligne ; std::getline(std::cin, ligne) ; )
			{
				fichier += ligne;
				fichier += '\n';
			}
			SVM::Outils::ExtensionSP extension;
			extension = SVM::Outils::Analyseur::analyse_chaine(fichier);
			SVM::Machine::Extension::DescriptionSP description;
			std::string configuration;
			SVM::Outils::Langage langage;
			bool licence;
			std::string shell;
			extension->verifie_et_complete("<stdin>",description,configuration,langage,licence,shell);
			if(std::string(clef_extension) == ":list")
			{
				for(const auto& c: extension->_directive)
				{
					std::cout << c.first << ": ";
					auto it=c.second._chaine.find('\n');
					if(it!=std::string::npos)
					{
						std::cout << c.second._chaine.substr(0,it) << "...";
					}
					else
					{
						std::cout << c.second._chaine;
					}
					std::cout << std::endl;
				}
				return 0;
			}
			auto it = extension->_directive.find(clef_extension);
			if(it==extension->_directive.end())
			{
				std::cerr << "Invalid key " << clef_extension << "." << std::endl;
				return 2;
			}
			std::cout << it->second._chaine << std::endl;
			return 0;
		}
		catch(SVM::Outils::Erreur& e)
		{
			std::cerr << e << std::endl;
			return 2;
		}
		catch(SVM::Machine::Element::Valeur::ExceptionDemarrage& e)
		{
			std::cerr << e << std::endl;
			return 2;
		}
	}

	std::cerr << std::endl;
	::aide(std::cerr,argv[0],reference_base);
	return 2;
}
